import { dockerNameValidationSchema, dockerTagValidationSchema, staticResourceMountPointSchema } from './BoxFormParams';

describe('models/ui|BoxFormParams validation', () => {
   describe('Static resource mount point', () => {
      it('whitespace', () => {
         expect(() => staticResourceMountPointSchema.validateSync('a b')).toThrow();
      });

      // https://st.yandex-team.ru/DEPLOY-1125#5e020c0d34b2182e6e1b8dc8
      it('invalid values', () => {
         expect(() => staticResourceMountPointSchema.validateSync('//abc/abd/')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/./abc/abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('./abc/abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('../abc/abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/../abc/abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc/../abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc/./abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc//abd')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc/abd/../')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc/abd/..')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc/abd/.')).toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('/abc/abd/./')).toThrow();
      });

      it('valid values', () => {
         expect(() => staticResourceMountPointSchema.validateSync('.abc/abd/')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('..abc/abd/')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/.abd/')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/..abd/')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/abd../')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/abd/..a../')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/abc./')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/abc.')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/abc..')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('abc/abc../')).not.toThrow();
         expect(() => staticResourceMountPointSchema.validateSync('.../abc/.../abc/...')).not.toThrow();
      });
   });

   describe('Docker name', () => {
      it('cannot be prefixed by a registry hostname', () => {
         expect(() => dockerNameValidationSchema.validateSync('http://registry.yandex.net')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('https://registry.yandex.net')).toThrow();
      });

      it('whitespace', () => {
         expect(() => dockerNameValidationSchema.validateSync('abc bcd')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync(' abc/bcd')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('abc/bcd ')).toThrow();
      });

      it('invalid values', () => {
         // bsclickhouse, bsclickhouse_ext_pre, bsclickhouse_pre, etruntaev-stage,
         // haas-bugs, index-ultra-pq2index, index-ultra-pq2precalc, procu-www-unstable,
         // unification_test
         expect(() => dockerNameValidationSchema.validateSync('yabs/infra/clickhouse-proxy:127.1')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('yabs/infra/clickhouse-ext-proxy:007.1')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('kinopoisk/nginx-alpine-etruntaev_test:0.3')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('haas/haas-bugs:v0.6')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('video/cm-common:')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('tools/procu-www:unstable')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('load/yandex-tank-pip:rc')).toThrow();

         // mba-yaoption-prod, mba-yaoption-regress, mba-yaoption-test, soc_api_stage
         expect(() => dockerNameValidationSchema.validateSync('oebs/yaoption/')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('/soc/soc-api')).toThrow();

         expect(() => dockerNameValidationSchema.validateSync('abc//bcd')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('.abc/bcd')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('abc/bcd.')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('/a')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('b/')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('.a')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('b.')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('_a')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('b_')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('--a')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('b--')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('deploy..ui')).toThrow();
         expect(() => dockerNameValidationSchema.validateSync('deploy___ui')).toThrow();
      });

      it('valid values', () => {
         expect(() => dockerNameValidationSchema.validateSync('a-beznosikov/qa-field-autofiller')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('v-solomatin/apimon-exporter')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('n-plus-one')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('n/plus-one/n')).not.toThrow();

         expect(() => dockerNameValidationSchema.validateSync('abc123')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('abc123/bcd234')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('abc.123-123.abc__b')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('deploy.ui')).not.toThrow();
         expect(() => dockerNameValidationSchema.validateSync('deploy---ui')).not.toThrow();

         expect(() =>
            dockerNameValidationSchema.validateSync('deploy.ui/deploy-ui/deploy--ui/deploy_ui/deploy__ui'),
         ).not.toThrow();
      });
   });

   describe('Docker tag', () => {
      it('max length', () => {
         expect(() =>
            dockerTagValidationSchema.validateSync(
               '12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678',
            ),
         ).not.toThrow();

         expect(() =>
            dockerTagValidationSchema.validateSync(
               '123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789',
            ),
         ).toThrow();
      });

      it('whitespace', () => {
         expect(() => dockerTagValidationSchema.validateSync('abc 123')).toThrow();
         expect(() => dockerTagValidationSchema.validateSync(' abc.123')).toThrow();
         expect(() => dockerTagValidationSchema.validateSync('abc.123 ')).toThrow();
      });

      it('invalid values', () => {
         expect(() => dockerTagValidationSchema.validateSync('.123')).toThrow();
         expect(() => dockerTagValidationSchema.validateSync('-123')).toThrow();
      });

      it('valid values', () => {
         // sha256: - это временный костыль для хешей, которые пользователи пишут в поле тега DEPLOY-5129
         // api, heatmap, heatmap-dev, linkeye, stage1, yandexpay-front-production
         // [
         //    'sha256:161256bb860c72a011d1f1e03d10020265dfe1fc8c8dcf581ddbab2153b04483',
         //    'sha256:e1803c1faee00f3a7f8456870fb91930a9105105eeae33bdc3adcb55e1bf7bc3',
         //    'sha256:db640b24315c21987ac280874855829006727cee3e9a22fcb23969f79deffa41',
         //    'sha256:312c0e03084d7f50d779bf43fcddc1b33558e3ad5394edc0be1b30c3cd0a83ed',
         //    'sha256:9f1a16115a5c801eca2d7f64768bdb5b7a8d7b6481577f28306ed0fe99c63aa7',
         //    'sha256:62feaef6d837af4f9c3ef5534a3b11435669056e5aa4ae83a7443e0727038adc',
         //    'sha256:3b4b0d6de9262f668f9836bf9675e78faf54b6a35afb40b2ec9ea63dea42a9be',
         //    'sha256:94752b80d3be83a99ef243e2b06f370335c87eb7877e7513aedda5595fed8465',
         //    'sha256:ba26adf6002f041c7227d1ba0e40e8e07bf76ab65fdbc95fc8f8da107a8e68fe',
         //    'sha256:658af352e2722705126723a734c6a83f5bdaeb52a0e486a2c58f54b51f8acb0c',
         //    'sha256:392e5851f8017fd924d6afdb9222ebce5f0ba635010ba4069d6220a36bc04fdb',
         // ];

         expect(() =>
            dockerTagValidationSchema.validateSync(
               'sha256:392e5851f8017fd924d6afdb9222ebce5f0ba635010ba4069d6220a36bc04fdb',
            ),
         ).not.toThrow();

         expect(() => dockerTagValidationSchema.validateSync('123')).not.toThrow();
         expect(() => dockerTagValidationSchema.validateSync('1.2.3')).not.toThrow();
         expect(() => dockerTagValidationSchema.validateSync('123.234.345')).not.toThrow();
         expect(() => dockerTagValidationSchema.validateSync('123-234_345')).not.toThrow();
      });
   });
});
