/* eslint-disable no-template-curly-in-string */
import { BYTES, formatNumber, isEmpty } from '@yandex-infracloud-ui/libs';
import { array, boolean, object, Schema, string, TestContext } from 'yup';

import { ValidationContext } from '../../../../components/huge-form';
import { environmentValidationSchema } from '../../../../modules/environment/models';

import { EResolvConf } from '../../../../proto-typings';
import { getPaths } from '../../../../utils';

import { DeployUnit } from '../DeployUnit';

import {
   boxIdSchema,
   checksumSchema,
   dynamicResourceUrlSchema,
   hasOnlyUniqueValues,
   IntegerPositiveNullableSchema,
   orderSchema,
   portSchema,
   resourceIdValidationSchema,
   StringNullableRequiredSchema,
   StringNullableSchema,
   StringRequiredSchema,
   unixPathSchema,
} from '../validation';

import {
   Box,
   BoxDockerImage,
   BoxDynamicResource,
   BoxJugglerSettings,
   BoxLayer,
   BoxRootFsSettings,
   BoxStaticResource,
   BoxVolume,
   CgroupFsMountMode,
   DynamicResourceNotifyPolicy,
   DynamicResourceNotifyPolicyExecAction,
   DynamicResourceNotifyPolicyHttpAction,
   DynamicResourceNotifyPolicyMode,
   getEmptyBox,
   JugglerBundle,
   LogrotateConfig,
   VolumeCreateMode,
   VolumeMountMode,
} from './Box';

export type BoxFormParams = Omit<Box, 'workloads'>;

export const boxPaths = getPaths<BoxFormParams>();

// region docker

// https://docs.docker.com/engine/reference/commandline/tag/
// 1. An image name is made up of slash-separated name components.
// 2. Name components may contain lowercase letters (a-z), digits (0-9) and separators.
// A separator is defined as a period, one or two underscores, or one or more dashes ([.]|_|__|[-]+).
// A name component may not start or end with a separator.
export const dockerNameComponentRegEx = /^[a-z0-9](([.]|_|__|[-]+)?[a-z0-9]+)*$/;

// A tag name must be valid ASCII and may contain lowercase and uppercase letters, digits, underscores, periods and dashes ([\w-.]).
// A tag name may not start with a period or a dash and may contain a maximum of 128 characters.
// sha256: - это временный костыль для хешей, которые пользователи сейчас пишут в поле тега DEPLOY-5129
export const dockerTagRegEx = /^(\w[\w-.]*|[sS][hH][aA]256[:].+)$/;

export const dockerHostRegEx = /^((https?:\/\/)?registry\.yandex\.net)/i;

export const dockerNameValidationSchema = StringNullableRequiredSchema('Docker name').test(
   'valid',
   'value is valid',
   function test(value: string) {
      if (value) {
         if (dockerHostRegEx.test(value)) {
            return this.createError({
               message: '${path} cannot be prefixed by a registry hostname (http(s)://registry.yandex.net)',
            });
         }

         if (/\s/.test(value)) {
            return this.createError({
               message: '${path} must not contain whitespace',
            });
         }

         if (/[:]/.test(value)) {
            return this.createError({
               message: "${path} must not contain ':'",
            });
         }

         const invalidNameComponents = value
            .split('/')
            .map(v => dockerNameComponentRegEx.test(v))
            .filter(v => !v);

         if (!isEmpty(invalidNameComponents)) {
            return this.createError({
               message:
                  'An image name is made up of slash-separated name components. Name components may contain lowercase letters, digits and separators. A separator is defined as a period, one or two underscores, or one or more dashes. A name component may not start or end with a separator.',
            });
         }
      }

      return true;
   },
);

export const dockerTagValidationSchema = StringNullableRequiredSchema('Docker tag')
   .max(128)
   .test('valid', 'value is valid', function test(value: string) {
      if (value) {
         if (/\s/.test(value)) {
            return this.createError({
               message: '${path} must not contain whitespace',
            });
         }

         if (!dockerTagRegEx.test(value)) {
            if (/[:]/.test(value)) {
               return this.createError({
                  message: "${path} must not contain ':'",
               });
            }

            return this.createError({
               message:
                  'A tag name must be valid ASCII and may contain lowercase and uppercase letters, digits, underscores, periods and dashes. A tag name may not start with a period or a dash and may contain a maximum of 128 characters.',
            });
         }
      }

      return true;
   });

export const dockerValidationSchema = object<BoxDockerImage>({
   enabled: boolean(),
   name: dockerNameValidationSchema,
   tag: dockerTagValidationSchema,
});

const boxDockerValidationSchema = object<BoxDockerImage>({
   enabled: boolean(),
   name: string().when('enabled', {
      is: true,
      then: dockerNameValidationSchema,
      otherwise: string().nullable(),
   }),
   tag: string().when('enabled', {
      is: true,
      then: dockerTagValidationSchema,
      otherwise: string().nullable(),
   }),
});

// endregion

const layerAndStaticResourceRefValidationSchema = (resourceField: 'layers' | 'staticResources') =>
   resourceIdValidationSchema.test('isValid', '${path}s must be valid', function IsValid(value: string) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (!context || !value || isEmpty(value)) {
         return true;
      }

      const deployUnitForm = (context.parentForms[0] as unknown) as DeployUnit;
      const { disks } = deployUnitForm;

      const boxForm = (context.form as unknown) as Box;
      const { virtualDiskIdRef } = boxForm;

      const disk = virtualDiskIdRef && disks.length > 1 ? disks.find(v => v.id === virtualDiskIdRef) : disks[0];

      if (!disk) {
         return true;
      }

      const resourceNames = {
         'layers': 'Layer',
         'staticResources': 'Static resource',
      };

      const resource =
         resourceField === 'layers'
            ? // eslint-disable-next-line no-underscore-dangle
              disk.layers.find(v => v._ref === value)
            : // eslint-disable-next-line no-underscore-dangle
              disk.staticResources.find(v => v._ref === value);

      if (!resource) {
         return this.createError({
            message: `${resourceNames[resourceField]} does not exist on disk '${disk.id}'. Please use existing ${
               resourceField === 'layers' ? 'layers' : 'static resources'
            }.`,
         });
      }

      const boxRefs =
         resourceField === 'layers'
            ? // eslint-disable-next-line no-underscore-dangle
              boxForm.layers.filter(v => v._layerRef === value).map(v => v._layerRef)
            : // eslint-disable-next-line no-underscore-dangle
              boxForm.staticResources.filter(v => v._staticResourceRef === value).map(v => v._staticResourceRef);

      if (!hasOnlyUniqueValues(boxRefs)) {
         return this.createError({
            message: `${resourceNames[resourceField]} must be unique.`,
         });
      }

      return true;
   });

export const layersValidationSchema = array().of(
   object<BoxLayer>({
      _layerRef: layerAndStaticResourceRefValidationSchema('layers').label('Layer ID'),
   }),
);

// region static resources

export const staticResourceMountPointSchema = unixPathSchema('Static resource mount point').test(
   'uniqueMountPoint',
   'Static resource should have unique mount point',
   // в пределах одного бокса mount points должны быть разные
   function UniqueMountPoint(mountPoint: string) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (context) {
         const boxForm = (context.form as unknown) as Box;
         const mountPoints = boxForm.staticResources.filter(v => v.mountPoint === mountPoint).map(v => v.mountPoint);

         if (!hasOnlyUniqueValues(mountPoints)) {
            return this.createError({
               message: `Static resource should have unique mount point (${mountPoint})`,
            });
         }
      }

      return true;
   },
);

export const staticResourcesValidationSchema = array().of(
   object<BoxStaticResource>({
      _order: orderSchema,
      mountPoint: staticResourceMountPointSchema.required(),
      _staticResourceRef: layerAndStaticResourceRefValidationSchema('staticResources').label('Static resource ID'),
   }),
);

// endregion

// region dynamic resources

const dynamicResourceIdSchema = resourceIdValidationSchema
   .label('Dynamic resource ID')
   .test('uniqueId', 'Dynamic resource should have unique ID', function UniqueId(id: string) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (context) {
         const boxContext = ([context.form, ...context.siblingForms, ...context.cousinForms] as unknown) as Box[];

         const dynamicResources = boxContext
            .filter(box => box.dynamicResources && !isEmpty(box.dynamicResources))
            .map(box => box.dynamicResources)
            .reduce((acc, value) => [...acc, ...value], [])
            .filter(dynamicResource => dynamicResource.id === id)
            .map(dynamicResource => dynamicResource.id);

         if (!hasOnlyUniqueValues(dynamicResources)) {
            return this.createError({
               message: `Dynamic resource should have unique ID (${id})`,
            });
         }
      }

      return true;
   });

const dynamicResourceCustomSettingsSchema = object({
   deployGroups: boolean(),
   requiredLabels: boolean(),
});

const dynamicResourceHttpActionSettingsSchema = object<DynamicResourceNotifyPolicyHttpAction>().when('mode', {
   is: DynamicResourceNotifyPolicyMode.Http,
   then: object({
      url: StringNullableRequiredSchema('URL').matches(/^https?:\/\/.+$/, '${path} format is invalid (http(s)://)'),
      expectedAnswer: StringNullableSchema('Expected answer'),
   }),
   otherwise: object(),
});

const dynamicResourceExecActionSettingsSchema = object<DynamicResourceNotifyPolicyExecAction>().when('mode', {
   is: DynamicResourceNotifyPolicyMode.Exec,
   then: object({
      commandLine: StringNullableRequiredSchema('Command line'),
      expectedAnswer: StringNullableSchema('Expected answer'),
   }),
   otherwise: object(),
});

const dynamicResourceUrlSettingsSchema = array().of(dynamicResourceUrlSchema);

const dynamicResourceNotifyPolicySettingsSchema = object<DynamicResourceNotifyPolicy>({
   mode: StringRequiredSchema('Notify policy mode').oneOf([
      DynamicResourceNotifyPolicyMode.Disabled,
      DynamicResourceNotifyPolicyMode.Exec,
      DynamicResourceNotifyPolicyMode.Http,
   ]) as Schema<DynamicResourceNotifyPolicyMode>,
   httpAction: dynamicResourceHttpActionSettingsSchema,

   execAction: dynamicResourceExecActionSettingsSchema,
});

const dynamicResourceVerificationSettingsSchema = object({
   checksum: checksumSchema,
   checkPeriodMs: IntegerPositiveNullableSchema('Check period ms'),
});

const dynamicResourceAdvancedSettingsSchema = object({
   verification: dynamicResourceVerificationSettingsSchema,
   allowDeduplication: boolean(),
   maxDownloadSpeed: IntegerPositiveNullableSchema('Max download speed'),
});

export const dynamicResourceValidationSchema = object<BoxDynamicResource>({
   id: dynamicResourceIdSchema,
   customSettings: dynamicResourceCustomSettingsSchema,
   urls: dynamicResourceUrlSettingsSchema,
   storageDir: unixPathSchema('Storage dir'),
   destination: unixPathSchema('Destination'),
   updateWindow: IntegerPositiveNullableSchema('Update window'),
   cachedRevisionsCount: IntegerPositiveNullableSchema('Cache rev count'),
   notifyPolicy: dynamicResourceNotifyPolicySettingsSchema,
   advancedSettings: dynamicResourceAdvancedSettingsSchema,
});

export const dynamicResourcesValidationSchema = array().of(dynamicResourceValidationSchema);

// endregion

export const boxJugglerValidationSchema = object<BoxJugglerSettings>({
   enabled: boolean(),
   port: portSchema,
   bundles: array()
      .of(object<JugglerBundle>())
      .when('enabled', {
         is: true,
         then: array().of(
            object({
               url: StringNullableSchema('URL')
                  .trim()
                  .matches(/^(https?:\/\/.+|rbtorrent:[a-f0-9]{40}|sbr:[\d]+)$/, {
                     message: '${path} format is invalid (http(s)://, rbtorrent:, sbr:)',
                     excludeEmptyString: true,
                  }),
            }),
         ),
         otherwise: array(),
      }),
});

// region resources

type SumLimitValue = 'cpu' | 'ram';
type SumLimitValuePerBox = 'cpuPerBox' | 'ramPerBox';

function sumLimitCheck(this: TestContext, value: number, valueType: SumLimitValue) {
   const context: ValidationContext | undefined = this.options.context as any;

   if (!context) {
      return true;
   }

   const siblingForms = (context.siblingForms as unknown) as Box[];
   const deployUnitForm = (context.parentForms[0] as unknown) as DeployUnit;
   const deployUnitValue = deployUnitForm[valueType];

   // нет смысла в проверке, если параметр не задан
   if (deployUnitValue === null) {
      return true;
   }

   const valuePerBox = `${valueType}PerBox` as SumLimitValuePerBox;
   const boxesValue =
      value + siblingForms.reduce((sum, box) => (box[valuePerBox] ? sum + (box[valuePerBox] ?? 0) : sum), 0);

   const valueName = valueType.toUpperCase();
   if (deployUnitValue < boxesValue) {
      return this.createError({
         message: `Boxes ${valueName} limit sum (${formatNumber(
            boxesValue,
         )}) should be less than Pod ${valueName} limit (${formatNumber(deployUnitValue)})`,
      });
   }

   return true;
}

// см. https://st.yandex-team.ru/DEPLOY-4768
export const boxCpuValidationSchema = IntegerPositiveNullableSchema('CPU per Box')
   .min(100, '${path} must be greater than or equal to 0.1 CPU (100 VCPU)')
   .test('cpuLimit', 'Boxes vCPU limit sum should be less than Pod vCPU limit', function check(value) {
      return sumLimitCheck.call(this, value, 'cpu');
   });

export const boxRamValidationSchema = IntegerPositiveNullableSchema('RAM per Box')
   .min(0.2 * BYTES.GB, '${path} must be greater than or equal to 0.2 GB (204.8 MB)')
   .test('ramLimit', 'Boxes RAM limit sum should be less than Pod RAM limit', function check(value) {
      return sumLimitCheck.call(this, value, 'ram');
   });

export const anonymousMemoryLimitSchema = IntegerPositiveNullableSchema('Anonymous memory limit per Box')
   .min(0.2 * BYTES.GB, '${path} must be greater than or equal to 0.2 GB (204.8 MB)')
   .test(
      'anonMemoryCheck',
      'Anonymous memory limit should be less than min(Box RAM, DeployUnit AnonMemoryLimit, DeployUnit RAM)',
      function RamLimit(value) {
         const context: ValidationContext | undefined = this.options.context as any;

         if (!context) {
            return true;
         }
         const box = (context.form as unknown) as Box;
         const deployUnitForm = (context.parentForms[0] as unknown) as DeployUnit;
         const deployUnitRAM = deployUnitForm.ram ?? Infinity;
         const deployUnitAnon = deployUnitForm.anonymousMemoryLimit ?? Infinity;

         if (box.ramPerBox) {
            if (value > box.ramPerBox) {
               return this.createError({
                  message: `There is no point in setting anonymous memory limit per box (${formatNumber(
                     value,
                  )}) > RAM per box (${formatNumber(box.ramPerBox)}).`,
               });
            }
         } else if (value > Math.min(deployUnitRAM, deployUnitAnon)) {
            return this.createError({
               message: `There is no point in setting anonymous memory limit per box (${formatNumber(value)}) >
               ${deployUnitRAM < deployUnitAnon ? 'Deploy Unit RAM' : 'Deploy Unit anonymous memory limit'}
             (${formatNumber(Math.min(deployUnitRAM, deployUnitAnon))}).`,
            });
         }

         return true;
      },
   );

// endregion

export const resolvConfValidationSchema = string()
   .oneOf([
      EResolvConf.EResolvConf_DEFAULT,
      EResolvConf.EResolvConf_NAT64,
      EResolvConf.EResolvConf_KEEP,
      EResolvConf.EResolvConf_NAT64_LOCAL,
   ])
   .required() as Schema<EResolvConf>;

export const logrotateConfigValidationSchema = object<LogrotateConfig>({
   rawConfig: StringNullableSchema('Raw config'),
   runPeriodMillisecond: IntegerPositiveNullableSchema('Run period'),
});

export const volumeMountPointValidationSchema = string()
   .label('Volume mount point')
   .matches(/^(\/)?[\w-_]+(.*)(\/)?$/)
   .nullable()
   .required()
   .test('isValid', '${path}s must be valid', function IsValid(mountPoint: string) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (mountPoint && context) {
         const boxForm = (context.form as unknown) as Box;

         const formatMountPoint = (value: string) =>
            value
               .split('/')
               .filter(v => !isEmpty(v))
               .join('/');

         const mountPointIds = boxForm.volumes.map(v => v.mountPoint).filter(v => v && !isEmpty(v)) as string[];

         const formattedValue = formatMountPoint(mountPoint);

         if (mountPoint !== mountPoint.replace('//', '')) {
            return this.createError({
               message: `${'${path}'} (${mountPoint}) must not contain double slash (//)`,
            });
         }

         const duplicates = mountPointIds.filter(v => formatMountPoint(v) === formattedValue);
         if (!hasOnlyUniqueValues(duplicates)) {
            return this.createError({
               message: `${'${path}'} (${mountPoint}) should be unique`,
            });
         }

         const nestedMountPoints = mountPointIds.filter(v => formattedValue.indexOf(`${formatMountPoint(v)}/`) === 0);
         if (!isEmpty(nestedMountPoints)) {
            return this.createError({
               message: `${'${path}'} (${mountPoint}) should not be nested (${nestedMountPoints.join(', ')})`,
            });
         }
      }

      return true;
   });

export const VolumeRefSchema = StringNullableRequiredSchema('Volume ref').test(
   'isExist',
   'Is volume exist',
   function IsExist(value: string | null) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (!context || !value) {
         return true;
      }

      const deployUnitForm = (context.parentForms[0] as unknown) as DeployUnit;

      const volumes = (deployUnitForm.disks ?? []).flatMap(v => v.volumes ?? []);
      // eslint-disable-next-line no-underscore-dangle
      const volume = volumes.find(v => v._ref === value);

      if (!volume) {
         return this.createError({
            message: 'Volume does not exist. Please use existing volumes.',
         });
      }

      return true;
   },
);

export const volumesValidationSchema = array().of(
   object<BoxVolume>({
      mode: string()
         .label('Volume mount mode')
         .oneOf([VolumeMountMode.ReadOnly, VolumeMountMode.ReadWrite])
         .required() as Schema<VolumeMountMode>,
      mountPoint: volumeMountPointValidationSchema,
      _volumeRef: VolumeRefSchema,
   }),
);

export const rootFsSettingsSchema = object<BoxRootFsSettings>({
   createMode: string()
      .label('Volume create mode')
      .oneOf([VolumeCreateMode.ReadOnly, VolumeCreateMode.ReadWrite])
      .required() as Schema<VolumeCreateMode>,
});

export const boxIdValidationSchema = boxIdSchema.test(
   'valid',
   'value is valid',
   function IsValid(value: string | null) {
      if (value && !isEmpty(value)) {
         if (/[.:@]/.test(value)) {
            return this.createError({
               message: '${path} should not include (.|:|@)',
            });
         }

         const context: ValidationContext | undefined = this.options.context as any;

         if (context) {
            const boxForms = ([context.form, ...context.siblingForms] as unknown) as Box[];

            // приводим значения к одному регистру, подчёркивания считаем эквивалентом дефиса #DEPLOY-3875
            const formatValue = (v: string) => v.toLocaleLowerCase().replace(/_/g, '-');

            const formattedValue = formatValue(value);

            const boxIdList = boxForms
               .map(v => v.id)
               .filter(v => v && !isEmpty(v) && formatValue(v) === formattedValue);

            if (!hasOnlyUniqueValues(boxIdList)) {
               return this.createError({
                  message: `Box IDs must be unique and case insensitive, dashes and underscores are considered equal (${boxIdList.join(
                     ', ',
                  )})`,
               });
            }
         }
      }

      return true;
   },
);

export const virtualDiskIdRefValidationSchema = string().nullable();
// TODO: раскомментировать, когда появится селект для virtualDiskIdRef #DEPLOY-5373
// .test('isValid', 'value is valid', function IsValid(value: string | null) {
//    const context: ValidationContext | undefined = this.options.context as any;

//    if (!context) {
//       return true;
//    }

//    const deployUnitForm = (context.parentForms[0] as unknown) as DeployUnit;
//    const disksIds = (deployUnitForm.disks ?? []).map(v => v.id);

//    if (disksIds.length > 1) {
//       if (!value) {
//          return this.createError({
//             message: 'Virtual disk ID is required field',
//          });
//       } else if (!disksIds.includes(value)) {
//          return this.createError({
//             message: 'Virtual disk ID is invalid. Please use existing disks.',
//          });
//       }
//    }

//    return true;
// });

export const boxValidationSchema = object<BoxFormParams>({
   id: boxIdValidationSchema,
   bindSkynet: boolean(),
   resolvConf: resolvConfValidationSchema,
   cpuPerBox: boxCpuValidationSchema,
   ramPerBox: boxRamValidationSchema,
   anonymousMemoryLimit: anonymousMemoryLimitSchema,
   threadLimit: IntegerPositiveNullableSchema('Thread limit').max(10000),
   environment: environmentValidationSchema,
   dockerImage: boxDockerValidationSchema,
   layers: array<BoxLayer>().when('dockerImage', {
      is: dockerImage => !dockerImage.enabled,
      then: layersValidationSchema.min(1, 'You should add at least one box layer'),
      otherwise: layersValidationSchema,
   }),
   logrotateConfig: logrotateConfigValidationSchema,
   staticResources: staticResourcesValidationSchema,
   dynamicResources: dynamicResourcesValidationSchema,
   juggler: boxJugglerValidationSchema,
   virtualDiskIdRef: virtualDiskIdRefValidationSchema,
   volumes: volumesValidationSchema,
   rootFsSettings: rootFsSettingsSchema,
   cgroupFsMountMode: string()
      .label('cgroupfs mount mode')
      .oneOf([CgroupFsMountMode.None, CgroupFsMountMode.ReadOnly, CgroupFsMountMode.ReadWrite])
      .required() as Schema<CgroupFsMountMode>,
});

export function boxToFormParams(item = getEmptyBox()): BoxFormParams {
   const { workloads, ...formParams } = item;

   return formParams;
}

export function formParamsToBox(params: BoxFormParams, old: Box): Box {
   return {
      ...old,
      ...params,
      workloads: [], // clear children
   };
}
