import { BYTES } from '@yandex-infracloud-ui/libs';

import { EResolvConf, TBox, TDeployUnitSpec, TPodAgentSpec, TPodTemplateSpec } from '../../../../proto-typings';
import { noop } from '../../../../utils';
import { DEFAULT_DISK_ID, DEFAULT_OS_LIST, EMPTY_CHECKSUM, HDD_BANDWIDTH_LIMIT_FACTOR } from '../../../constants';
import { DeepPartial } from '../../../typeHelpers';
import { DeployUnitConverter, DiskType } from '../DeployUnit';
import { StageParentNodeIds } from '../StageParentNodeIds';
import { StagePatcherVisitor } from '../StagePatcherVisitor';

import { Box } from './Box';
import { BoxConverter } from './BoxConverter';
import { BoxPatcher } from './BoxPatcher';

const getInitialBoxSpec = (): DeepPartial<TBox> => ({
   id: 'box',
});

const LAYER = {
   baseDefault: {
      id: 'base-layer-0',
      url: DEFAULT_OS_LIST[0].value,
      checksum: EMPTY_CHECKSUM,
      virtual_disk_id_ref: 'disk-1',
   },
   baseCustom: {
      id: 'base-layer-1',
      url: 'rbtorrent:custom',
      checksum: EMPTY_CHECKSUM,
      virtual_disk_id_ref: 'disk-2',
   },
   layer: {
      id: 'layer',
      url: 'rbtorrent:layer',
      checksum: EMPTY_CHECKSUM,
      virtual_disk_id_ref: 'disk-1',
   },
   layerWithoutUrl: {
      id: 'layerWithoutUrl',
      // url: undefined,
      checksum: EMPTY_CHECKSUM,
      virtual_disk_id_ref: 'disk-1',
   },
   layerWithoutChecksum: {
      id: 'layerWithoutChecksum',
      url: 'rbtorrent:layerWithoutChecksum',
      // checksum: undefined,
      virtual_disk_id_ref: 'disk-2',
   },
   // layerWithCustomChecksum: {
   //    id: 'layerWithCustomChecksum',
   //    url: 'rbtorrent:layerWithCustomChecksum',
   //    checksum: 'CUSTOM_CHECKSUM',
   //    virtual_disk_id_ref: 'disk-2',
   // },
};

const getInitialPodAgentSpec = (): DeepPartial<TPodAgentSpec> => ({
   // ...
   boxes: [getInitialBoxSpec()],
   resources: {
      layers: [
         {
            ...LAYER.baseDefault,
         },
         {
            ...LAYER.baseCustom,
         },
         {
            ...LAYER.layer,
         },
         {
            ...LAYER.layerWithoutUrl,
         },
         {
            ...LAYER.layerWithoutChecksum,
         },
         // {
         //    ...LAYER.layerWithCustomChecksum,
         // },
      ],
   },
});

const getInitialPodTemplateSpec = (): DeepPartial<TPodTemplateSpec> => ({
   spec: {
      host_infra: {
         monitoring: {},
      },
      pod_agent_payload: {
         spec: getInitialPodAgentSpec(),
      },
      disk_volume_requests: [
         {
            id: DEFAULT_DISK_ID,
            storage_class: DiskType.HDD,
            quota_policy: {
               capacity: 15 * BYTES.GB,
               bandwidth_guarantee: 25 * BYTES.MB,
               bandwidth_limit: 100 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
            },
            labels: {
               x: 3,
               used_by_infra: true,
            },
         },
      ],
   },
});

const getInitialDeployUnitSpec = (): DeepPartial<TDeployUnitSpec> => ({
   replica_set: {
      // ...
      replica_set_template: {
         pod_template_spec: getInitialPodTemplateSpec(),
      },
   },
});

/**
 * @example
 * it('should xxx', check(
 *    spec => {
 *
 *    },
 *    b => {
 *
 *    },
 *    expected => {
 *
 *    },
 *    spec => {
 *
 *    }
 *  ))
 */
const check = (
   patchSpecBefore: (s: DeepPartial<TBox>) => void,
   patchModel: (m: Box) => void,
   patchExpectedSpec: (s: DeepPartial<TBox>) => void,
   checkOnlyBranch?: (s: TBox) => any,
) => () => {
   const visitor = new StagePatcherVisitor();
   const spec = getInitialBoxSpec() as TBox;
   const podTemplateSpec = getInitialPodTemplateSpec() as TPodTemplateSpec;
   const deployUnitSpec = getInitialDeployUnitSpec() as TDeployUnitSpec;
   const disks = DeployUnitConverter.getDisks(podTemplateSpec);
   patchSpecBefore(spec);

   const model = BoxConverter.fromApi(spec, podTemplateSpec, deployUnitSpec, {}, disks);
   patchModel(model);

   const expectedPatchedSpec = getInitialBoxSpec();
   patchExpectedSpec(expectedPatchedSpec);

   const patchedSpec = BoxPatcher.patch(
      visitor,
      new StageParentNodeIds({ id: 'test-stage' }, { id: 'test-du' }),
      spec,
      model,
      disks,
   );

   if (checkOnlyBranch) {
      expect(checkOnlyBranch(patchedSpec)).toEqual(checkOnlyBranch(expectedPatchedSpec as TBox));
   } else {
      expect(patchedSpec).toEqual(expectedPatchedSpec);
   }
};

describe('models/ui|BoxPatcher', () => {
   it(
      'should patch resolv.conf: add',
      check(
         noop,
         b => {
            b.resolvConf = EResolvConf.EResolvConf_KEEP;
         },
         expected => {
            expected.resolv_conf = EResolvConf.EResolvConf_KEEP;
         },
      ),
   );

   it(
      'should patch resolv.conf: edit',
      check(
         spec => {
            spec.resolv_conf = EResolvConf.EResolvConf_KEEP;
         },
         b => {
            b.resolvConf = EResolvConf.EResolvConf_NAT64;
         },
         expected => {
            expected.resolv_conf = EResolvConf.EResolvConf_NAT64;
         },
      ),
   );

   it(
      'should patch resolv.conf: clear',
      check(
         spec => {
            spec.resolv_conf = EResolvConf.EResolvConf_KEEP;
         },
         b => {
            b.resolvConf = EResolvConf.EResolvConf_DEFAULT;
         },
         expected => {
            expected.resolv_conf = undefined;
         },
      ),
   );

   it(
      'should patch bindSkynet: add',
      check(
         noop,
         b => {
            b.bindSkynet = true;
         },
         expected => {
            expected.bind_skynet = true;
         },
      ),
   );

   it(
      'should patch bindSkynet: clear',
      check(
         spec => {
            spec.bind_skynet = true;
         },
         b => {
            b.bindSkynet = false;
         },
         expected => {
            expected.bind_skynet = false;
         },
      ),
   );

   describe('should patch resources', () => {
      describe('should patch cpu', () => {
         it(
            'should patch cpu: add',
            check(
               noop,
               b => {
                  b.cpuPerBox = 100;
               },
               expected => {
                  expected.compute_resources = {
                     vcpu_limit: 100,
                  };
               },
            ),
         );

         it(
            'should patch cpu: edit',
            check(
               spec => {
                  spec.compute_resources = {
                     vcpu_limit: 100,
                  };
               },
               b => {
                  b.cpuPerBox = 200;
               },
               expected => {
                  expected.compute_resources = {
                     vcpu_limit: 200,
                  };
               },
            ),
         );

         it(
            'should patch cpu: clear',
            check(
               spec => {
                  spec.compute_resources = {
                     vcpu_limit: 100,
                  };
               },
               b => {
                  b.cpuPerBox = null;
               },
               expected => {
                  expected.compute_resources = {};
               },
            ),
         );
      });

      describe('should patch ram', () => {
         it(
            'should patch ram: add',
            check(
               noop,
               b => {
                  b.ramPerBox = BYTES.GB;
               },
               expected => {
                  expected.compute_resources = {
                     memory_limit: BYTES.GB,
                  };
               },
            ),
         );

         it(
            'should patch ram: edit',
            check(
               spec => {
                  spec.compute_resources = {
                     memory_limit: BYTES.GB,
                  };
               },
               b => {
                  b.ramPerBox = 2 * BYTES.GB;
               },
               expected => {
                  expected.compute_resources = {
                     memory_limit: 2 * BYTES.GB,
                  };
               },
            ),
         );

         it(
            'should patch ram: clear',
            check(
               spec => {
                  spec.compute_resources = {
                     memory_limit: BYTES.GB,
                  };
               },
               b => {
                  b.ramPerBox = null;
               },
               expected => {
                  expected.compute_resources = {};
               },
            ),
         );
      });

      describe('should patch anonymous memory limit', () => {
         it(
            'should patch anonymous memory limit: add',
            check(
               noop,
               b => {
                  b.anonymousMemoryLimit = BYTES.GB;
               },
               expected => {
                  expected.compute_resources = {
                     anonymous_memory_limit: BYTES.GB,
                  };
               },
            ),
         );

         it(
            'should patch anonymous memory limit: edit',
            check(
               spec => {
                  spec.compute_resources = {
                     anonymous_memory_limit: BYTES.GB,
                  };
               },
               b => {
                  b.anonymousMemoryLimit = 2 * BYTES.GB;
               },
               expected => {
                  expected.compute_resources = {
                     anonymous_memory_limit: 2 * BYTES.GB,
                  };
               },
            ),
         );

         it(
            'should patch anonymous memory limit: clear',
            check(
               spec => {
                  spec.compute_resources = {
                     anonymous_memory_limit: BYTES.GB,
                  };
               },
               b => {
                  b.anonymousMemoryLimit = null;
               },
               expected => {
                  expected.compute_resources = {};
               },
            ),
         );
      });

      describe('should patch thread limit', () => {
         it(
            'should patch thread_limit: add',
            check(
               noop,
               b => {
                  b.threadLimit = 10;
               },
               expected => {
                  expected.compute_resources = {
                     thread_limit: 10,
                  };
               },
            ),
         );

         it(
            'should patch thread_limit: edit',
            check(
               spec => {
                  spec.compute_resources = {
                     thread_limit: 10,
                  };
               },
               b => {
                  b.threadLimit = 20;
               },
               expected => {
                  expected.compute_resources = {
                     thread_limit: 20,
                  };
               },
            ),
         );

         it(
            'should patch thread_limit: clear',
            check(
               spec => {
                  spec.compute_resources = {
                     thread_limit: 10,
                  };
               },
               b => {
                  b.threadLimit = null;
               },
               expected => {
                  expected.compute_resources = {};
               },
            ),
         );
      });
   });
});
