import { deepClone } from '@yandex-infracloud-ui/libs';

import { patchEnvironment } from '../../../../modules/environment';

import {
   ECgroupFsMountMode,
   EResolvConf,
   EVolumeCreateMode,
   EVolumeMountMode,
   TBox,
   TMountedStaticResource,
   TMountedVolume,
} from '../../../../proto-typings';

import { patchBoolean, patchList, patchNumber, patchObject, patchString, skipEmpty } from '../../utils';
import { StageParentNodeIds } from '../StageParentNodeIds';
import { StagePatcherVisitor } from '../StagePatcherVisitor';

import {
   Box,
   BoxStaticResource,
   CgroupFsMountMode,
   defaultVolumeCreateMode,
   VolumeCreateMode,
   BoxVolume,
   VolumeMountMode,
} from './Box';

import { DeployUnitDisk } from '../DeployUnit';

const rawRootfsCreateModes: Record<VolumeCreateMode, EVolumeCreateMode> = {
   [VolumeCreateMode.ReadOnly]: EVolumeCreateMode.EVolumeCreateMode_READ_ONLY,
   [VolumeCreateMode.ReadWrite]: EVolumeCreateMode.EVolumeCreateMode_READ_WRITE,
};

const rawCgroupfsModes: Record<CgroupFsMountMode, ECgroupFsMountMode> = {
   [CgroupFsMountMode.None]: ECgroupFsMountMode.ECgroupFsMountMode_NONE,
   [CgroupFsMountMode.ReadOnly]: ECgroupFsMountMode.ECgroupFsMountMode_RO,
   [CgroupFsMountMode.ReadWrite]: ECgroupFsMountMode.ECgroupFsMountMode_RW,
};

export class BoxPatcher {
   public static patch(
      visitor: StagePatcherVisitor,
      parentNodeIds: StageParentNodeIds,
      raw: TBox,
      box: Box,
      disks: DeployUnitDisk[],
   ): TBox {
      return new BoxPatcher(visitor, parentNodeIds, raw, box, disks).toValue();
   }

   private static patchVolumes(volumes: BoxVolume[], specVolumes: TMountedVolume[], disks: DeployUnitDisk[]) {
      const disksVolumes = disks.flatMap(v => v.volumes ?? []);
      const specVolumesNew: TMountedVolume[] = [];

      for (const volume of volumes) {
         // eslint-disable-next-line no-underscore-dangle
         const diskVolume =
            // eslint-disable-next-line no-underscore-dangle
            volume._volumeRef !== null ? disksVolumes.find(v => v._ref === volume._volumeRef) : undefined;

         // можно было бы удалять ссылки, если volume был удалён, но пока этого делать не будем,
         // так у пользователя будет возможность задуматься, если он делает что-то не то
         // когда он не сможет обновить невалидную спеку (есть валидация на бекенде)
         // if (diskVolume?.id && !diskVolume.removed) {
         if (diskVolume) {
            const exist =
               // eslint-disable-next-line no-underscore-dangle
               volume._order !== undefined && specVolumes?.[volume._order] ? specVolumes[volume._order] : undefined;

            if (!exist) {
               specVolumesNew.push({
                  mode:
                     volume.mode === VolumeMountMode.ReadWrite
                        ? EVolumeMountMode.EVolumeMountMode_READ_WRITE
                        : undefined,
                  mount_point: volume.mountPoint,
                  volume_ref: diskVolume.id,
               } as TMountedVolume);
            } else {
               patchString(exist, 'volume_ref', () => diskVolume.id);
               patchString(exist, 'mount_point', () => volume.mountPoint);
               patchString(exist, 'mode', () =>
                  volume.mode === VolumeMountMode.ReadWrite ? EVolumeMountMode.EVolumeMountMode_READ_WRITE : undefined,
               );

               specVolumesNew.push(exist);
            }
         }
      }

      return skipEmpty(specVolumesNew);
   }

   private static patchStaticResources(
      staticResources: BoxStaticResource[],
      specStaticResources: TMountedStaticResource[],
      disk: DeployUnitDisk | undefined,
   ) {
      /* eslint-disable no-underscore-dangle */
      const specStaticResourcesNew: TMountedStaticResource[] = [];

      for (const staticResource of staticResources) {
         const diskStaticResource = disk?.staticResources.find(v => v._ref === staticResource._staticResourceRef);

         if (diskStaticResource) {
            const exist =
               staticResource._order !== undefined && specStaticResources?.[staticResource._order]
                  ? specStaticResources[staticResource._order]
                  : undefined;

            if (!exist) {
               specStaticResourcesNew.push({
                  resource_ref: diskStaticResource.id,
                  mount_point: staticResource.mountPoint ?? '',
               });
            } else {
               patchString(exist, 'resource_ref', () => diskStaticResource.id);
               patchString(exist, 'mount_point', () => staticResource.mountPoint ?? '');

               specStaticResourcesNew.push(exist);
            }
         }
      }

      return specStaticResourcesNew;
      /* eslint-enable no-underscore-dangle */
   }

   constructor(
      private visitor: StagePatcherVisitor,
      private parentNodeIds: StageParentNodeIds,
      private raw: TBox,
      private box: Box,
      private disks: DeployUnitDisk[],
   ) {}

   private toValue(): TBox {
      const { raw, box, disks } = this;
      const spec: TBox = deepClone(raw ?? {}) as any;

      // если не указан virtualDiskIdRef, то бе
      // eslint-disable-next-line no-underscore-dangle
      const disk = disks.find(v => box.virtualDiskIdRef && v.id === box.virtualDiskIdRef) ?? disks[0];

      patchObject(spec, 'rootfs', rootfs => {
         patchString(rootfs, 'create_mode', rawMode => {
            const mode = box.rootFsSettings.createMode;
            if (!rawMode && mode === defaultVolumeCreateMode) {
               // не меняем дефолт
               return rawMode;
            }
            return rawRootfsCreateModes[mode] ?? rawMode;
         });

         patchList(rootfs, 'layer_refs', () => {
            const boxLayers: string[] = [];

            box.layers.forEach(boxLayer => {
               // eslint-disable-next-line no-underscore-dangle
               const layer = disk?.layers.find(v => v._ref === boxLayer._layerRef);

               if (layer) {
                  boxLayers.push(layer.id);
               }
            });

            return skipEmpty(boxLayers);
         });

         return rootfs;
      });

      patchList(spec, 'static_resources', staticResources =>
         BoxPatcher.patchStaticResources(box.staticResources, staticResources, disk),
      );

      patchList(spec, 'volumes', specVolumes => BoxPatcher.patchVolumes(box.volumes, specVolumes, disks));

      patchString(spec, 'resolv_conf', () =>
         box.resolvConf === EResolvConf.EResolvConf_DEFAULT ? undefined : box.resolvConf,
      );

      patchBoolean(spec, 'bind_skynet', () => box.bindSkynet);

      patchObject(spec, 'compute_resources', computeResources => {
         patchNumber(computeResources, 'vcpu_limit', () => box.cpuPerBox);
         patchNumber(computeResources, 'memory_limit', () => box.ramPerBox);
         patchNumber(computeResources, 'thread_limit', () => box.threadLimit);
         patchNumber(computeResources, 'anonymous_memory_limit', () => box.anonymousMemoryLimit);

         return computeResources;
      });

      patchList(spec, 'env', specEnv =>
         skipEmpty(patchEnvironment(specEnv, box.environment, this.visitor.secretResolver!, this.parentNodeIds)),
      );

      patchString(spec, 'cgroup_fs_mount_mode', () =>
         box.cgroupFsMountMode === CgroupFsMountMode.None ? undefined : rawCgroupfsModes[box.cgroupFsMountMode],
      );

      return spec;
   }
}
