import { BYTES, EnumOption } from '@yandex-infracloud-ui/libs';

import {
   EResourceAccessPermissions,
   ETransmitSystemLogs,
   EVolumePersistenceType,
   TDeployUnitSpec,
   TDeployUnitStatus,
} from '../../../../proto-typings';

import { Entity } from '../../../../redux/models';
import { IntRange } from '../../../../utils/helpers/intRange';
import {
   DEFAULT_DISK_ID,
   DEFAULT_DISK_SIZE,
   DEFAULT_HDD_BANDWIDTH_GUARANTEE,
   DEFAULT_LAYER_ID,
   DEFAULT_LAYER_REF,
   DEFAULT_LIVENESS_LIMIT_RATIO_FOR_DEFAULT_ENDPOINT_SET,
   DEFAULT_NETWORK_BANDWIDTH_GUARANTEE,
   DEFAULT_PORT_FOR_DEFAULT_ENDPOINT_SET,
   DEFAULT_PROTOCOL_FOR_DEFAULT_ENDPOINT_SET,
   DefaultOS,
   EMPTY_CHECKSUM,
   HDD_BANDWIDTH_LIMIT_FACTOR,
   SIMPLE_HTTP_SERVER_LAYER_ID,
   SIMPLE_HTTP_SERVER_LAYER_REF,
   SIMPLE_HTTP_SERVER_LAYER_URL,
} from '../../../constants';
import { LinkToSecret } from '../../secrets';
import { SidecarName, sidecarsUpdateConfig } from '../../Sidecars';

import { Box, getEmptyBox } from '../Box';
import { DeployUnitEmptyParams } from '../stage-levels';
import { YasmTags } from '../yasm';

// region DeployUnitType

export enum DeployUnitType {
   PerCluster = 'replica_set',
   MultiCluster = 'multi_cluster_replica_set',
}

export const DefaultDeployUnitType = DeployUnitType.PerCluster;

export const DeployUnitTypeOptions: EnumOption[] = [
   { value: DeployUnitType.PerCluster, title: 'Per-cluster replica set' },
   { value: DeployUnitType.MultiCluster, title: 'Multi-cluster replica set' },
];

// endregion

// region release rules
// TODO: это всё про релизные правила,
// возможно стоит туда перенести, чтобы не путать с ресурсами пода?

export enum ResourceType {
   Layer = 'layer',
   StaticResource = 'static_resource',
   DynamicResource = 'dynamic_resource',
}

export interface ResourceStatic extends Entity {
   type: ResourceType.Layer | ResourceType.StaticResource;
}

export interface ResourceDynamic extends Entity {
   type: ResourceType.DynamicResource;
   deployGroupMark: string;
}

export type Resource = ResourceStatic | ResourceDynamic;

// endregion

export enum AntiaffinityType {
   Node = 'node',
   Rack = 'rack',
}

export interface AntiaffinityRecord {
   perNode: number | null;
   perRack: number | null;
}

export const getDefaultAntiaffinity = (): AntiaffinityRecord => ({ perNode: null, perRack: 1 });

export interface DeployUnitEndpointSet {
   _order?: number;
   id: string | null;
   port: number | null;
   removed?: boolean;
   liveness_limit_ratio?: number | null;
   // protocol?: string;
}

export const DeployUnitDefaultEndpointSet = {
   port: DEFAULT_PORT_FOR_DEFAULT_ENDPOINT_SET,
   protocol: DEFAULT_PROTOCOL_FOR_DEFAULT_ENDPOINT_SET,
   liveness_limit_ratio: DEFAULT_LIVENESS_LIMIT_RATIO_FOR_DEFAULT_ENDPOINT_SET,
};

export interface DeployUnitNetwork {
   networkId: string;
   customSettings: boolean;
   virtualServiceIds: (string | null)[];
   ipv4AddressPoolId: string | null;
}

export interface DeployUnitNetworkBandwidth {
   limit: number | null;
   guarantee: number | null;
}

// region Static Resources

export const unmodifiedAccessPermissionsLabel = 'default'; // не очень понятно что такое "unmodified"

export enum StaticResourceType {
   Url = 'url',
   Files = 'files',
   // SkyGet = 'sky_get',
   Unknown = 'unknown',
}

export enum StaticResourceFileType {
   Raw = 'raw',
   Secret = 'secret',
   Unknown = 'unknown',
}

export interface RawFile {
   _order?: number;
   name: string | null;
   type: StaticResourceFileType.Raw;
   raw: string | null;
}

export interface SecretFile {
   _order?: number;
   name: string | null;
   type: StaticResourceFileType.Secret;
   secret: LinkToSecret | null;
}

export interface UnknownFile {
   _order?: number;
   name: string | null;
   type: StaticResourceFileType.Unknown;
}

export type StaticResourceFile = RawFile | SecretFile | UnknownFile;

export interface StaticResourceVerification {
   enabled: boolean;
   checksum: string | null;
}
export interface StaticResourceUrl extends Entity {
   _order?: number;
   verification: StaticResourceVerification;
   accessPermissions: EResourceAccessPermissions;
   type: StaticResourceType.Url;
   url: string;
   _ref: string;
   removed?: boolean;
}

export interface StaticResourceFiles extends Entity {
   _order?: number;
   verification: StaticResourceVerification;
   accessPermissions: EResourceAccessPermissions;
   type: StaticResourceType.Files;
   files: StaticResourceFile[];
   _ref: string;
   removed?: boolean;
}

export interface StaticResourceUnknown extends Entity {
   _order?: number;
   verification: StaticResourceVerification;
   accessPermissions: EResourceAccessPermissions;
   type: StaticResourceType.Unknown;
   _ref: string;
   removed?: boolean;
}

export type DiskStaticResource = StaticResourceUrl | StaticResourceFiles | StaticResourceUnknown;

// endregion

// region Disks

export enum DiskType {
   HDD = 'hdd',
   SSD = 'ssd',
}

export const DiskTypeOptions: EnumOption[] = [
   { value: DiskType.HDD, title: 'HDD' },
   { value: DiskType.SSD, title: 'SSD' },
];

export interface DiskBandwidthLimit {
   defaultSettings: boolean;
   default: number | null;
   custom: number | null;
}

export interface DiskBandwidth {
   guarantee: number | null;
   limit: DiskBandwidthLimit;
}

export interface DeployUnitDisk extends Entity {
   bandwidth: DiskBandwidth;
   size: number | null;
   type: DiskType;
   layers: DiskLayer[];
   staticResources: DiskStaticResource[];
   volumes: DiskVolume[];
}

export interface VolumeLayer {
   _layerRef: string | null; // ссылка на disk layer
}

export interface VolumeStaticResource {
   _order?: number;
   volumeRelativeMountPoint: string | null;
   _staticResourceRef: string | null; // ссылка на disk static resource
}

export interface DiskVolume {
   _order?: number;
   id: string;
   layers: VolumeLayer[];
   staticResources: VolumeStaticResource[];
   _ref: string;
   removed?: boolean;
   persistenceType: EVolumePersistenceType;
}

export const customBaseLayerLabel = 'custom';

export enum DiskLayerType {
   Url = 'url',
   Unknown = 'unknown',
}

export enum LayerSourceFileStoragePolicy {
   None = 'none',
   Keep = 'keep',
   Remove = 'remove',
}

export interface DiskLayer extends Entity {
   _order?: number;
   type: DiskLayerType;
   checksum: string;
   url?: string;
   layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy;
   _ref: string;
   removed?: boolean;
}

export const getDefaultDisk = (id?: string): DeployUnitDisk => ({
   id: id || DEFAULT_DISK_ID,
   type: DiskType.HDD,
   size: DEFAULT_DISK_SIZE,
   bandwidth: {
      guarantee: DEFAULT_HDD_BANDWIDTH_GUARANTEE,
      limit: {
         defaultSettings: true,
         default: DEFAULT_HDD_BANDWIDTH_GUARANTEE * HDD_BANDWIDTH_LIMIT_FACTOR,
         custom: DEFAULT_HDD_BANDWIDTH_GUARANTEE * HDD_BANDWIDTH_LIMIT_FACTOR,
      },
   },
   layers: [
      {
         type: DiskLayerType.Url,
         id: DEFAULT_LAYER_ID,
         _ref: DEFAULT_LAYER_REF,
         url: DefaultOS,
         checksum: EMPTY_CHECKSUM,
         layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
      },
      {
         type: DiskLayerType.Url,
         id: SIMPLE_HTTP_SERVER_LAYER_ID,
         _ref: SIMPLE_HTTP_SERVER_LAYER_REF,
         url: SIMPLE_HTTP_SERVER_LAYER_URL,
         checksum: EMPTY_CHECKSUM,
         layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
      },
   ],
   staticResources: [],
   volumes: [],
});

// endregion

export interface DeployUnitLocation {
   /**
    * Only for type === DeployUnitType.MultiCluster
    */
   antiaffinity: AntiaffinityRecord | null;

   enabled: boolean;

   /**
    * Only for type === DeployUnitType.PerCluster
    */
   disruptionBudget: number | null;

   /**
    * Only for type === DeployUnitType.PerCluster
    */
   maxTolerableDowntimeSeconds: number | null;

   /**
    * Only for type === DeployUnitType.PerCluster
    */
   maxTolerableDowntimePods: number | null;

   podCount: number;
}

export type DeployUnitLocationMap = Record<string, DeployUnitLocation>;

export interface PodNodeFilters {
   requireAvx2: boolean;
   requireAvx: boolean;
   requireBandwidth10G: boolean;
   requireIntel: boolean;
}

export interface CustomTopicRequest {
   topicName: string | null;
   tvmClientId: number | null;
   secret: LinkToSecret | null;
}

export interface LogbrokerDestroyPolicy {
   maxTries: number | null;
   restartPeriodMs: number | null;
}

export interface LogbrokerPodAdditionalResourcesRequest {
   setCpuToZero: boolean;
}

export interface LogbrokerConfig {
   customTopicRequest: CustomTopicRequest;
   destroyPolicy: LogbrokerDestroyPolicy;
   podAdditionalResourcesRequest: LogbrokerPodAdditionalResourcesRequest;
}

export interface InfraComponents {
   allowAutomaticUpdates: boolean;
}

export interface PatchersRevision {
   label: number | null;
   value: number | null;
}

export const podNodeFiltersMap: { [k in keyof PodNodeFilters]: string } = {
   requireAvx: '[/labels/cpu_flags/avx] = true',
   requireAvx2: '[/labels/cpu_flags/avx2] = true',
   requireBandwidth10G: '[/labels/extras/network/bandwidth_10G] = true',
   requireIntel: 'is_substr("Intel", string([/labels/cpu_model]))',
};

export interface AlertingSettings {
   state: boolean; // enum с запасом на будущее, но пока используется только одно значение
   notificationChannel: string | null;
}

export enum EnvironmentSettings {
   UNKNOWN = 'UNKNOWN',
   TESTING = 'TESTING',
   PRESTABLE = 'PRESTABLE',
   STABLE = 'STABLE',
}

export interface DeployUnitSettings {
   alerting: AlertingSettings;
   environment: EnvironmentSettings | null;
}

export interface DuYasm {
   yasmTags: YasmTags;
   podAgent: YasmPodAgent;
}

export interface YasmPodAgent {
   addPodAgentUserSignals: boolean;
}

export const getDefaultYasm = (): DuYasm => ({
   yasmTags: {
      itype: null,
      tags: [],
   },
   podAgent: {
      addPodAgentUserSignals: false,
   },
});

export const getDefaultYasmByItype = (itype?: string): DuYasm => {
   const yasm = getDefaultYasm();
   yasm.yasmTags.itype = itype ?? null;
   return yasm;
};

export interface DeployUnit extends Entity {
   anonymousMemoryLimit: number | null;

   /**
    * Only for type === DeployUnitType.PerCluster
    */
   antiaffinity: AntiaffinityRecord | null;
   boxes: Box[];
   cpu: number | null;
   defaultLayerSourceFileStoragePolicy: LayerSourceFileStoragePolicy;
   disks: DeployUnitDisk[];

   /**
    * Only for type === DeployUnitType.MultiCluster
    */
   disruptionBudget: number | null;

   /**
    * Only for type === DeployUnitType.MultiCluster
    */
   maxTolerableDowntimeSeconds: number | null;

   /**
    * Only for type === DeployUnitType.MultiCluster
    */
   maxTolerableDowntimePods: number | null;
   endpointSets: DeployUnitEndpointSet[];
   locations: DeployUnitLocationMap;
   networkDefaults: DeployUnitNetwork;
   networkBandwidth: DeployUnitNetworkBandwidth;
   nodeFilters: PodNodeFilters;
   ram: number | null;
   resources: Resource[];
   type: DeployUnitType;
   tvm: DeployUnitTvm;
   soxService: boolean;
   yasm: DuYasm;
   tempDiskIsolation: boolean; // https://st.yandex-team.ru/DEPLOY-3287 SPI-15289
   sidecars: Record<SidecarName, Sidecar>;
   podAgentSidecarDiskType: DiskType | null; // выбор типа диска для сайдкара подового агента для нескольких дисков
   logbrokerSidecarDiskType: DiskType | null; // выбор типа диска для сайдкара логов для нескольких дисков
   perLocationSettings: PerLocationSettings;
   revision: number | null;
   patchersRevision: PatchersRevision;
   logbrokerConfig: LogbrokerConfig;
   collectPortometricsFromSidecars: boolean;
   nodeSegmentId: string | null;
   infraComponents: InfraComponents;
   transmitSystemLogs: ETransmitSystemLogs;
   settings: DeployUnitSettings;

   /**
    * ID данное этому DU при заведении узла в форме.
    * В отличие от id - не редактируется и не меняется, вплоть до сохранения.
    */
   initialId?: string;
}

export interface DeployUnitRawRecord {
   id: string;
   spec: TDeployUnitSpec;
   status: TDeployUnitStatus | undefined;
}

/**
 * Все доступные пользователям ревизии патчеров
 */
export const activePatcherRevisions = new IntRange(1, 14);

/**
 * Ревизия, которая будет поставлена на бекенде при отсутствии значения
 */
export const defaultPatcherRevision = 1;

/**
 * Предпочитаемая ревизия патчера
 * Предлагается для новых деплой юнитов
 * НЕ предлагается для существующих деплой юнитов, так как смена ревизии может вызвать переаллокацию подов и иметь обратно несовместимые изменения
 */
export const preferredPatcherRevision = 13;

// region TVM

export enum TvmBlackbox {
   Prod = 'Prod',
   Test = 'Test',
   ProdYaTeam = 'ProdYaTeam',
   TestYaTeam = 'TestYaTeam',
   Stress = 'Stress',
}

export const TvmBlackboxOptions: EnumOption[] = [
   { value: TvmBlackbox.Prod, title: 'Prod' },
   { value: TvmBlackbox.Test, title: 'Test' },
   { value: TvmBlackbox.ProdYaTeam, title: 'ProdYaTeam' },
   { value: TvmBlackbox.TestYaTeam, title: 'TestYaTeam' },
   { value: TvmBlackbox.Stress, title: 'Stress' },
];

export enum TvmClientMode {
   GetCheck = 'getCheck',
   CheckOnly = 'checkOnly',
}

export const TvmClientModeOptions: EnumOption[] = [
   { value: TvmClientMode.GetCheck, title: 'Get/Check' },
   { value: TvmClientMode.CheckOnly, title: 'Check only' },
];

export interface TvmClientItemRecord {
   _order?: number;
   // abc?: string;
   app: number | null;
   alias: string | null;
}

export interface TvmClient {
   _order?: number;
   mode: TvmClientMode;
   source: TvmClientItemRecord;
   destinations?: TvmClientItemRecord[];
   secret: LinkToSecret | null;
}

export interface DeployUnitTvm {
   enabled: boolean;
   clientPort: number | null;
   blackbox: string;
   clients: TvmClient[];
   cpuLimit: number | null; // ms
   memoryLimit: number | null; // bytes
   diskType: DiskType | null;
}

export const getDefaultTvmClient = (): TvmClient => ({
   mode: TvmClientMode.GetCheck,
   source: {
      app: null,
      alias: null,
   },
   destinations: [
      {
         app: null,
         alias: null,
      },
   ],
   secret: null,
});

export const getDefaultTvm = (): DeployUnitTvm => ({
   enabled: false,
   clientPort: null,
   blackbox: TvmBlackbox.ProdYaTeam,
   clients: [getDefaultTvmClient()],
   cpuLimit: null,
   memoryLimit: null,
   diskType: null,
});

// endregion

const getDefaultSidecarsList = (): Record<SidecarName, Sidecar> => {
   const list = {} as Record<SidecarName, Sidecar>;
   for (const s of Object.keys(sidecarsUpdateConfig)) {
      const sidecarId: SidecarName = s as SidecarName;
      list[sidecarId] = {
         resourceRevision: null,
         overrideLabels: [],
         labelRevision: null,
      };
   }
   return list;
};

export interface OverrideLabel {
   key: string;
   value: string;
}

export interface Sidecar {
   resourceRevision: number | null;
   overrideLabels: OverrideLabel[];
   labelRevision: number | null;
}

export enum PerLocationStrategy {
   Parallel = 'parallel',
   Sequential = 'sequential',
}

export const defaultPerLocationStrategy = PerLocationStrategy.Parallel;

export interface PerLocationSettings {
   /** Явно указываем, что мы хотим менять спеку */
   isCustom: boolean;
   strategy: PerLocationStrategy;
   locationOrder: string[];
   needApproval: Set<string>;
}

// endregion

const { clusters } = window.CONFIG; // TODO use config.clusters
const locations: DeployUnitLocationMap = {};

if (clusters?.[0]?.value) {
   locations[clusters?.[0]?.value] = {
      podCount: 1,
      disruptionBudget: 1,
      antiaffinity: null,
      enabled: true,
      maxTolerableDowntimePods: null,
      maxTolerableDowntimeSeconds: null,
   } as DeployUnitLocation;
}

// дефолтный бокс со слоями
// используется для emptyDeployUnit, в котором есть эти слои
// в других деплой юнитах пользователей этих слоёв может не быть
const getDefaultBox = () => {
   const defaultBox = getEmptyBox();

   defaultBox.layers = [
      {
         _layerRef: DEFAULT_LAYER_REF,
      },
      {
         _layerRef: SIMPLE_HTTP_SERVER_LAYER_REF,
      },
   ];

   defaultBox.workloads.forEach(workload => {
      workload.commands.start.command = `/simple_http_server 80 'Hello my dear @${window.USER.login}!'`;
   });

   return defaultBox;
};

export const getEmptyDeployUnit = (params?: DeployUnitEmptyParams): DeployUnit => ({
   revision: 1,
   anonymousMemoryLimit: null,
   antiaffinity: DefaultDeployUnitType === DeployUnitType.PerCluster ? getDefaultAntiaffinity() : null,
   boxes: [getDefaultBox()],
   cpu: 100, // 100 VCPU, 0.1 CPU
   disks: [getDefaultDisk()],
   defaultLayerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
   disruptionBudget: DefaultDeployUnitType === DeployUnitType.PerCluster ? null : 1,
   maxTolerableDowntimeSeconds: null,
   maxTolerableDowntimePods: null,
   endpointSets: [
      {
         id: null,
         port: 80,
      },
   ],
   id: 'deployUnit',
   locations,
   networkDefaults: { networkId: '', customSettings: false, virtualServiceIds: [], ipv4AddressPoolId: null },
   networkBandwidth: {
      limit: null,
      guarantee: DEFAULT_NETWORK_BANDWIDTH_GUARANTEE, // https://st.yandex-team.ru/DEPLOY-4833
   },
   nodeFilters: {
      requireAvx2: false,
      requireAvx: false,
      requireBandwidth10G: false,
      requireIntel: false,
   },
   ram: BYTES.GB,
   resources: [],
   type: DefaultDeployUnitType,
   tvm: getDefaultTvm(),
   soxService: false,
   tempDiskIsolation: true,
   yasm: getDefaultYasmByItype(params?.projectId),
   sidecars: getDefaultSidecarsList(),
   perLocationSettings: {
      isCustom: true, // @nikolaichev: параметризировать по DeployUnitType
      strategy: PerLocationStrategy.Parallel,
      locationOrder: [],
      needApproval: new Set(),
   },
   patchersRevision: { label: null, value: preferredPatcherRevision },
   podAgentSidecarDiskType: null,
   logbrokerSidecarDiskType: null,
   logbrokerConfig: {
      customTopicRequest: {
         topicName: null,
         tvmClientId: null,
         secret: null,
      },
      destroyPolicy: {
         maxTries: null,
         restartPeriodMs: null,
      },
      podAdditionalResourcesRequest: {
         setCpuToZero: false,
      },
   },
   collectPortometricsFromSidecars: false,
   nodeSegmentId: null,
   infraComponents: {
      allowAutomaticUpdates: false,
   },
   transmitSystemLogs: ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE,
   settings: {
      alerting: {
         state: false,
         notificationChannel: null,
      },
      environment: EnvironmentSettings.STABLE,
   },
});

export const isDuAdvancedRequirementsEmpty = (du: DeployUnit) => !Object.values(du.nodeFilters).some(Boolean);
