/* eslint-disable no-underscore-dangle */
import { BYTES } from '@yandex-infracloud-ui/libs';

import {
   ELayerSourceFileStoragePolicy,
   EResourceAccessPermissions,
   ETransmitSystemLogs,
   EVolumePersistenceType,
   TDeployUnitSpec,
   TStageSpec_TDeployUnitSettings_TAlerting_EState,
   TStageSpec_TStageDynamicResourceSpec,
   TTvmConfig_EMode,
} from '../../../../proto-typings';

import { hexRegExp } from '../../../../utils';

import { DEFAULT_DISK_ID, HDD_BANDWIDTH_LIMIT_FACTOR, SSD_BANDWIDTH_LIMIT_FACTOR } from '../../../constants';

import { DeepPartial } from '../../../typeHelpers';

import {
   DeployUnitLocationMap,
   DeployUnitType,
   DiskLayerType,
   DiskType,
   LayerSourceFileStoragePolicy,
   Resource,
   ResourceType,
   StaticResourceType,
   TvmBlackbox,
   TvmClientMode,
} from './DeployUnit';

import { DeployUnitConverter } from './DeployUnitConverter';

// noinspection SpellCheckingInspection
const perClusterSpec: DeepPartial<TDeployUnitSpec> = {
   network_defaults: {
      network_id: '_SEARCHSAND_',
      ip4_address_pool_id: '1513:1381',
   },
   tvm_config: {
      mode: TTvmConfig_EMode.ENABLED,
      client_port: 123,
      blackbox_environment: TvmBlackbox.Stress,
      clients: [
         {
            source: {
               app_id: 1234,
               alias: '1234',
            },
         },
         {
            source: {
               app_id: 1234,
               alias: '1234',
            },
            destinations: [
               {
                  app_id: 2345,
                  alias: '2345',
               },
               {
                  app_id: 3456,
                  alias: '3456',
               },
            ],
            secret_selector: {
               alias: 'tvm_secret',
               id: 'secret_key',
            },
         },
         {
            source: {
               app_id: 1234,
               alias: '1234',
            },
            secret_selector: {
               alias: 'tvm_secret',
               id: 'secret_key',
            },
         },
         {
            source: {
               app_id: 1234,
               alias: '1234',
            },
            destinations: [
               {
                  app_id: 2345,
                  alias: '2345',
               },
               {
                  app_id: 3456,
                  alias: '3456',
               },
            ],
         },
      ],
   },
   endpoint_sets: [
      {},
      {
         id: 'es',
         port: 80,
         liveness_limit_ratio: 0.3,
      },
   ],
   logbroker_config: {
      destroy_policy: {
         max_tries: 3,
         restart_period_ms: 12000,
      },
   },
   replica_set: {
      replica_set_template: {
         pod_template_spec: {
            spec: {
               resource_requests: {
                  vcpu_guarantee: 100,
                  vcpu_limit: 100,
                  memory_guarantee: 1073741824,
                  memory_limit: 1073741824,
               },
               // TODO: @nodejsgirl тесты на (не) кастомные настройки
               ip6_address_requests: [
                  {
                     network_id: '_SEARCHSAND_',
                     vlan_id: 'backbone',
                     enable_dns: true,
                  },
                  {
                     network_id: '_SEARCHSAND_',
                     vlan_id: 'fastbone',
                     enable_dns: true,
                  },
               ],
               disk_volume_requests: [
                  {
                     id: `${DEFAULT_DISK_ID}-SSD`,
                     storage_class: DiskType.SSD,
                     quota_policy: {
                        capacity: 10 * BYTES.GB,
                        bandwidth_guarantee: 20 * BYTES.MB,
                        bandwidth_limit: 20 * BYTES.MB * SSD_BANDWIDTH_LIMIT_FACTOR,
                     },
                     labels: {
                        x: 3,
                        used_by_infra: true,
                     },
                  },
               ],
               pod_agent_payload: {
                  spec: {
                     volumes: [
                        {
                           id: 'volume1',
                           // virtual_disk_id_ref: undefined
                        },
                        {
                           id: 'volume2',
                           virtual_disk_id_ref: `${DEFAULT_DISK_ID}-SSD`,
                           generic: {
                              layer_refs: ['volume_layer'],
                           },
                        },
                        {
                           id: 'volume3',
                           virtual_disk_id_ref: 'incorrect_virtual_disk_id_ref',
                        },
                     ],
                     resources: {
                        layers: [
                           {
                              id: 'base-layer-0',
                              checksum: 'EMPTY:',
                              url: 'sbr:2061235470',
                              // virtual_disk_id_ref: undefined
                           },
                           {
                              id: 'simple_http_server',
                              checksum: 'EMPTY:',
                              url: 'rbtorrent:simplehttpserver',
                              virtual_disk_id_ref: `${DEFAULT_DISK_ID}-SSD`,
                              layer_source_file_storage_policy:
                                 ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP,
                           },
                           {
                              id: 'volume_layer',
                              checksum: 'EMPTY:',
                              url: 'rbtorrent:volume',
                              layer_source_file_storage_policy:
                                 ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_REMOVE,
                           },
                           {
                              id: 'meta',
                              meta: {
                                 sandbox_resource: {
                                    task_type: 'SAMPLE_RELEASE_TO_YA_DEPLOY',
                                    task_id: '636599712',
                                    resource_type: 'SAMPLE_RELEASE_INTEGRATION_LAYER',
                                    resource_id: '1414625542',
                                 },
                              },
                              checksum: 'MD5:md5',
                              url: 'rbtorrent:meta',
                           },
                        ],
                        static_resources: [
                           {
                              id: 'my-static',
                              meta: {
                                 sandbox_resource: {
                                    task_type: 'SAMPLE_RELEASE_TO_YA_DEPLOY',
                                    task_id: '636599712',
                                    resource_type: 'SAMPLE_RELEASE_INTEGRATION_STATIC',
                                    resource_id: '1414625516',
                                 },
                              },
                              verification: {
                                 checksum: 'MD5:md5',
                                 check_period_ms: 180000,
                              },
                              url: 'rbtorrent:mystatic',
                           },
                           {
                              id: 'my-static-2',
                              meta: {
                                 sandbox_resource: {
                                    task_type: 'SAMPLE_RELEASE_TO_YA_DEPLOY',
                                    task_id: '636599712',
                                    resource_type: 'SAMPLE_RELEASE_INTEGRATION_STATIC',
                                    resource_id: '1414625516',
                                 },
                              },
                              verification: {
                                 disabled: true,
                              },
                              url: 'rbtorrent:mystatic',
                           },
                        ],
                     },
                  },
               },
               secrets: {
                  'tvm_secret': {
                     delegation_token: 'delegation_token',
                     secret_id: 'sec-0123456789',
                     secret_version: 'ver-0123456789',
                  },
               },
               // TODO: @nodejsgirl тесты
               host_infra: {
                  monitoring: {},
               },
            },
         },
         constraints: {
            antiaffinity_constraints: [
               {
                  key: 'rack',
                  max_pods: 1,
               },
            ],
         },
      },
      per_cluster_settings: {
         sas: {
            pod_count: 1,
            deployment_strategy: {
               max_unavailable: 1,
            },
         },
         vla: {
            deployment_strategy: {
               max_unavailable: 1,
            },
         },
      },
   },
   // sidecars
   pod_agent_sandbox_info: { revision: 1234567890 },
   dynamic_resource_updater_sandbox_info: { revision: 2345678901 },
   logbroker_tools_sandbox_info: { revision: 3456789012 },
   tvm_sandbox_info: { revision: 4567890123 },
};

// noinspection SpellCheckingInspection
const multiClusterSpec: DeepPartial<TDeployUnitSpec> = {
   network_defaults: {
      network_id: '_ZAPRAVKI_PROD_NETS_',
      virtual_service_ids: [
         'test-awacs-balancer1.yandex-team.ru',
         'test-awacs-balancer2.yandex-team.ru',
         'test-awacs-balancer3.yandex-team.ru',
      ],
   },
   images_for_boxes: {
      Kibana: {
         registry_host: 'registry.yandex.net',
         name: 'zapravki/elk/kibana',
         tag: 'deploy-old-version-9',
      },
      'logstash-box': {
         registry_host: 'registry.yandex.net',
         name: 'zapravki/elk/logstash',
         tag: 'deploy-old-version-15',
      },
   },
   // logbroker_config: {
   //    destroy_policy: {
   //       max_tries: 3,
   //       restart_period_ms: 10000,
   //    },
   // },
   multi_cluster_replica_set: {
      replica_set: {
         pod_template_spec: {
            spec: {
               resource_requests: {
                  vcpu_guarantee: 8000,
                  vcpu_limit: 8000,
                  memory_guarantee: 25769803776,
                  memory_limit: 25769803776,
               },
               // TODO: @nodejsgirl тесты на (не) кастомные настройки
               ip6_address_requests: [
                  {
                     network_id: '_ZAPRAVKI_PROD_NETS_',
                     vlan_id: 'backbone',
                     enable_dns: true,
                  },
                  {
                     network_id: '_ZAPRAVKI_PROD_NETS_',
                     vlan_id: 'fastbone',
                     enable_dns: true,
                  },
               ],
               disk_volume_requests: [
                  {
                     id: `${DEFAULT_DISK_ID}-HDD`,
                     storage_class: DiskType.HDD,
                     quota_policy: {
                        capacity: 15 * BYTES.GB,
                        bandwidth_guarantee: 25 * BYTES.MB,
                        bandwidth_limit: 100 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                     },
                     labels: {
                        x: 3,
                        used_by_infra: true,
                     },
                  },
                  {
                     id: `${DEFAULT_DISK_ID}-HDD-2`,
                     storage_class: DiskType.HDD,
                     // quota_policy: {
                     //    capacity: 15 * BYTES.GB,
                     //    bandwidth_guarantee: 25 * BYTES.MB,
                     //    bandwidth_limit: 100 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                     // },
                  },
               ],
               pod_agent_payload: {
                  spec: {
                     resources: {
                        // TODO: для мультикластера не надо отдельный тест?
                        layers: [
                           {
                              virtual_disk_id_ref: `${DEFAULT_DISK_ID}-HDD`,
                              id: 'RootFSLayer',
                              checksum: 'MD5:376d92002ab78f5bee7f39e09b23bc27',
                              sky_get: { resid: 'rbtorrent:cb372d6907734cda29a63145825a7c0ddbb3a026' },
                           },
                        ],
                        default_layer_source_file_storage_policy:
                           ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP,
                     },
                     volumes: [
                        {
                           id: 'volume1',
                           virtual_disk_id_ref: `${DEFAULT_DISK_ID}-HDD-2`,
                        },
                        {
                           id: 'volume2',
                           virtual_disk_id_ref: `${DEFAULT_DISK_ID}-HDD`,
                        },
                        {
                           id: 'volume3',
                           virtual_disk_id_ref: `${DEFAULT_DISK_ID}-HDD-2`,
                        },
                        {
                           id: 'incorrect_volume1',
                           // virtual_disk_id_ref: undefined,
                        },
                        {
                           id: 'volume4',
                           virtual_disk_id_ref: `${DEFAULT_DISK_ID}-HDD`,
                        },
                        {
                           id: 'incorrect_volume2',
                           virtual_disk_id_ref: 'incorrect_virtual_disk_id_ref',
                        },
                     ],
                     transmit_system_logs_policy: {
                        transmit_system_logs: ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED,
                     },
                  },
               },
               // TODO: @nodejsgirl тесты
               host_infra: {
                  monitoring: {},
               },
            },
         },
         deployment_strategy: {
            max_unavailable: 2,
         },
         clusters: [
            {
               cluster: 'man',
               spec: {
                  replica_count: 1,
                  constraints: {
                     antiaffinity_constraints: [
                        {
                           key: 'rack',
                           max_pods: 1,
                        },
                     ],
                  },
               },
            },
            {
               cluster: 'myt',
               spec: {
                  replica_count: 1,
                  constraints: {
                     antiaffinity_constraints: [
                        {
                           key: 'rack',
                           max_pods: 1,
                        },
                     ],
                  },
               },
            },
            {
               cluster: 'sas',
               spec: {
                  replica_count: 1,
                  constraints: {
                     antiaffinity_constraints: [
                        {
                           key: 'rack',
                           max_pods: 1,
                        },
                     ],
                  },
               },
            },
         ],
      },
   },
   // empty tvm
   // tvm_config: {},

   // empty endpoint_sets
   // endpoint_sets: [],

   // empty sidecars
   // pod_agent_sandbox_info: {},
   // dynamic_resource_updater_sandbox_info: {},
   // logbroker_tools_sandbox_info: {},
   // tvm_sandbox_info: {},
};

const myDynamic = {
   deploy_unit_ref: 'xxx',
   dynamic_resource: {
      deploy_groups: [
         {
            mark: 'all',
            storage_options: {
               box_ref: 'phantom2d',
               cached_revisions_count: 3,
               destination: '/resources/uatraits',
               storage_dir: '/resources/storage/uatraits',

               // http_action: undefined,
               // exec_action: undefined,
               // verification: {
               //    checksum: 'EMPTY:',
               //    check_period_ms: 60000,
               // },
               // allow_deduplication: false,
               // max_download_speed: 5,
            },
            urls: ['sbr:1234567890'],
         },
      ],
      revision: 9,
      update_window: 99,
   },
} as TStageSpec_TStageDynamicResourceSpec;

describe('models/ui|DeployUnitConverter', () => {
   describe('per cluster deploy unit type', () => {
      const du = DeployUnitConverter.fromApi(
         {
            id: 'xxx',
            spec: perClusterSpec as TDeployUnitSpec,
            status: undefined,
         },
         {},
         {
            'my-dynamic': myDynamic,
         },
         {
            'xxx': {
               alerting: {
                  state: TStageSpec_TDeployUnitSettings_TAlerting_EState.ENABLED,
                  notification_channels: { 'ERROR': 'channel' },
               },
            } as any,
         },
      );

      it('should extract id', () => {
         expect(du.id).toBe('xxx');
      });

      it('should extract defaultLayerSourceFileStoragePolicy', () => {
         expect(du.defaultLayerSourceFileStoragePolicy).toBe(LayerSourceFileStoragePolicy.None);
      });

      it('should detect type', () => {
         expect(du.type).toBe(DeployUnitType.PerCluster);
      });

      it('should extract resources', () => {
         expect(du.resources).toEqual([
            { id: 'base-layer-0', type: ResourceType.Layer },
            { id: 'simple_http_server', type: ResourceType.Layer },
            { id: 'volume_layer', type: ResourceType.Layer },
            { id: 'meta', type: ResourceType.Layer },
            { id: 'my-static', type: ResourceType.StaticResource },
            { id: 'my-static-2', type: ResourceType.StaticResource },
            { id: 'my-dynamic', type: ResourceType.DynamicResource, deployGroupMark: 'all' },
         ] as Resource[]);
      });

      it('should extract transmit_system_logs', () => {
         expect(du.transmitSystemLogs).toEqual(ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE);
      });

      it('should extract disruptionBudget', () => {
         expect(du.disruptionBudget).toBeNull();
      });

      it('should extract locations', () => {
         expect(du.locations).toEqual({
            sas: {
               antiaffinity: null,
               disruptionBudget: 1,
               enabled: true,
               maxTolerableDowntimePods: null,
               maxTolerableDowntimeSeconds: null,
               podCount: 1,
            },
            vla: {
               antiaffinity: null,
               disruptionBudget: 1,
               enabled: true,
               maxTolerableDowntimePods: null,
               maxTolerableDowntimeSeconds: null,
               podCount: 0,
            },
         } as DeployUnitLocationMap);
      });

      it('should extract disks: SSD, default limit', () => {
         const volumeRef0 = du.disks[0].volumes[0]._ref;
         const volumeRef1 = du.disks[0].volumes[1]._ref;
         const volumeRef2 = du.disks[0].volumes[2]._ref;

         const layerRef0 = du.disks[0].layers[0]._ref;
         const layerRef1 = du.disks[0].layers[1]._ref;
         const layerRef2 = du.disks[0].layers[2]._ref;
         const layerRef3 = du.disks[0].layers[3]._ref;

         // eslint-disable-next-line no-underscore-dangle
         const layerVolumeRef = du.disks[0].volumes[1].layers[0]._layerRef;

         const staticResourceRef0 = du.disks[0].staticResources[0]._ref;
         const staticResourceRef1 = du.disks[0].staticResources[1]._ref;

         // рандомные значения, проверяем только формат
         expect(volumeRef0).toMatch(hexRegExp);
         expect(volumeRef1).toMatch(hexRegExp);
         expect(volumeRef2).toMatch(hexRegExp);

         expect(layerRef0).toMatch(hexRegExp);
         expect(layerRef1).toMatch(hexRegExp);
         expect(layerRef2).toMatch(hexRegExp);
         expect(layerRef3).toMatch(hexRegExp);

         expect(staticResourceRef0).toMatch(hexRegExp);
         expect(staticResourceRef1).toMatch(hexRegExp);

         expect(du.disks).toEqual([
            {
               id: `${DEFAULT_DISK_ID}-SSD`,
               type: DiskType.SSD,
               bandwidth: {
                  guarantee: 20 * BYTES.MB,
                  limit: {
                     defaultSettings: true,
                     default: 20 * BYTES.MB * SSD_BANDWIDTH_LIMIT_FACTOR,
                     custom: 20 * BYTES.MB * SSD_BANDWIDTH_LIMIT_FACTOR,
                  },
               },
               size: 10 * BYTES.GB,
               volumes: [
                  {
                     _order: 0,
                     id: 'volume1',
                     layers: [],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef0,
                  },
                  {
                     _order: 1,
                     id: 'volume2',
                     layers: [
                        {
                           _layerRef: layerVolumeRef,
                        },
                     ],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef1,
                  },
                  {
                     _order: 2,
                     id: 'volume3',
                     layers: [],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef2,
                  },
               ],
               layers: [
                  {
                     _order: 0,
                     id: 'base-layer-0',
                     type: DiskLayerType.Url,
                     url: 'sbr:2061235470',
                     checksum: 'EMPTY:',
                     _ref: layerRef0,
                     layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
                  },
                  {
                     _order: 1,
                     id: 'simple_http_server',
                     type: DiskLayerType.Url,
                     url: 'rbtorrent:simplehttpserver',
                     checksum: 'EMPTY:',
                     _ref: layerRef1,
                     layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.Keep,
                  },
                  {
                     _order: 2,
                     id: 'volume_layer',
                     type: DiskLayerType.Url,
                     url: 'rbtorrent:volume',
                     checksum: 'EMPTY:',
                     _ref: layerRef2,
                     layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.Remove,
                  },
                  {
                     _order: 3,
                     id: 'meta',
                     type: DiskLayerType.Url,
                     // meta: {
                     //    sandbox_resource: {
                     //       task_type: 'SAMPLE_RELEASE_TO_YA_DEPLOY',
                     //       task_id: '636599712',
                     //       resource_type: 'SAMPLE_RELEASE_INTEGRATION_LAYER',
                     //       resource_id: '1414625542',
                     //    },
                     // },
                     url: 'rbtorrent:meta',
                     checksum: 'MD5:md5',
                     _ref: layerRef3,
                     layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
                  },
               ],
               staticResources: [
                  {
                     _order: 0,
                     id: 'my-static',
                     type: StaticResourceType.Url,
                     url: 'rbtorrent:mystatic',
                     accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                     verification: { enabled: true, checksum: 'MD5:md5' },
                     _ref: staticResourceRef0,
                  },
                  {
                     _order: 1,
                     id: 'my-static-2',
                     type: StaticResourceType.Url,
                     url: 'rbtorrent:mystatic',
                     accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                     verification: { enabled: false, checksum: null },
                     _ref: staticResourceRef1,
                  },
               ],
            },
         ]);
      });

      it('should extract endpoint sets', () => {
         expect(du.endpointSets).toEqual([
            {
               _order: 0,
               id: null,
               liveness_limit_ratio: null,
               port: null,
            },
            {
               _order: 1,
               id: 'es',
               liveness_limit_ratio: 0.3,
               port: 80,
            },
         ]);
      });

      it('should extract tvm_config settings', () => {
         expect(du.tvm).toEqual({
            enabled: true,
            clientPort: 123,
            blackbox: TvmBlackbox.Stress,
            cpuLimit: null,
            memoryLimit: null,
            diskType: null,
            clients: [
               {
                  _order: 0,
                  mode: TvmClientMode.CheckOnly,
                  source: {
                     app: 1234,
                     alias: '1234',
                  },
                  destinations: [
                     {
                        app: null,
                        alias: null,
                     },
                  ],
                  secret: null,
               },
               {
                  _order: 1,
                  mode: TvmClientMode.GetCheck,
                  source: {
                     app: 1234,
                     alias: '1234',
                  },
                  destinations: [
                     {
                        _order: 0,
                        app: 2345,
                        alias: '2345',
                     },
                     {
                        _order: 1,
                        app: 3456,
                        alias: '3456',
                     },
                  ],
                  secret: {
                     alias: 'tvm_secret',
                     key: 'secret_key',
                  },
               },
               {
                  _order: 2,
                  mode: TvmClientMode.GetCheck,
                  source: {
                     app: 1234,
                     alias: '1234',
                  },
                  destinations: [
                     {
                        app: null,
                        alias: null,
                     },
                  ],
                  secret: {
                     alias: 'tvm_secret',
                     key: 'secret_key',
                  },
               },
               {
                  _order: 3,
                  mode: TvmClientMode.GetCheck,
                  source: {
                     app: 1234,
                     alias: '1234',
                  },
                  destinations: [
                     {
                        _order: 0,
                        app: 2345,
                        alias: '2345',
                     },
                     {
                        _order: 1,
                        app: 3456,
                        alias: '3456',
                     },
                  ],
                  secret: null,
               },
            ],
         });
      });

      it('should extract empty network', () => {
         expect(du.networkDefaults.virtualServiceIds).toEqual([]);
      });

      it('should extract logbroker destroy policy', () => {
         expect(du.logbrokerConfig.destroyPolicy).toEqual({
            maxTries: 3,
            restartPeriodMs: 12000,
         });
      });

      it('should extract sidecars', () => {
         expect(du.sidecars).toEqual({
            PodAgent: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: 1234567890,
            },
            DynamicResource: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: 2345678901,
            },
            Logbroker: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: 3456789012,
            },
            TVM: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: 4567890123,
            },
         });
      });

      it('should extract deploy unit settings: alerting', () => {
         expect(du.settings.alerting).toEqual({
            state: true,
            notificationChannel: 'channel',
         });
      });

      it('should extract ipv4 Address pool id', () => {
         expect(du.networkDefaults.ipv4AddressPoolId).toEqual('1513:1381');
      });
   });

   describe('multi-cluster deploy unit type', () => {
      const du = DeployUnitConverter.fromApi(
         {
            id: 'yyy',
            spec: multiClusterSpec as TDeployUnitSpec,
            status: undefined,
         },
         {},
         {},
         {}, // empty alerting
      );

      it('should extract id', () => {
         expect(du.id).toBe('yyy');
      });

      it('should extract defaultLayerSourceFileStoragePolicy', () => {
         expect(du.defaultLayerSourceFileStoragePolicy).toBe(LayerSourceFileStoragePolicy.Keep);
      });

      it('shoud extract network defaults', () => {
         expect(du.networkDefaults.networkId).toEqual('_ZAPRAVKI_PROD_NETS_');
         expect(du.networkDefaults.virtualServiceIds).toEqual([
            'test-awacs-balancer1.yandex-team.ru',
            'test-awacs-balancer2.yandex-team.ru',
            'test-awacs-balancer3.yandex-team.ru',
         ]);
      });

      it('should detect type', () => {
         expect(du.type).toBe(DeployUnitType.MultiCluster);
      });

      it('should extract resources', () => {
         expect(du.resources).toEqual([
            {
               id: 'RootFSLayer',
               type: 'layer',
            },
         ] as Resource[]);
      });

      it('should extract transmit_system_logs', () => {
         expect(du.transmitSystemLogs).toEqual(ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED);
      });

      it('should extract disruptionBudget', () => {
         expect(du.disruptionBudget).toBe(2);
      });

      it('should extract locations', () => {
         expect(du.locations).toEqual({
            man: {
               antiaffinity: { perNode: null, perRack: 1 },
               disruptionBudget: null,
               enabled: true,
               maxTolerableDowntimePods: null,
               maxTolerableDowntimeSeconds: null,
               podCount: 1,
            },
            myt: {
               antiaffinity: { perNode: null, perRack: 1 },
               disruptionBudget: null,
               enabled: true,
               maxTolerableDowntimePods: null,
               maxTolerableDowntimeSeconds: null,
               podCount: 1,
            },
            sas: {
               antiaffinity: { perNode: null, perRack: 1 },
               disruptionBudget: null,
               enabled: true,
               maxTolerableDowntimePods: null,
               maxTolerableDowntimeSeconds: null,
               podCount: 1,
            },
         } as DeployUnitLocationMap);
      });

      it('should extract disks: HDD, custom limit', () => {
         const volumeRef1 = du.disks[0].volumes[0]._ref;
         const volumeRef4 = du.disks[0].volumes[1]._ref;

         const volumeRef0 = du.disks[1].volumes[0]._ref;
         const volumeRef2 = du.disks[1].volumes[1]._ref;

         const layerRef1 = du.disks[0].layers[0]._ref;

         // рандомные значения, проверяем только формат
         expect(volumeRef0).toMatch(hexRegExp);
         expect(volumeRef1).toMatch(hexRegExp);
         expect(volumeRef2).toMatch(hexRegExp);
         expect(volumeRef4).toMatch(hexRegExp);

         expect(du.disks).toEqual([
            {
               id: `${DEFAULT_DISK_ID}-HDD`,
               type: DiskType.HDD,
               bandwidth: {
                  guarantee: 25 * BYTES.MB,
                  limit: {
                     defaultSettings: false,
                     default: 25 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                     custom: 100 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                  },
               },
               size: 15 * BYTES.GB,
               volumes: [
                  {
                     _order: 1,
                     id: 'volume2',
                     layers: [],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef1,
                  },
                  {
                     _order: 4,
                     id: 'volume4',
                     layers: [],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef4,
                  },
               ],
               layers: [
                  {
                     _order: 0,
                     id: 'RootFSLayer',
                     checksum: 'MD5:376d92002ab78f5bee7f39e09b23bc27',
                     // sky_get: { resid: 'rbtorrent:cb372d6907734cda29a63145825a7c0ddbb3a026' },
                     type: DiskLayerType.Unknown,
                     _ref: layerRef1,
                     layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
                  },
               ],
               staticResources: [],
            },
            {
               id: `${DEFAULT_DISK_ID}-HDD-2`,
               type: DiskType.HDD,
               bandwidth: {
                  guarantee: null, // 25 * BYTES.MB,
                  limit: {
                     defaultSettings: true, // false,
                     default: null, // 25 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                     custom: null, // 100 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                  },
               },
               size: null, // 15 * BYTES.GB,
               volumes: [
                  {
                     _order: 0,
                     id: 'volume1',
                     layers: [],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef0,
                  },
                  {
                     _order: 2,
                     id: 'volume3',
                     layers: [],
                     staticResources: [],
                     persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                     _ref: volumeRef2,
                  },
                  // {
                  //    id: 'incorrect_volume1',
                  // },
                  // {
                  //    id: 'incorrect_volume2',
                  // },
               ],
               layers: [],
               staticResources: [],
            },
         ]);
      });

      it('should extract empty endpoint sets', () => {
         expect(du.endpointSets).toEqual([]);
      });

      it('should extract empty tvm_config settings', () => {
         expect(du.tvm).toEqual({
            enabled: false,
            clientPort: null,
            blackbox: TvmBlackbox.ProdYaTeam,
            cpuLimit: null,
            memoryLimit: null,
            diskType: null,
            clients: [
               {
                  mode: TvmClientMode.GetCheck,
                  source: {
                     app: null,
                     alias: null,
                  },
                  destinations: [
                     {
                        app: null,
                        alias: null,
                     },
                  ],
                  secret: null,
               },
            ],
         });
      });

      it('should extract logbroker destroy policy', () => {
         expect(du.logbrokerConfig.destroyPolicy).toEqual({
            maxTries: null,
            restartPeriodMs: null,
         });
      });

      it('should extract empty sidecars', () => {
         expect(du.sidecars).toEqual({
            PodAgent: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: null,
            },
            DynamicResource: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: null,
            },
            Logbroker: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: null,
            },
            TVM: {
               labelRevision: null,
               overrideLabels: [],
               resourceRevision: null,
            },
         });
      });

      it('should extract deploy unit settings: empty alerting', () => {
         expect(du.settings.alerting).toEqual({
            state: false,
            notificationChannel: null,
         });
      });
   });

   describe('getNodeFilters', () => {
      it('avx + avx2 + bandwidth', () => {
         expect(
            DeployUnitConverter.getNodeFilters(
               '[/labels/cpu_flags/avx] = true AND [/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true',
            ),
         ).toMatchInlineSnapshot(`
            Object {
              "requireAvx": true,
              "requireAvx2": true,
              "requireBandwidth10G": true,
              "requireIntel": false,
            }
         `);
      });
   });
});
