import { BYTES, IJsonable, isEmpty, isEqual } from '@yandex-infracloud-ui/libs';

import {
   ELayerSourceFileStoragePolicy,
   EResourceAccessPermissions,
   ETransmitSystemLogs,
   EVolumePersistenceType,
   TAntiaffinityConstraint,
   TDeployUnitSpec,
   TDeployUnitSpec_TDeploySettings,
   TDeployUnitSpec_TDeploySettings_EDeployStrategy,
   TNetworkDefaults,
   TPodSpec_TIP6AddressRequest,
   TPodTemplateSpec,
   TSidecarVolume_EStorageClass,
   TStage,
   TStageSpec,
   TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment,
   TStageSpec_TDeployUnitSettings_TAlerting_EState,
   TTvmConfig,
   TTvmConfig_EMode,
} from '../../../../proto-typings';

import { getHexRef } from '../../../../utils';
import { DEFAULT_DISK_ID, HDD_BANDWIDTH_LIMIT_FACTOR, SSD_BANDWIDTH_LIMIT_FACTOR } from '../../../constants';
import {
   generateMultisecretKey,
   getLinkToSecret,
   getSecretUsagePath,
   SecretUsage,
   SecretUsageType,
} from '../../secrets';
import {
   SidecarName,
   SidecarsForUpdating,
   SidecarsForUpdatingUnion,
   SidecarsSpecPathsType,
   sidecarsUpdateConfig,
} from '../../Sidecars';

import { Box, BoxConverter } from '../Box';
import { convertYasmLabels, YasmTags } from '../yasm';

import {
   AntiaffinityRecord,
   AntiaffinityType,
   DefaultDeployUnitType,
   defaultPerLocationStrategy,
   DeployUnit,
   DeployUnitDisk,
   DeployUnitEndpointSet,
   DeployUnitLocationMap,
   DeployUnitNetwork,
   DeployUnitNetworkBandwidth,
   DeployUnitRawRecord,
   DeployUnitSettings,
   DeployUnitTvm,
   DeployUnitType,
   DiskLayer,
   DiskLayerType,
   DiskStaticResource,
   DiskType,
   DiskVolume,
   DuYasm,
   getDefaultTvm,
   InfraComponents,
   LayerSourceFileStoragePolicy,
   LogbrokerConfig,
   PatchersRevision,
   PerLocationSettings,
   PerLocationStrategy,
   PodNodeFilters,
   podNodeFiltersMap,
   RawFile,
   Resource,
   ResourceType,
   SecretFile,
   Sidecar,
   StaticResourceFiles,
   StaticResourceFileType,
   StaticResourceType,
   StaticResourceUnknown,
   StaticResourceUrl,
   TvmClient,
   TvmClientMode,
   UnknownFile,
   VolumeLayer,
   VolumeStaticResource,
   EnvironmentSettings,
} from './DeployUnit';

const layerSourceFileStoragePolicies: Record<ELayerSourceFileStoragePolicy, LayerSourceFileStoragePolicy> = {
   [ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_NONE]: LayerSourceFileStoragePolicy.None,
   [ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP]: LayerSourceFileStoragePolicy.Keep,
   [ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_REMOVE]: LayerSourceFileStoragePolicy.Remove,
};

// @nikolaichev монолит, необходимо разделить
export class DeployUnitConverter implements DeployUnit, IJsonable {
   public static fromApi(
      raw: DeployUnitRawRecord,
      labels: TStage['labels'],
      stageDynamicResources: TStageSpec['dynamic_resources'] | undefined,
      deployUnitsSettings: TStageSpec['deploy_unit_settings'] | undefined,
   ): DeployUnit {
      return new DeployUnitConverter(raw, labels, stageDynamicResources, deployUnitsSettings).toJSON();
   }

   public static getDisks(podTemplateSpec: TPodTemplateSpec): DeployUnitDisk[] {
      const diskVolumeRequests = podTemplateSpec.spec?.disk_volume_requests ?? [];
      const countDisks = diskVolumeRequests.length;

      return diskVolumeRequests.map(disk => {
         const diskType = disk.storage_class === DiskType.SSD ? DiskType.SSD : DiskType.HDD;

         const guarantee = disk.quota_policy?.bandwidth_guarantee ?? null;
         const limit = disk.quota_policy?.bandwidth_limit ?? null;

         const defaultHddLimit = guarantee ? guarantee * HDD_BANDWIDTH_LIMIT_FACTOR : null;
         const defaultSsdLimit = guarantee ? guarantee * SSD_BANDWIDTH_LIMIT_FACTOR : null;

         const layers: DiskLayer[] = [];
         const staticResources: DiskStaticResource[] = [];
         const volumes: DiskVolume[] = [];

         const podAgentPayloadSpec = podTemplateSpec.spec?.pod_agent_payload?.spec;

         (podAgentPayloadSpec?.resources?.layers ?? []).forEach((layer, i) => {
            if ((layer.virtual_disk_id_ref && layer.virtual_disk_id_ref === disk.id) || countDisks === 1) {
               layers.push({
                  _order: i,
                  id: layer.id,
                  type: layer.url !== undefined ? DiskLayerType.Url : DiskLayerType.Unknown,
                  ...(layer.url ? { url: layer.url } : {}),
                  checksum: layer.checksum,
                  layerSourceFileStoragePolicy:
                     layerSourceFileStoragePolicies[
                        layer.layer_source_file_storage_policy ?? layerSourceFileStoragePolicies.none
                     ],
                  _ref: getHexRef(),
               });
            }
         });

         (podAgentPayloadSpec?.resources?.static_resources ?? []).forEach((staticResource, staticResourceIndex) => {
            if (
               (staticResource.virtual_disk_id_ref && staticResource.virtual_disk_id_ref === disk.id) ||
               countDisks === 1
            ) {
               if (staticResource.url) {
                  const resource: StaticResourceUrl = {
                     _order: staticResourceIndex,
                     id: staticResource.id,
                     type: StaticResourceType.Url,
                     url: staticResource.url,
                     verification: {
                        enabled:
                           staticResource.verification?.disabled !== undefined
                              ? !staticResource.verification.disabled
                              : true,
                        checksum: staticResource.verification?.checksum ?? null,
                     },
                     accessPermissions:
                        staticResource.access_permissions ??
                        EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                     _ref: getHexRef(),
                  };
                  staticResources.push(resource);
               } else if (staticResource.files) {
                  const files =
                     staticResource.files.files?.map((file, fileIndex) => {
                        const name = file.file_name;

                        if (file.raw_data) {
                           return {
                              _order: fileIndex,
                              type: StaticResourceFileType.Raw,
                              name,
                              raw: file.raw_data,
                           } as RawFile;
                        }

                        if (file.secret_data) {
                           return {
                              _order: fileIndex,
                              type: StaticResourceFileType.Secret,
                              name,
                              secret: file.secret_data ? getLinkToSecret(file.secret_data) : undefined,
                           } as SecretFile;
                        }

                        if (file.multi_secret_data) {
                           return {
                              _order: fileIndex,
                              type: StaticResourceFileType.Secret,
                              name,
                              secret: file.multi_secret_data.secret_alias
                                 ? {
                                      alias: file.multi_secret_data.secret_alias,
                                      key: generateMultisecretKey(file.multi_secret_data.format),
                                   }
                                 : undefined,
                           } as SecretFile;
                        }

                        return {
                           _order: fileIndex,
                           type: StaticResourceFileType.Unknown,
                           name: file.file_name,
                        } as UnknownFile;
                     }) ?? [];

                  const resource: StaticResourceFiles = {
                     _order: staticResourceIndex,
                     id: staticResource.id,
                     type: StaticResourceType.Files,
                     files,
                     verification: {
                        enabled:
                           staticResource.verification?.disabled !== undefined
                              ? !staticResource.verification.disabled
                              : true,
                        checksum: staticResource.verification?.checksum ?? null,
                     },
                     accessPermissions:
                        staticResource.access_permissions ??
                        EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                     _ref: getHexRef(),
                  };

                  staticResources.push(resource);
               } else {
                  const resource: StaticResourceUnknown = {
                     _order: staticResourceIndex,
                     id: staticResource.id,
                     type: StaticResourceType.Unknown,
                     verification: {
                        enabled:
                           staticResource.verification?.disabled !== undefined
                              ? !staticResource.verification.disabled
                              : true,
                        checksum: staticResource.verification?.checksum ?? null,
                     },
                     accessPermissions:
                        staticResource.access_permissions ??
                        EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                     _ref: getHexRef(),
                  };

                  staticResources.push(resource);
               }
            }
         });

         (podAgentPayloadSpec?.volumes ?? []).forEach((volume, volimeIndex) => {
            if ((volume.virtual_disk_id_ref && volume.virtual_disk_id_ref === disk.id) || countDisks === 1) {
               const volumeLayers: VolumeLayer[] = [];
               const volumeStaticResources: VolumeStaticResource[] = [];

               (volume.generic?.layer_refs ?? []).forEach(id => {
                  const volumeLayer = layers.find(v => v.id === id);

                  if (volumeLayer) {
                     volumeLayers.push({
                        // eslint-disable-next-line no-underscore-dangle
                        _layerRef: volumeLayer?._ref,
                     });
                  }
               });

               (volume.static_resources ?? []).forEach((staticResource, staticResourceIndex) => {
                  const volumeStaticResource = staticResources.find(v => v.id === staticResource.resource_ref);

                  if (volumeStaticResource) {
                     volumeStaticResources.push({
                        _order: staticResourceIndex,
                        volumeRelativeMountPoint: staticResource.volume_relative_mount_point,
                        // eslint-disable-next-line no-underscore-dangle
                        _staticResourceRef: volumeStaticResource?._ref,
                     });
                  }
               });

               volumes.push({
                  _order: volimeIndex,
                  id: volume.id,
                  layers: volumeLayers,
                  staticResources: volumeStaticResources,
                  persistenceType: volume.persistence_type ?? EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
                  _ref: getHexRef(),
               });
            }
         });

         return {
            id: disk.id || DEFAULT_DISK_ID,
            type: diskType,
            size: disk.quota_policy?.capacity || null,
            bandwidth: {
               guarantee,
               limit: {
                  defaultSettings: diskType === DiskType.SSD ? limit === defaultSsdLimit : limit === defaultHddLimit,
                  default: diskType === DiskType.SSD ? defaultSsdLimit : defaultHddLimit,
                  custom: limit,
               },
            },
            volumes,
            layers,
            staticResources,
         };
      });
   }

   public static getDiskTypeFromSidecarStorageClass(storageClass: TSidecarVolume_EStorageClass): DiskType | null {
      return {
         [TSidecarVolume_EStorageClass.HDD]: DiskType.HDD,
         [TSidecarVolume_EStorageClass.SSD]: DiskType.SSD,
         [TSidecarVolume_EStorageClass.AUTO]: null,
      }[storageClass];
   }

   public static getDeployUnitType(spec: TDeployUnitSpec): DeployUnitType {
      if (spec.multi_cluster_replica_set) {
         return DeployUnitType.MultiCluster;
      }

      if (spec.replica_set) {
         return DeployUnitType.PerCluster;
      }

      return DefaultDeployUnitType;
   }

   public static getPodTemplateSpec(spec: TDeployUnitSpec): TPodTemplateSpec {
      const type = DeployUnitConverter.getDeployUnitType(spec);

      // в спеке "кривого" стейджа может не быть pod_template_spec (как и любых других полей)
      // это не должно ломать UI
      const empty = {} as TPodTemplateSpec;

      switch (type) {
         case DeployUnitType.MultiCluster:
            return spec.multi_cluster_replica_set?.replica_set?.pod_template_spec ?? empty;
         case DeployUnitType.PerCluster:
            return spec.replica_set?.replica_set_template?.pod_template_spec ?? empty;
      }

      return empty;
   }

   public static getSecretUsages(du: DeployUnit, alias: string | null): SecretUsage[] {
      const result: SecretUsage[] = [];

      (du.disks || []).forEach(disk => {
         (disk.staticResources || []).forEach(staticResource => {
            if (staticResource.type !== StaticResourceType.Files) {
               return;
            }

            for (const file of staticResource.files ?? []) {
               const path = getSecretUsagePath({
                  field: `staticResources[${staticResource.id}].files[${file.name}]`,
               });

               if (file.type === StaticResourceFileType.Secret) {
                  if (file.secret?.alias && (alias === null || file.secret.alias === alias)) {
                     result.push({ key: file.secret.key, path, type: SecretUsageType.StaticResource });
                  }
               }
            }
         });
      });

      // TVM
      if (du.tvm) {
         for (let i = 0; i < du.tvm.clients.length; i += 1) {
            const tvmClient = du.tvm.clients[i];
            if (tvmClient.secret && (alias === null || tvmClient.secret.alias === alias)) {
               // TODO change type of tvmClient.secret to SecretUsage
               result.push({
                  key: tvmClient.secret.key,
                  path: getSecretUsagePath({ field: `tvm.clients[${i}].secret` }),
                  type: SecretUsageType.Tvm,
               });
            }
         }
      }

      // logbroker
      if (du.logbrokerConfig.customTopicRequest) {
         const { secret } = du.logbrokerConfig.customTopicRequest;

         if (secret && (alias === null || secret.alias === alias)) {
            result.push({
               key: secret.key,
               path: getSecretUsagePath({ field: `logbrokerConfig.customTopicRequest.secret` }),
               type: SecretUsageType.LogBroker,
            });
         }
      }

      // From boxes
      for (const box of du.boxes) {
         result.push(...BoxConverter.getSecretUsages(box, alias));
      }

      return result;
   }

   public static getNodeFilters(filter: string): PodNodeFilters {
      const parts = new Set(filter.split(' AND '));

      return (Object.keys(podNodeFiltersMap) as (keyof PodNodeFilters)[]).reduce(
         (acc, k) => {
            acc[k] = parts.has(podNodeFiltersMap[k]);

            return acc;
         },
         {
            requireAvx2: false,
            requireAvx: false,
            requireBandwidth10G: false,
            requireIntel: false,
         } as PodNodeFilters,
      );
   }

   public static getPerLocationStrategy(
      rawStrategy: TDeployUnitSpec_TDeploySettings_EDeployStrategy,
   ): PerLocationStrategy {
      const strategyMap: Record<TDeployUnitSpec_TDeploySettings_EDeployStrategy, PerLocationStrategy> = {
         [TDeployUnitSpec_TDeploySettings_EDeployStrategy.SEQUENTIAL]: PerLocationStrategy.Sequential,
         [TDeployUnitSpec_TDeploySettings_EDeployStrategy.PARALLEL]: PerLocationStrategy.Parallel,
      };
      return strategyMap[rawStrategy] ?? defaultPerLocationStrategy;
   }

   public static isCustomPerLocationSettings(deploySettings: TDeployUnitSpec_TDeploySettings | null): boolean {
      if (!deploySettings) {
         return false;
      }
      if (Object.keys(deploySettings).length === 0) {
         return false;
      }
      const { deploy_strategy, cluster_sequence } = deploySettings;
      if (
         deploy_strategy &&
         DeployUnitConverter.getPerLocationStrategy(deploy_strategy) !== defaultPerLocationStrategy
      ) {
         // явно указанная стратегия отличается от дефолтной
         return true;
      }
      if (cluster_sequence) {
         // само наличие этого поля предполагает последовательную выкатку
         return true;
      }
      return false;
   }

   public static getEnvironmentSettings(
      env: TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment | undefined,
   ): EnvironmentSettings {
      if (!env) {
         return EnvironmentSettings.UNKNOWN;
      }
      return {
         [TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.TESTING]: EnvironmentSettings.TESTING,
         [TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.PRESTABLE]: EnvironmentSettings.PRESTABLE,
         [TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.STABLE]: EnvironmentSettings.STABLE,
         [TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.UNKNOWN]: EnvironmentSettings.UNKNOWN,
      }[env];
   }

   public anonymousMemoryLimit: number | null;

   public antiaffinity: AntiaffinityRecord | null = null;

   public boxes: Box[];

   public cpu: number | null;

   public defaultLayerSourceFileStoragePolicy: LayerSourceFileStoragePolicy;

   public disks: DeployUnitDisk[];

   public disruptionBudget: number | null = null;

   public endpointSets: DeployUnitEndpointSet[];

   public id: string;

   public locations: DeployUnitLocationMap;

   public maxTolerableDowntimePods: number | null = null;

   public maxTolerableDowntimeSeconds: number | null = null;

   public networkDefaults: DeployUnitNetwork;

   public networkBandwidth: DeployUnitNetworkBandwidth;

   public nodeFilters: PodNodeFilters;

   public ram: number | null;

   public resources: Resource[];

   public soxService: boolean;

   public tempDiskIsolation: boolean;

   public tvm: DeployUnitTvm;

   public type: DeployUnitType;

   public yasm: DuYasm;

   public sidecars: Record<SidecarName, Sidecar>;

   public perLocationSettings: PerLocationSettings;

   public revision: number | null;

   public patchersRevision: PatchersRevision;

   public podAgentSidecarDiskType: DiskType | null;

   public logbrokerSidecarDiskType: DiskType | null;

   public logbrokerConfig: LogbrokerConfig;

   public collectPortometricsFromSidecars: boolean;

   public nodeSegmentId: string | null;

   public infraComponents: InfraComponents;

   public transmitSystemLogs: ETransmitSystemLogs;

   public settings: DeployUnitSettings;

   private readonly spec: TDeployUnitSpec;

   constructor(
      private raw: DeployUnitRawRecord,
      labels: TStage['labels'] = {},
      stageDynamicResources: TStageSpec['dynamic_resources'] = {},
      deployUnitsSettings: TStageSpec['deploy_unit_settings'] = {},
   ) {
      // TODO: неконсистентно (то spec, то this.spec и т.д.)
      const { id, spec } = raw;
      this.id = id;
      this.spec = spec;

      this.soxService = spec.sox_service;

      this.type = DeployUnitConverter.getDeployUnitType(this.spec);

      const podTemplateSpec = DeployUnitConverter.getPodTemplateSpec(this.spec)!;

      switch (this.type) {
         case DeployUnitType.MultiCluster: {
            const deploymentStrategy = this.spec.multi_cluster_replica_set?.replica_set?.deployment_strategy;

            this.disruptionBudget = deploymentStrategy?.max_unavailable ?? null;
            this.maxTolerableDowntimeSeconds = deploymentStrategy?.max_tolerable_downtime_seconds ?? null;
            this.maxTolerableDowntimePods = deploymentStrategy?.max_tolerable_downtime_pods ?? null;

            this.locations = this.getMultiClusterLocations();

            break;
         }

         case DeployUnitType.PerCluster: {
            this.antiaffinity = this.getAntiaffinity(
               this.spec.replica_set?.replica_set_template?.constraints?.antiaffinity_constraints,
            );

            this.locations = this.getPerClusterLocations();

            break;
         }
      }

      this.nodeFilters = DeployUnitConverter.getNodeFilters(podTemplateSpec.spec?.node_filter ?? '');

      // resources
      const dynamicResources = this.getDynamicResources(stageDynamicResources);
      this.resources = this.getResources(podTemplateSpec, dynamicResources);

      this.endpointSets =
         spec.endpoint_sets?.map((v, i) => ({
            _order: i,
            id: v.id ?? null,
            port: v.port ?? null,
            liveness_limit_ratio: v.liveness_limit_ratio ?? null,
         })) ?? [];

      this.networkDefaults = this.getNetwork(spec?.network_defaults, podTemplateSpec);
      this.networkBandwidth = {
         limit: podTemplateSpec.spec?.resource_requests?.network_bandwidth_limit ?? null,
         guarantee: podTemplateSpec.spec?.resource_requests?.network_bandwidth_guarantee ?? null,
      };

      this.tvm = this.getTvm(spec.tvm_config, podTemplateSpec);

      this.defaultLayerSourceFileStoragePolicy =
         layerSourceFileStoragePolicies[
            podTemplateSpec.spec?.pod_agent_payload?.spec?.resources?.default_layer_source_file_storage_policy ??
               layerSourceFileStoragePolicies.none
         ];

      this.disks = DeployUnitConverter.getDisks(podTemplateSpec);

      // resource quotas
      this.cpu = podTemplateSpec.spec?.resource_requests?.vcpu_guarantee ?? null;
      this.ram = podTemplateSpec.spec?.resource_requests?.memory_guarantee ?? null;
      this.anonymousMemoryLimit = podTemplateSpec.spec?.resource_requests?.anonymous_memory_limit ?? null;

      this.yasm = DeployUnitConverter.getYasm(podTemplateSpec);

      // children
      this.boxes = this.getBoxes(podTemplateSpec, dynamicResources, this.disks);
      this.tempDiskIsolation = !podTemplateSpec.labels?.['disable-disk-isolation-spi-15289'];

      // sidecars
      this.sidecars = {} as Record<SidecarName, Sidecar>;
      for (const sidecarName of SidecarsForUpdating) {
         this.sidecars[sidecarName] = this.getSidecars(sidecarName, id, labels);
      }

      this.perLocationSettings = this.getPerLocationSettings(this.spec.deploy_settings);
      this.revision = this.spec.revision ?? null;

      this.patchersRevision = {
         label: labels.du_patchers_target_revision?.[id] ?? null,
         value: spec.patchers_revision ? Number(spec.patchers_revision) : null,
      };

      const podAgentSidecarStorageClass = podTemplateSpec.spec?.pod_agent_payload?.meta?.sidecar_volume?.storage_class;
      this.podAgentSidecarDiskType = podAgentSidecarStorageClass
         ? DeployUnitConverter.getDiskTypeFromSidecarStorageClass(podAgentSidecarStorageClass)
         : null;

      const logbrokerSidecarStorageClass = raw.spec.logbroker_config?.sidecar_volume?.storage_class;
      this.logbrokerSidecarDiskType = logbrokerSidecarStorageClass
         ? DeployUnitConverter.getDiskTypeFromSidecarStorageClass(logbrokerSidecarStorageClass)
         : null;

      this.logbrokerConfig = DeployUnitConverter.getLogbrokerConfig(spec);

      this.collectPortometricsFromSidecars = Boolean(spec.collect_portometrics_from_sidecars);

      this.nodeSegmentId = podTemplateSpec.labels?.['yd.node_segment_id'] ?? null;

      this.infraComponents = { allowAutomaticUpdates: Boolean(spec.infra_components?.allow_automatic_updates) };

      this.transmitSystemLogs =
         podTemplateSpec.spec?.pod_agent_payload?.spec?.transmit_system_logs_policy?.transmit_system_logs ??
         ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE;

      this.settings = this.getSettings(deployUnitsSettings);
   }

   public toJSON(): DeployUnit {
      return {
         anonymousMemoryLimit: this.anonymousMemoryLimit,
         antiaffinity: this.antiaffinity,
         boxes: this.boxes,
         cpu: this.cpu,
         defaultLayerSourceFileStoragePolicy: this.defaultLayerSourceFileStoragePolicy,
         disks: this.disks,
         disruptionBudget: this.disruptionBudget,
         endpointSets: this.endpointSets,
         id: this.id,
         locations: this.locations,
         maxTolerableDowntimePods: this.maxTolerableDowntimePods,
         maxTolerableDowntimeSeconds: this.maxTolerableDowntimeSeconds,
         networkDefaults: this.networkDefaults,
         networkBandwidth: this.networkBandwidth,
         nodeFilters: this.nodeFilters,
         ram: this.ram,
         resources: this.resources,
         soxService: this.soxService,
         tempDiskIsolation: this.tempDiskIsolation,
         tvm: this.tvm,
         type: this.type,
         yasm: this.yasm,
         sidecars: this.sidecars,
         perLocationSettings: this.perLocationSettings,
         revision: this.revision,
         patchersRevision: this.patchersRevision,
         podAgentSidecarDiskType: this.podAgentSidecarDiskType,
         logbrokerSidecarDiskType: this.logbrokerSidecarDiskType,
         logbrokerConfig: this.logbrokerConfig,
         collectPortometricsFromSidecars: this.collectPortometricsFromSidecars,
         nodeSegmentId: this.nodeSegmentId,
         infraComponents: this.infraComponents,
         transmitSystemLogs: this.transmitSystemLogs,
         settings: this.settings,
      };
   }

   private getAntiaffinity(constraints: TAntiaffinityConstraint[] | undefined = []): AntiaffinityRecord {
      const perNode = constraints.find(c => c.key === AntiaffinityType.Node);
      const perRack = constraints.find(c => c.key === AntiaffinityType.Rack);

      return {
         perRack: perRack?.max_pods ?? null,
         perNode: perNode?.max_pods ?? null,
      };
   }

   private getBoxes(
      podTemplateSpec: TPodTemplateSpec,
      dynamicResources: TStageSpec['dynamic_resources'],
      disks: DeployUnitDisk[],
   ): Box[] {
      const agentSpec = podTemplateSpec.spec?.pod_agent_payload?.spec;

      return (agentSpec?.boxes ?? []).map(box =>
         BoxConverter.fromApi(box, podTemplateSpec, this.raw.spec, dynamicResources, disks),
      );
   }

   private getDynamicResources(stageDynamicResources: TStageSpec['dynamic_resources']) {
      const dynamicResources: TStageSpec['dynamic_resources'] = {};

      if (stageDynamicResources) {
         for (const key in stageDynamicResources) {
            if (stageDynamicResources[key].deploy_unit_ref === this.id) {
               dynamicResources[key] = stageDynamicResources[key];
            }
         }
      }

      return dynamicResources;
   }

   private getMultiClusterLocations(): DeployUnitLocationMap {
      if (!this.spec.multi_cluster_replica_set?.replica_set?.clusters) {
         return {};
      }

      return this.spec.multi_cluster_replica_set?.replica_set?.clusters?.reduce((acc, { cluster, spec }) => {
         acc[cluster] = {
            enabled: true,
            antiaffinity: this.getAntiaffinity(spec?.constraints?.antiaffinity_constraints),
            podCount: spec?.replica_count ?? 0,

            disruptionBudget: null,
            maxTolerableDowntimePods: null,
            maxTolerableDowntimeSeconds: null,
         };

         return acc;
      }, {} as DeployUnitLocationMap);
   }

   private getNetwork(
      networkDefaults: TNetworkDefaults | undefined,
      podTemplateSpec: TPodTemplateSpec,
   ): DeployUnitNetwork {
      const settings = podTemplateSpec.spec?.ip6_address_requests;
      const networkId = networkDefaults?.network_id ?? '';
      let customSettings = true;

      if (settings === undefined) {
         customSettings = false;
      }

      // проверяем на дефолтные настройки, которые проставлял UI в старой/новой форме (DEPLOY-3850)
      // сейчас backend их сам подставляет с актуальной networkId, если поля ip6_address_requests нет
      if (settings?.length === 2) {
         const backbone = settings.find(v =>
            isEqual(v, {
               'enable_dns': true,
               'network_id': networkId,
               'vlan_id': 'backbone',
            } as TPodSpec_TIP6AddressRequest),
         );

         const fastbone = settings.find(v =>
            isEqual(v, {
               'enable_dns': true,
               'network_id': networkId,
               'vlan_id': 'fastbone',
            } as TPodSpec_TIP6AddressRequest),
         );

         if (backbone && fastbone) {
            customSettings = false;
         }
      }

      return {
         networkId,
         customSettings,
         virtualServiceIds: networkDefaults?.virtual_service_ids ?? [],
         ipv4AddressPoolId: networkDefaults?.ip4_address_pool_id ?? null,
      };
   }

   private getPerClusterLocations(): DeployUnitLocationMap {
      if (!this.spec.replica_set?.per_cluster_settings) {
         return {};
      }

      return Object.entries(this.spec.replica_set?.per_cluster_settings).reduce((acc, [cluster, spec]) => {
         const deploymentStrategy = spec.deployment_strategy;
         acc[cluster] = {
            enabled: true,
            antiaffinity: null,
            podCount: spec.pod_count ?? 0,

            disruptionBudget: deploymentStrategy?.max_unavailable ?? null,
            maxTolerableDowntimePods: deploymentStrategy?.max_tolerable_downtime_pods ?? null,
            maxTolerableDowntimeSeconds: deploymentStrategy?.max_tolerable_downtime_seconds ?? null,
         };

         return acc;
      }, {} as DeployUnitLocationMap);
   }

   private getPerLocationSettings(deploySettings: TDeployUnitSpec_TDeploySettings | undefined): PerLocationSettings {
      let strategy = PerLocationStrategy.Parallel;
      const locationOrder: string[] = [];
      const needApproval: Set<string> = new Set();

      if (this.type === DeployUnitType.MultiCluster) {
         return {
            isCustom: false,
            strategy: PerLocationStrategy.Parallel,
            locationOrder: [],
            needApproval: new Set(),
         };
      }

      if (deploySettings?.cluster_sequence) {
         strategy = PerLocationStrategy.Sequential;
         for (const locationItem of deploySettings?.cluster_sequence) {
            const location = locationItem.yp_cluster;
            locationOrder.push(location);
            if (locationItem.need_approval) {
               needApproval.add(location);
            }
         }
      }

      // порядок if важен, так как cluster_sequence без deploy_strategy определяет Sequential
      if (deploySettings?.deploy_strategy) {
         strategy = DeployUnitConverter.getPerLocationStrategy(deploySettings.deploy_strategy);
      }

      return {
         isCustom: DeployUnitConverter.isCustomPerLocationSettings(deploySettings ?? null),
         strategy,
         locationOrder,
         needApproval,
      };
   }

   private getResources(
      podTemplateSpec: TPodTemplateSpec,
      dynamicResources: TStageSpec['dynamic_resources'],
   ): Resource[] {
      const result: Resource[] = [];

      const resources = podTemplateSpec.spec?.pod_agent_payload?.spec?.resources;

      // layers
      for (const layer of resources?.layers ?? []) {
         result.push({ id: layer.id, type: ResourceType.Layer });
      }

      // static resources
      for (const staticResource of resources?.static_resources ?? []) {
         result.push({ id: staticResource.id, type: ResourceType.StaticResource });
      }

      // dynamic resources
      for (const key in dynamicResources) {
         if (dynamicResources.hasOwnProperty(key)) {
            const deployGroup = dynamicResources[key].dynamic_resource?.deploy_groups?.[0];

            if (deployGroup !== undefined) {
               result.push({
                  id: key,
                  type: ResourceType.DynamicResource,
                  deployGroupMark: deployGroup.mark ?? '',
               });
            }
         }
      }

      return result;
   }

   private getSettings(allSettings: TStageSpec['deploy_unit_settings']): DeployUnitSettings {
      const settings = allSettings[this.id] ?? {};

      return {
         alerting: {
            state: settings.alerting?.state === TStageSpec_TDeployUnitSettings_TAlerting_EState.ENABLED,
            notificationChannel: settings.alerting?.notification_channels?.ERROR ?? null,
         },
         environment: DeployUnitConverter.getEnvironmentSettings(settings.environment),
      };
   }

   private getSidecars(type: SidecarsForUpdatingUnion, deployUnitId: string, labels: TStage['labels']) {
      const sidecar = sidecarsUpdateConfig[type];
      const labelLayer = labels?.du_sidecar_target_revision?.[deployUnitId]?.[sidecar.labelKey];
      const labelRevision = labelLayer ? Number(labelLayer) : null;
      const specPath = this.spec?.[sidecar.specPath as SidecarsSpecPathsType];

      return {
         resourceRevision: specPath?.revision ?? null,
         overrideLabels: Object.keys(specPath?.override ?? {}).map(key => ({
            key,
            value: specPath?.override[key] ?? '',
         })),
         labelRevision,
      };
   }

   private getTvm(specTvm: TTvmConfig | undefined = undefined, podTemplateSpec: TPodTemplateSpec): DeployUnitTvm {
      const defaultTvm = getDefaultTvm();
      const defaultTvmClient = defaultTvm.clients[0];
      const defaultDestination = defaultTvmClient.destinations![0];

      const clients: TvmClient[] =
         specTvm?.clients?.map((client, i) => ({
            _order: i,
            destinations:
               client.destinations?.map((destination, _order) => ({
                  _order,
                  // abc: destination?.abc_service_id ?? '',
                  app: destination?.app_id ?? defaultDestination.app,
                  alias: destination?.alias ?? defaultDestination.alias,
               })) ?? defaultTvmClient.destinations,
            mode:
               isEmpty(client.destinations) && isEmpty(client.secret_selector)
                  ? TvmClientMode.CheckOnly
                  : TvmClientMode.GetCheck,
            secret: client.secret_selector ? getLinkToSecret(client.secret_selector) : null,
            source: {
               // abc: client.source?.abc_service_id ?? '',
               app: client.source?.app_id ?? defaultTvmClient.source.app,
               alias: client.source?.alias ?? defaultTvmClient.source.alias,
            },
         })) ?? defaultTvm.clients;

      const tvmStorageClass = specTvm?.sidecar_volume?.storage_class;

      const tvm: DeployUnitTvm = {
         blackbox: specTvm?.blackbox_environment ?? defaultTvm.blackbox,
         clientPort: specTvm?.client_port ?? defaultTvm.clientPort,
         clients,
         enabled: specTvm?.mode === TTvmConfig_EMode.ENABLED,
         cpuLimit: specTvm?.cpu_limit ?? null,
         memoryLimit: specTvm?.memory_limit_mb ? specTvm.memory_limit_mb * BYTES.MB : null,
         diskType: tvmStorageClass ? DeployUnitConverter.getDiskTypeFromSidecarStorageClass(tvmStorageClass) : null,
      };

      return tvm;
   }

   private static getLogbrokerConfig(spec: TDeployUnitSpec): LogbrokerConfig {
      const logbrokerConfig = spec.logbroker_config;
      const customTopicRequest = logbrokerConfig?.custom_topic_request;
      const destroyPolicy = logbrokerConfig?.destroy_policy;
      const podAdditionalResourcesRequest = logbrokerConfig?.pod_additional_resources_request;

      return {
         customTopicRequest: {
            topicName: customTopicRequest?.topic_name ?? null,
            tvmClientId: customTopicRequest?.tvm_client_id ?? null,
            secret: customTopicRequest?.secret_selector ? getLinkToSecret(customTopicRequest?.secret_selector) : null,
         },
         destroyPolicy: {
            maxTries: destroyPolicy?.max_tries ?? null,
            restartPeriodMs: destroyPolicy?.restart_period_ms ?? null,
         },
         podAdditionalResourcesRequest: {
            setCpuToZero:
               podAdditionalResourcesRequest?.vcpu_guarantee === 0 && podAdditionalResourcesRequest?.vcpu_limit === 0,
         },
      };
   }

   private static getYasm(podTemplateSpec: TPodTemplateSpec): DuYasm {
      const yasmRecord = podTemplateSpec.spec?.host_infra?.monitoring;

      const { itype, tags } = convertYasmLabels(yasmRecord?.labels);
      const yasmTags: YasmTags = { itype, tags };

      const podAgent = yasmRecord?.pod_agent;

      return {
         yasmTags,
         podAgent: podAgent
            ? {
                 addPodAgentUserSignals: podAgent.add_pod_agent_user_signals ?? false,
              }
            : {
                 addPodAgentUserSignals: false,
              },
      };
   }
}
