import { DeployUnitType } from './DeployUnit';

import { DeployUnitFormParams, deployUnitSchema, deployUnitToFormParams } from './DeployUnitFormParams';

describe('models/ui|DeployUnitFormParams validation', () => {
   const getValidParams = (): DeployUnitFormParams => ({
      ...deployUnitToFormParams(),
      networkDefaults: {
         networkId: '__TEST_NETWORK__',
         customSettings: false,
         virtualServiceIds: [],
         ipv4AddressPoolId: null,
      },
   });

   it('should be valid for default', () => {
      const params = getValidParams();

      expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
   });

   describe('DeployUnitType.MultiCluster', () => {
      describe('check disruption budget', () => {
         it('should be valid', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.MultiCluster,
               disruptionBudget: 1,
            };

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });

         it('should be invalid', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.MultiCluster,
               disruptionBudget: null,
            };

            expect(() => deployUnitSchema.validateSync(params)).toThrow('Disruption budget is a required field');
         });
      });

      describe('check maxTolerableDowntimePods relative to disruptionBudget', () => {
         it('should be valid if less', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.MultiCluster,
               disruptionBudget: 4,
               maxTolerableDowntimePods: 2,
            };

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });

         it('should be invalid if greater', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.MultiCluster,
               disruptionBudget: 1,
               maxTolerableDowntimePods: 2,
            };

            expect(() => deployUnitSchema.validateSync(params)).toThrowError(
               'Maintenance budget must be less than or equal to Disruption budget (1)',
            );
         });
      });

      describe('check location', () => {
         it('should not check disabled location', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.MultiCluster,
               disruptionBudget: 1,
               locations: {
                  sas: {
                     enabled: false,
                     podCount: undefined as any,
                     maxTolerableDowntimeSeconds: null,
                     disruptionBudget: null,
                     maxTolerableDowntimePods: null,
                     antiaffinity: null,
                  },
               },
            };

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();

            params.locations.sas.enabled = true;

            expect(() => deployUnitSchema.validateSync(params)).toThrow('Pods quantity is a required field');
         });

         it('should check enabled location', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.MultiCluster,
               disruptionBudget: 1,
               locations: {
                  sas: {
                     enabled: true,
                     podCount: undefined as any,
                     maxTolerableDowntimeSeconds: null,
                     disruptionBudget: null,
                     maxTolerableDowntimePods: null,
                     antiaffinity: null,
                  },
               },
            };

            expect(() => deployUnitSchema.validateSync(params)).toThrow('Pods quantity is a required field');

            params.locations.sas.podCount = 1;

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });
      });
   });

   describe('DeployUnitType.PerCluster', () => {
      describe('check disruption budget', () => {
         it('should be valid', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.PerCluster,
               disruptionBudget: null,
            };

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();

            params.disruptionBudget = 1;

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });
      });

      describe('check maxTolerableDowntimePods relative to disruptionBudget', () => {
         it('should be valid if less', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.PerCluster,
               locations: {
                  sas: {
                     enabled: true,
                     podCount: 1,

                     antiaffinity: null,
                     disruptionBudget: 4,
                     maxTolerableDowntimePods: 2,
                     maxTolerableDowntimeSeconds: null,
                  },
               },
            };

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });

         it('should be invalid if greater', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.PerCluster,
               locations: {
                  sas: {
                     enabled: true,
                     podCount: 1,

                     antiaffinity: null,
                     disruptionBudget: 3,
                     maxTolerableDowntimePods: 5,
                     maxTolerableDowntimeSeconds: null,
                  },
               },
            };

            expect(() => deployUnitSchema.validateSync(params)).toThrowError(
               'Maintenance budget must be less than or equal to Disruption budget (3)',
            );
         });
      });

      describe('check location', () => {
         it('should not check disabled location', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.PerCluster,
               disruptionBudget: null,
               locations: {
                  sas: {
                     enabled: false,
                     podCount: undefined as any,
                     maxTolerableDowntimeSeconds: null,
                     disruptionBudget: null,
                     maxTolerableDowntimePods: null,
                     antiaffinity: null,
                  },
               },
            };

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();

            params.locations.sas.enabled = true;

            expect(() => deployUnitSchema.validateSync(params)).toThrow('Pods quantity is a required field');
         });

         it('should check enabled location', () => {
            const params: DeployUnitFormParams = {
               ...getValidParams(),
               type: DeployUnitType.PerCluster,
               disruptionBudget: null,
               locations: {
                  sas: {
                     enabled: true,
                     podCount: undefined as any,
                     maxTolerableDowntimeSeconds: null,
                     disruptionBudget: null,
                     maxTolerableDowntimePods: null,
                     antiaffinity: null,
                  },
               },
            };

            expect(() => deployUnitSchema.validateSync(params)).toThrow('Pods quantity is a required field');

            params.locations.sas.podCount = 1;

            expect(() => deployUnitSchema.validateSync(params)).toThrow('Disruption budget is a required field');

            params.locations.sas.disruptionBudget = 1;

            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });
      });
   });

   describe('DeployUnitType.LogbrokerConfig', () => {
      describe('destroy policy', () => {
         it('should check restart period', () => {
            const params = {
               ...getValidParams(),
               logbrokerConfig: {
                  destroyPolicy: { restartPeriodMs: 9000 },
               },
            };
            expect(() => deployUnitSchema.validateSync(params)).toThrow(
               'Restart period in ms must be greater than or equal to 10000',
            );
            params.logbrokerConfig.destroyPolicy.restartPeriodMs = 10000;
            expect(() => deployUnitSchema.validateSync(params)).not.toThrow();
         });
      });
   });
});
