/* eslint-disable no-template-curly-in-string */
import { BYTES, formatNumber, isEmpty, record, setAsArray } from '@yandex-infracloud-ui/libs';
import { array, ArraySchema, boolean, number, object, ref, Schema, string, StringSchema } from 'yup';

import { ValidationContext } from '../../../../components/huge-form';
// noinspection ES6PreferShortImport
import { getLinkToSecretValidationSchema } from '../../../../modules/secrets/helpers';
import { EResourceAccessPermissions, ETransmitSystemLogs, EVolumePersistenceType } from '../../../../proto-typings';
import { getPaths } from '../../../../utils';
import { LinkToSecret } from '../../secrets';

import { SidecarName, SidecarsForUpdating, sidecarsUpdateConfig } from '../../Sidecars';

import { Box } from '../Box';

import {
   checksumSchema,
   deployUnitIdSchema,
   hasOnlyUniqueValues,
   IntegerNullableSchema,
   IntegerPositiveNullableSchema,
   layerUrlSchema,
   orderSchema,
   portSchema,
   requiredIfEnabled,
   resourceIdValidationSchema,
   schemeIfEnabled,
   staticResourceUrlSchema,
   StringNullableRequiredSchema,
   StringNullableSchema,
   StringRequiredSchema,
   tvmPortSchema,
   unixPathSchema,
} from '../validation';

import { yasmTagsValidationSchema } from '../yasm';

import {
   activePatcherRevisions,
   AlertingSettings,
   AntiaffinityRecord,
   CustomTopicRequest,
   DeployUnit,
   DeployUnitDisk,
   DeployUnitEndpointSet,
   DeployUnitLocation,
   DeployUnitLocationMap,
   DeployUnitNetwork,
   DeployUnitNetworkBandwidth,
   DeployUnitSettings,
   DeployUnitTvm,
   DeployUnitType,
   DiskBandwidth,
   DiskLayer,
   DiskLayerType,
   DiskStaticResource,
   DiskType,
   DiskVolume,
   DuYasm,
   EnvironmentSettings,
   getEmptyDeployUnit,
   InfraComponents,
   LayerSourceFileStoragePolicy,
   LogbrokerConfig,
   LogbrokerDestroyPolicy,
   LogbrokerPodAdditionalResourcesRequest,
   PatchersRevision,
   PerLocationSettings,
   PerLocationStrategy,
   StaticResourceFile,
   StaticResourceFileType,
   StaticResourceType,
   TvmBlackbox,
   TvmClient,
   TvmClientItemRecord,
   TvmClientMode,
   VolumeLayer,
   VolumeStaticResource,
   YasmPodAgent,
} from './DeployUnit';

export type DeployUnitFormParams = Omit<
   DeployUnit,
   'boxes' | 'resources' | 'soxService' | 'revision' | 'locationApproves' | 'locationsWaitingForApprove' | 'overrides'
>;

export const duPaths = getPaths<DeployUnitFormParams>();

// region Disks

// noinspection PointlessArithmeticExpressionJS
export const diskTypeSchema = string()
   .label('Disk type')
   .oneOf([DiskType.SSD, DiskType.HDD])
   .required() as Schema<DiskType>;

export const diskTypeNullableSchema = string()
   .nullable()
   .label('Disk type')
   .oneOf([DiskType.SSD, DiskType.HDD, null]) as Schema<DiskType | null>;

export const diskSizeSchema = IntegerPositiveNullableSchema('Disk size')
   .min(1, '${path} must be greater than 0 bytes')
   .required();

export const bandwidthSchema = (label: string) =>
   IntegerPositiveNullableSchema(label)
      .min(BYTES.MB, '${path} must be greater than or equal to 1 MB/s')
      .max(2 * BYTES.GB, '${path} must be less than or equal to 2 GB/s');

export const diskBandwidthGuaranteeSchema = bandwidthSchema('Disk bandwidth guarantee');

export const diskBandwidthDefaultLimitSchema = (type: DiskType, guarantee: number) =>
   number().when('defaultSettings', {
      is: true,
      then: bandwidthSchema('Disk bandwidth default limit'),
      otherwise: number().nullable().notRequired(),
   });

export const diskBandwidthCustomLimitSchema = (type: DiskType, guarantee: number) =>
   number().when('defaultSettings', {
      is: false,
      then: bandwidthSchema('Disk bandwidth custom limit'),
      otherwise: number().nullable().notRequired(),
   });

export const diskBandwidthLimitSchema = (type: DiskType, guarantee: number) =>
   object({
      defaultSettings: boolean(),
      default: diskBandwidthDefaultLimitSchema(type, guarantee),
      custom: diskBandwidthCustomLimitSchema(type, guarantee),
   });

export const diskBandwidthSchema = (type: DiskType) =>
   object({
      guarantee: diskBandwidthGuaranteeSchema,
      limit: object().when('guarantee', (guarantee: number | null) =>
         guarantee !== null ? diskBandwidthLimitSchema(type, guarantee) : object(),
      ),
   });

type ResourceField = 'layers' | 'staticResources' | 'volumes';

const diskResourceIdValidationSchema = (resourceField: ResourceField) =>
   resourceIdValidationSchema.test('isValid', '${path}s must be valid', function IsValid(value: string) {
      if (!value || isEmpty(value)) {
         return true;
      }

      const context: ValidationContext | undefined = this.options.context as any;

      if (!context) {
         return true;
      }

      const resourceNames = {
         'layers': 'Layer',
         'staticResources': 'Static resource',
         'volumes': 'Volume',
      };

      const form = (context.form as unknown) as DeployUnit;
      const { disks } = form;

      const idList = disks
         .flatMap(disk => {
            if (resourceField === 'layers') {
               return disk.layers.map(v => v.id);
            }
            if (resourceField === 'staticResources') {
               return disk.staticResources.map(v => v.id);
            }
            return disk.volumes.map(v => v.id);
         })
         .filter(v => v === value);

      if (!hasOnlyUniqueValues(idList)) {
         return this.createError({
            message: `${resourceNames[resourceField]} should have unique ID (${value})`,
         });
      }

      return true;
   });

const layerSourceFileStoragePolicySchema = string()
   .oneOf([LayerSourceFileStoragePolicy.None, LayerSourceFileStoragePolicy.Keep, LayerSourceFileStoragePolicy.Remove])
   .required() as Schema<LayerSourceFileStoragePolicy>;

export const diskLayersSchema = array().of(
   object<DiskLayer>({
      _order: orderSchema,
      id: string().when('removed', {
         is: true,
         then: string().notRequired(),
         otherwise: diskResourceIdValidationSchema('layers').label('Layer ID'),
      }),
      type: string()
         .label('Layer type')
         .oneOf([DiskLayerType.Url, DiskLayerType.Unknown])
         .required() as Schema<DiskLayerType>,
      url: string().when(['removed', 'type'], {
         is: (removed, type) => !removed && type === DiskLayerType.Url,
         then: layerUrlSchema,
         otherwise: string().notRequired(),
      }),
      checksum: string().when(['removed', 'type'], {
         is: (removed, type) => !removed && type === DiskLayerType.Url,
         then: checksumSchema,
         otherwise: string().notRequired(),
      }),
      removed: boolean().notRequired(),
      _ref: StringRequiredSchema('Layer ref'),
      layerSourceFileStoragePolicy: layerSourceFileStoragePolicySchema.label('Layer fource file storage policy'),
   }).test('isUsed', 'Is layer used', function IsUsed(value: DiskLayer) {
      if (!value.removed) {
         return true;
      }

      const context: ValidationContext | undefined = this.options.context as any;

      if (!context) {
         return true;
      }

      const path = this.path.split('.');

      const form = (context.form as unknown) as DeployUnit;
      const disk = form.disks?.find((_, i) => `disks[${i}]` === path[0]);
      // eslint-disable-next-line no-underscore-dangle
      const diskLayer = disk?.volumes.flatMap(v => v.layers).find(v => v._layerRef === value._ref);

      const boxForms = (context.childrenForms as unknown) as Box[];
      // eslint-disable-next-line no-underscore-dangle
      const boxLayer = boxForms.flatMap(v => v.layers).find(v => v._layerRef === value._ref);

      if (diskLayer || boxLayer) {
         return this.createError({
            path: `${this.path}.removed`,
            message: `Layer '${value.id}' is used and cannot be removed.`,
         });
      }

      return true;
   }),
);

// region static resources

const fileNameValidationSchema = StringNullableSchema('File name')
   .min(1)
   .required()
   .test('uniqueName', 'Files should have unique names', function UniqueNames(name: string) {
      const context: ValidationContext | undefined = this.options.context as any;
      const path = this.path.split('.');

      if (!context || path.length < 2) {
         return true;
      }

      const form = (context.form as unknown) as DeployUnit;
      const disk = form.disks?.find((_, i) => `disks[${i}]` === path[0]);
      const staticResource = disk?.staticResources?.find((_, i) => `staticResources[${i}]` === path[1]);

      if (!staticResource) {
         return true;
      }

      if (staticResource?.type === StaticResourceType.Files) {
         if (!isEmpty(staticResource.files)) {
            const fileNames = staticResource.files?.filter(file => file.name === name).map(file => file.name) ?? [];

            if (!hasOnlyUniqueValues(fileNames)) {
               return this.createError({
                  message: `Files should have unique names (${name})`,
               });
            }
         }
      }

      return true;
   });

export const fileRawValidationSchema = StringNullableSchema('Raw file').when('type', {
   is: StaticResourceFileType.Raw,
   then: StringNullableSchema('Raw file')
      .min(1)
      .max(5 * BYTES.KB)
      .required(),
   otherwise: string().notRequired(),
});

export const fileSecretValidationSchema = object<LinkToSecret>().label('Secret').nullable().when('type', {
   is: StaticResourceFileType.Secret,
   then: getLinkToSecretValidationSchema().required(),
   otherwise: object().notRequired(),
});

const staticResourceFileValidationSchema = object<StaticResourceFile>({
   _order: orderSchema,
   name: fileNameValidationSchema,

   type: (StringRequiredSchema('File type')
      .oneOf([StaticResourceFileType.Raw, StaticResourceFileType.Secret, StaticResourceFileType.Unknown])
      .required() as Schema<StaticResourceFileType>) as any,

   raw: fileRawValidationSchema,
   secret: fileSecretValidationSchema,
});

const staticResourceFilesValidationSchema = array().of(staticResourceFileValidationSchema);
// TODO: #DEPLOY-5283
// нужно сделать анонсы и обойти стейджи с такими ресурсами
// раскомментировать, когда таких стейджей не останется
// .test('validFiles', 'Files are invalid', function ValidFiles(value: StaticResourceFile[]) {
//    const hasSecretFile = value.some(v => v.type === StaticResourceFileType.Secret);
//    const hasRawFile = value.some(v => v.type === StaticResourceFileType.Raw);

//    if (hasSecretFile) {
//       if (hasRawFile) {
//          return this.createError({
//             path: `${this.path}.files`,
//             message: `You should use separate static resources for 'Raw' and 'Secret' files`,
//          });
//       }
//    }

//    return true;
// });

const staticResourceAccessPermissionsValidationSchema = StringRequiredSchema('Static Resource access permissions')
   .oneOf([
      EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
      EResourceAccessPermissions.EResourceAccessPermissions_600,
      EResourceAccessPermissions.EResourceAccessPermissions_660,
   ])
   .required() as Schema<EResourceAccessPermissions>;

export const diskStaticResourcesSchema = array().of(
   object<DiskStaticResource>({
      _order: orderSchema,
      id: string().when('removed', {
         is: true,
         then: string().notRequired(),
         otherwise: diskResourceIdValidationSchema('staticResources').label('Static resource ID'),
      }),
      type: (StringRequiredSchema('Static Resource type').oneOf([
         StaticResourceType.Url,
         StaticResourceType.Files,
         StaticResourceType.Unknown,
      ]) as Schema<StaticResourceType>) as any,
      url: string().when(['removed', 'type'], {
         is: (removed, type) => !removed && type === StaticResourceType.Url,
         then: staticResourceUrlSchema,
         otherwise: string().notRequired(),
      }),
      files: array()
         .of(object<StaticResourceFile>())
         .when(['removed', 'type'], {
            is: (removed, type) => !removed && type === StaticResourceType.Files,
            then: staticResourceFilesValidationSchema,
            otherwise: array().notRequired(),
         }),
      verification: object({
         enabled: boolean(),
         checksum: string().when(['removed', 'type'], {
            is: (removed, type) => !removed && type === StaticResourceType.Url,
            then: checksumSchema,
            otherwise: string().nullable().notRequired(),
         }),
      }),
      accessPermissions: staticResourceAccessPermissionsValidationSchema,
      removed: boolean().notRequired(),
      _ref: StringRequiredSchema('Static resource ref'),
   })
      .test('isUsed', 'Is static resource used', function IsUsed(value: DiskStaticResource) {
         if (!value.removed) {
            return true;
         }

         const context: ValidationContext | undefined = this.options.context as any;

         if (!context) {
            return true;
         }

         // TODO DEPLOY-5220
         // const path = this.path.split('.');
         // const form = (context.form as unknown) as DeployUnit;
         // const disk = form.disks?.find((_, i) => `disks[${i}]` === path[0]);
         // const diskStaticResource = disk?.volumes.flatMap(v => v.staticResources).find(v => v.id === value.id);
         // // TODO ref
         // // const diskLayer = disk?.volumes.flatMap(v => v.staticResources).find(v => v._layerRef === value.ref);

         const boxChildrenForms = (context.childrenForms as unknown) as Box[];
         const boxStaticResource = boxChildrenForms
            .flatMap(v => v.staticResources)
            // eslint-disable-next-line no-underscore-dangle
            .find(v => v._staticResourceRef === value._ref);

         if (boxStaticResource) {
            return this.createError({
               path: `${this.path}.removed`,
               message: `Static resource '${value.id}' is used and cannot be removed.`,
            });
         }

         return true;
      })
      .test('emptyFiles', 'Static resource files should not be empty', (value: DiskStaticResource) => {
         if (!value.removed && value.type === StaticResourceType.Files) {
            return !isEmpty(value.files);
         }

         return true;
      }),
   // TODO: #DEPLOY-5284
   // нужно сделать анонсы и обойти стейджи с такими ресурсами
   // раскомментировать, когда таких стейджей не останется
   // .test(
   //    'validAccessPermissions',
   //    'Access permissions are invalid',
   //    function ValidAccessPermissions(value: DiskStaticResource) {
   //       const context: ValidationContext | undefined = this.options.context as any;

   //       if (!context) {
   //          return true;
   //       }
   //       const stageForm = (context.parentForms[0] as unknown) as Stage;

   //       const { soxService } = stageForm;

   //       if (!soxService) {
   //          return true;
   //       }

   //       if (value.type === StaticResourceType.Files) {
   //          const hasSecretFile = value.files?.some(v => v.type === StaticResourceFileType.Secret);

   //          if (hasSecretFile) {
   //             if (value.accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED) {
   //                return this.createError({
   //                   path: `${this.path}.accessPermissions`,
   //                   message: `Secret files should not have '${unmodifiedAccessPermissionsLabel}' access permissions`,
   //                });
   //             }
   //          }
   //       }

   //       return true;
   //    },
   // ),
);

// endregion

export const volumeLayersSchema = array().of(
   object<VolumeLayer>({
      _layerRef: StringNullableRequiredSchema('Layer ref').test(
         'uniqueId',
         'Volume layer should have unique ID',
         function UniqueId(value: string) {
            const context: ValidationContext | undefined = this.options.context as any;
            const path = this.path.split('.');

            if (!context || path.length < 2) {
               return true;
            }

            const form = (context.form as unknown) as DeployUnit;
            const disk = form.disks?.find((_, i) => `disks[${i}]` === path[0]);
            const volume = disk?.volumes?.find((_, i) => `volumes[${i}]` === path[1]);
            const layers = volume?.layers;

            if (!layers || isEmpty(layers)) {
               return true;
            }

            // eslint-disable-next-line no-underscore-dangle
            const layerRefs = layers.map(v => v._layerRef).filter(v => v === value);

            if (!hasOnlyUniqueValues(layerRefs)) {
               return this.createError({
                  message: 'Volume layers should have unique ID',
               });
            }

            return true;
         },
      ),
   }),
);

export const volumeStaticResourcesSchema = array().of(
   object<VolumeStaticResource>({
      _staticResourceRef: StringNullableRequiredSchema('Static resource ref').test(
         'uniqueId',
         'Volume static resource should have unique ID',
         function UniqueId(value: string) {
            const context: ValidationContext | undefined = this.options.context as any;
            const path = this.path.split('.');

            if (!context || path.length < 2) {
               return true;
            }

            const form = (context.form as unknown) as DeployUnit;
            const disk = form.disks?.find((_, i) => `disks[${i}]` === path[0]);
            const volume = disk?.volumes?.find((_, i) => `volumes[${i}]` === path[1]);
            const staticResources = volume?.staticResources;

            if (!staticResources || isEmpty(staticResources)) {
               return true;
            }

            // eslint-disable-next-line no-underscore-dangle
            const staticResourceRefs = staticResources.map(v => v._staticResourceRef).filter(v => v === value);

            if (!hasOnlyUniqueValues(staticResourceRefs)) {
               return this.createError({
                  message: 'Volume static resources should have unique ID',
               });
            }

            return true;
         },
      ),
      volumeRelativeMountPoint: unixPathSchema('Volume relative mount point'),
   }),
);

export const diskVolumesSchema = array().of(
   object<DiskVolume>({
      /**
       *  Уникальное имя volume (в рамках pod'а).
       *  Должно состоять только из символов, разрешенных в имени porto контейнера. (a..z, A..Z, 0..9, _-@:.)
       */
      id: string().when('removed', {
         is: true,
         then: string().notRequired(),
         otherwise: diskResourceIdValidationSchema('volumes').label('Volume ID'),
      }),
      layers: array<VolumeLayer>().when('removed', {
         is: true,
         then: array().notRequired(),
         otherwise: volumeLayersSchema,
      }),
      staticResources: array<VolumeStaticResource>().when('removed', {
         is: true,
         then: array().notRequired(),
         otherwise: volumeStaticResourcesSchema,
      }),
      persistenceType: string()
         .label('Persistence type')
         .oneOf([
            EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
            EVolumePersistenceType.EVolumePersistenceType_NON_PERSISTENT,
         ])
         .required() as StringSchema<EVolumePersistenceType>,
      _ref: StringRequiredSchema('Volume ref'),
      removed: boolean().notRequired(),
   }).test('isUsed', 'Is volume used', function IsUsed(value: DiskVolume) {
      if (!value.removed) {
         return true;
      }

      const context: ValidationContext | undefined = this.options.context as any;

      if (!context) {
         return true;
      }

      const childrenForms = (context.childrenForms as unknown) as Box[];
      // eslint-disable-next-line no-underscore-dangle
      const volume = childrenForms.flatMap(v => v.volumes).find(v => v._volumeRef === value._ref);

      if (volume) {
         return this.createError({
            path: `${this.path}.removed`,
            message: `Volume '${value.id}' is used and cannot be removed.`,
         });
      }

      return true;
   }),
);

export const deployUnitDisksSchema = array().of(
   object<DeployUnitDisk>({
      id: string().label('Disk ID'),
      type: diskTypeSchema,
      size: diskSizeSchema,
      bandwidth: object<DiskBandwidth>().when('type', (type: DiskType) => diskBandwidthSchema(type)),
      layers: diskLayersSchema,
      staticResources: diskStaticResourcesSchema,
      volumes: diskVolumesSchema,
   }),
);

// endregion

// region Endpoint sets

export const endpointSetIdSchema = StringNullableSchema('Endpoint set')
   .matches(/^[\w-.:]*$/)
   .test('uniqueId', 'Endpoint set should have unique ID', function UniqueId(id: string) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (context) {
         const formContext = (context.form as unknown) as DeployUnit;

         const endpointSetIds = isEmpty(id)
            ? formContext.endpointSets.map(v => v.id).filter(v => isEmpty(v))
            : formContext.endpointSets.map(v => v.id).filter(v => v === id);

         return hasOnlyUniqueValues(endpointSetIds);
      }

      return true;
   });

export const endpointSetLivenessLimitRatioSchema = number().nullable().label('Liveness limit ratio').min(0).max(1);

export const endpointSetSchema = object<DeployUnitEndpointSet>({
   id: endpointSetIdSchema,
   port: portSchema,
   liveness_limit_ratio: endpointSetLivenessLimitRatioSchema,
});

export const deployUnitEndpointSetsSchema = array().of(endpointSetSchema);

// endregion

export const deployUnitNetworkSchema = object<DeployUnitNetwork>({
   networkId: string()
      .label('Project network')
      .matches(/^_[A-Z0-9_]+_$/)
      .required(),
   customSettings: boolean(),
   virtualServiceIds: array().of(string().nullable()),
   ipv4AddressPoolId: string().nullable(),
});

// https://st.yandex-team.ru/DEPLOY-3594#5fe1dcf95e7dbe5e8f34cd19
// https://st.yandex-team.ru/DEPLOY-4676#60ee94b77fbb3061ae5172ba
const networkBandwidthSchema = number()
   .nullable()
   .min(BYTES.MB, '${path} must be greater than or equal to 1 MB/s')
   .max(10 * BYTES.GB, '${path} must be less than or equal to 10 GB/s');

export const deployUnitNetworkBandwidthSchema = object<DeployUnitNetworkBandwidth>({
   guarantee: networkBandwidthSchema.label('Network bandwidth guarantee'),
   limit: networkBandwidthSchema.label('Network bandwidth limit'),
});

// region TVM
const tvmBlackboxSchema = string()
   .oneOf([TvmBlackbox.Prod, TvmBlackbox.Test, TvmBlackbox.ProdYaTeam, TvmBlackbox.TestYaTeam, TvmBlackbox.Stress])
   .required();

const tvmClientModeSchema = string()
   .label('TVM mode')
   .oneOf([TvmClientMode.GetCheck, TvmClientMode.CheckOnly])
   .required() as StringSchema<TvmClientMode>;

const tvmClientSourceSchema = object<TvmClientItemRecord>({
   app: number().label('Source').positive().nullable(),
   alias: string().label('Source alias').nullable(),
}).required();

const tvmClientDestinationSchema = object<TvmClientItemRecord>({
   app: number().label('Destination').positive().nullable().required(),
   alias: string().label('Destination alias').nullable(),
}).required();

const tvmClientDestinationsSchema = array()
   .of(object<TvmClientItemRecord>())
   .when(['mode', 'source'], {
      is: (mode, source) => mode === TvmClientMode.GetCheck && source.app !== null,
      then: array().of(tvmClientDestinationSchema),
      otherwise: array().notRequired(),
   });

const tvmClientSecretSchema = object<LinkToSecret>()
   .label('Secret')
   .nullable()
   .when(['mode', 'source'], {
      is: (mode, source) => mode === TvmClientMode.GetCheck && source.app !== null,
      then: getLinkToSecretValidationSchema().required(),
      otherwise: object().notRequired(),
   });

const tvmClientSchema = object<TvmClient>({
   mode: tvmClientModeSchema,
   source: tvmClientSourceSchema,
   destinations: tvmClientDestinationsSchema,
   secret: tvmClientSecretSchema,
});

const tvmClientsSchema = array().when(
   'enabled',
   schemeIfEnabled<ArraySchema<TvmClient>>(s => s.of(tvmClientSchema).min(1)),
) as ArraySchema<TvmClient>;

const deployUnitTvmSchema = object<DeployUnitTvm>({
   enabled: boolean().label('TMV mode').required(),
   clientPort: number().nullable().when('enabled', schemeIfEnabled(tvmPortSchema)),
   blackbox: string().nullable().when('enabled', schemeIfEnabled(tvmBlackboxSchema)) as Schema<TvmBlackbox>,
   clients: tvmClientsSchema,
   diskType: diskTypeNullableSchema,
   cpuLimit: number().nullable(),
   memoryLimit: number().nullable(),
});
// endregion

const deployUnitTypeSchema = string()
   .label('Deploy Unit type')
   .oneOf([DeployUnitType.PerCluster, DeployUnitType.MultiCluster])
   .required() as Schema<DeployUnitType>;

const disruptionBudgetSchema = number().label('Disruption budget').min(0).nullable();

const rootDisruptionBudgetSchema = number().when('type', {
   is: DeployUnitType.MultiCluster,
   then: disruptionBudgetSchema.required(),
   otherwise: number().nullable(),
});

// region maintenance
const maxTolerableDowntimePodsSchema = number()
   .label('Maintenance budget')
   .positive()
   .max(ref('disruptionBudget'), '${path} must be less than or equal to Disruption budget (${max})')
   .nullable();

const maxTolerableDowntimeSecondsSchema = number().label('Max maintenance duration').positive().max(3600).nullable();

const rootMaxTolerableDowntimePodsSchema = number().positive().when('type', {
   is: DeployUnitType.MultiCluster,
   then: maxTolerableDowntimePodsSchema,
   otherwise: number().nullable(),
});

const rootMaxTolerableDowntimeSecondsSchema = number().positive().when('type', {
   is: DeployUnitType.MultiCluster,
   then: maxTolerableDowntimeSecondsSchema,
   otherwise: number().nullable(),
});
// endregion

const antiaffinitySchema = object<AntiaffinityRecord>({
   perNode: number().label('Per node').positive().nullable(),
   perRack: number().label('Per rack').positive().nullable(),
}).nullable();

const deployUnitMultiClusterLocationSchema = object<DeployUnitLocation>({
   enabled: boolean().required(),
   podCount: number().label('Pods quantity').min(0).when('enabled', requiredIfEnabled),
   antiaffinity: antiaffinitySchema.nullable(),

   disruptionBudget: number().nullable(), // not required
   maxTolerableDowntimePods: number().nullable(), // not required
   maxTolerableDowntimeSeconds: number().nullable(), // not required
});

const deployUnitPerClusterLocationSchema = object<DeployUnitLocation>({
   enabled: boolean().required(),
   podCount: number().label('Pods quantity').min(0).when('enabled', requiredIfEnabled),
   antiaffinity: antiaffinitySchema.nullable(),

   disruptionBudget: disruptionBudgetSchema.when(['enabled', 'podCount'], {
      is: (enabled, podCount) => enabled && podCount && podCount > 0,
      then: disruptionBudgetSchema.required(),
      otherwise: number().notRequired(),
   }),
   maxTolerableDowntimePods: maxTolerableDowntimePodsSchema,
   maxTolerableDowntimeSeconds: maxTolerableDowntimeSecondsSchema,
});

const cpuSchema = IntegerPositiveNullableSchema('CPU per Pod')
   .required()
   .min(100, '${path} must be greater than or equal to 0.1 CPU (100 VCPU)')
   .test('cpuLimit', 'Boxes vCPU limit sum should be less than Pod vCPU limit', function CpuLimit(value) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (!context) {
         return true;
      }

      const deployUnitCPU = value;
      const childrenForms = (context.childrenForms as unknown) as Box[];

      const boxesCPU = childrenForms.reduce((sum, box) => (box.cpuPerBox ? sum + box.cpuPerBox : sum), 0);

      if (deployUnitCPU < boxesCPU) {
         return this.createError({
            message: `Boxes CPU limit sum (${formatNumber(boxesCPU)}) should be less than Pod CPU limit (${formatNumber(
               deployUnitCPU,
            )})`,
         });
      }

      return true;
   });

const ramSchema = IntegerPositiveNullableSchema('RAM per Pod')
   .required()
   .min(0.2 * BYTES.GB, '${path} must be greater than or equal to 0.2 GB (204.8 MB)') /* DEPLOY-3102 */
   .test('ramLimit', 'Boxes RAM limit sum should be less than Pod RAM limit', function RamLimit(value) {
      const context: ValidationContext | undefined = this.options.context as any;

      if (!context) {
         return true;
      }

      const deployUnitRAM = value;
      const childrenForms = (context.childrenForms as unknown) as Box[];

      const boxesRAM = childrenForms.reduce((sum, box) => (box.ramPerBox ? sum + box.ramPerBox : sum), 0);

      if (deployUnitRAM < boxesRAM) {
         return this.createError({
            message: `Boxes RAM limit sum (${formatNumber(boxesRAM)}) should be less than Pod RAM limit (${formatNumber(
               deployUnitRAM,
            )})`,
         });
      }

      return true;
   });

const deployUnitSettingsSchema = object<DeployUnitSettings>({
   alerting: object<AlertingSettings>({
      state: boolean(),
      notificationChannel: string().nullable(),
   }),
   environment: string()
      .label('Environment')
      .oneOf([
         EnvironmentSettings.TESTING,
         EnvironmentSettings.PRESTABLE,
         EnvironmentSettings.STABLE,
         EnvironmentSettings.UNKNOWN,
      ]) as Schema<EnvironmentSettings>,
});

const deployUnitYasmSchema = object<DuYasm>({
   yasmTags: yasmTagsValidationSchema,
   podAgent: object<YasmPodAgent>({
      addPodAgentUserSignals: boolean(),
   }),
});

// region sidecars
const sidecarsSchemaObject: Record<SidecarName, any> = {} as Record<SidecarName, any>;
for (const sidecarName of SidecarsForUpdating) {
   const sidecar = sidecarsUpdateConfig[sidecarName];

   sidecarsSchemaObject[sidecarName] = object({
      resourceRevision: number().nullable().label(`${sidecar.title} sandbox revision`), // DU spec
      overrideLabels: array().of(
         object({
            key: string().label(`${sidecar.title} override label key`),
            value: string().label(`${sidecar.title} override label value`),
         }),
      ),
      labelRevision: number().nullable().label(`${sidecar.title} sandbox revision`), // Stage spec
   });
}
const sidecarsSchema = object(sidecarsSchemaObject);
// end region

// region logbroker config
const destroyPolicySchema = object<LogbrokerDestroyPolicy>({
   maxTries: IntegerPositiveNullableSchema('Max tries'),
   restartPeriodMs: IntegerPositiveNullableSchema('Restart period in ms').min(10000),
});

const customTopicRequestSchema = object<CustomTopicRequest>({
   topicName: StringNullableSchema('Topic name'),
   tvmClientId: IntegerPositiveNullableSchema('TVM client ID').nullable(),
   secret: object<LinkToSecret>().nullable(),
}).test(
   'requiredFields',
   'Required fields if the Topic name field is not empty',
   function isRequiredField(value: CustomTopicRequest) {
      if (value.topicName && !isEmpty(value.topicName)) {
         const { tvmClientId, secret } = value;
         const tvmClientIdError = !tvmClientId || isEmpty(tvmClientId);
         const secretError = !secret || !secret.key || isEmpty(secret.key);

         if (tvmClientIdError || secretError) {
            return this.createError({
               message:
                  tvmClientIdError && secretError
                     ? `TVM client ID and Secret are required if the Topic name field is not empty.`
                     : `${
                          tvmClientIdError ? 'TVM client ID' : 'Secret'
                       } is required if the Topic name field is not empty`,
            });
         }
      }

      return true;
   },
);

const podAdditionalResourcesRequestSchema = object<LogbrokerPodAdditionalResourcesRequest>({
   setCpuToZero: boolean(),
});

const logbrokerConfigSchema = object<LogbrokerConfig>({
   customTopicRequest: customTopicRequestSchema,
   destroyPolicy: destroyPolicySchema,
   podAdditionalResourcesRequest: podAdditionalResourcesRequestSchema,
});
// end region

// region perLocationSettingsSchema
const perLocationSettingsSchema = object<PerLocationSettings>({
   isCustom: boolean(),
   strategy: string().oneOf([
      PerLocationStrategy.Parallel,
      PerLocationStrategy.Sequential,
   ]) as Schema<PerLocationStrategy>,
   locationOrder: array().of(string()),
   needApproval: (setAsArray().of(string()) as any) as Schema<Set<string>>,
});
// end region

// region InfraComponentsSchema
const InfraComponentsSchema = object<InfraComponents>({
   allowAutomaticUpdates: boolean(),
});
// end region

// region patchersRevisionSchema
const PatchersRevisionSchema = object<PatchersRevision>({
   label: IntegerNullableSchema('Runtime version label'),
   value: IntegerNullableSchema('Runtime version').min(activePatcherRevisions.begin).max(activePatcherRevisions.end),
});
// end region

export const deployUnitSchema = object<DeployUnitFormParams>({
   anonymousMemoryLimit: number().nullable().positive().label('Anonymous memory limit'),
   antiaffinity: antiaffinitySchema,
   cpu: cpuSchema,
   defaultLayerSourceFileStoragePolicy: layerSourceFileStoragePolicySchema.label(
      'Default layer fource file storage policy',
   ),
   disks: deployUnitDisksSchema,
   disruptionBudget: rootDisruptionBudgetSchema,
   endpointSets: deployUnitEndpointSetsSchema,
   id: deployUnitIdSchema,
   locations: object().when('type', {
      is: DeployUnitType.MultiCluster,
      then: record(deployUnitMultiClusterLocationSchema),
      otherwise: record(deployUnitPerClusterLocationSchema),
   }) as Schema<DeployUnitLocationMap>,
   maxTolerableDowntimePods: rootMaxTolerableDowntimePodsSchema,
   maxTolerableDowntimeSeconds: rootMaxTolerableDowntimeSecondsSchema,
   networkDefaults: deployUnitNetworkSchema,
   networkBandwidth: deployUnitNetworkBandwidthSchema,
   nodeFilters: object(),
   ram: ramSchema,
   tvm: deployUnitTvmSchema,
   type: deployUnitTypeSchema,
   tempDiskIsolation: boolean(),
   yasm: deployUnitYasmSchema,
   settings: deployUnitSettingsSchema,
   sidecars: sidecarsSchema as Schema<Record<SidecarName, any>>,
   perLocationSettings: perLocationSettingsSchema,
   patchersRevision: PatchersRevisionSchema,
   logbrokerSidecarDiskType: diskTypeNullableSchema,
   podAgentSidecarDiskType: diskTypeNullableSchema,
   logbrokerConfig: logbrokerConfigSchema,
   collectPortometricsFromSidecars: boolean(),
   nodeSegmentId: string().nullable(),
   infraComponents: InfraComponentsSchema,
   transmitSystemLogs: string().oneOf([
      ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE,
      ETransmitSystemLogs.ETransmitSystemLogsPolicy_DISABLED,
      ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED,
   ]) as Schema<ETransmitSystemLogs>,
});

export function deployUnitToFormParams(item = getEmptyDeployUnit()): DeployUnitFormParams {
   return {
      anonymousMemoryLimit: item.anonymousMemoryLimit,
      antiaffinity: item.antiaffinity,
      cpu: item.cpu,
      defaultLayerSourceFileStoragePolicy: item.defaultLayerSourceFileStoragePolicy,
      disks: item.disks,
      disruptionBudget: item.disruptionBudget,
      endpointSets: item.endpointSets,
      id: item.id,
      locations: item.locations,
      maxTolerableDowntimePods: item.maxTolerableDowntimePods,
      maxTolerableDowntimeSeconds: item.maxTolerableDowntimeSeconds,
      networkDefaults: item.networkDefaults,
      networkBandwidth: item.networkBandwidth,
      nodeFilters: item.nodeFilters,
      ram: item.ram,
      tempDiskIsolation: item.tempDiskIsolation,
      tvm: item.tvm,
      type: item.type,
      yasm: item.yasm,
      sidecars: item.sidecars,
      perLocationSettings: item.perLocationSettings,
      patchersRevision: item.patchersRevision,
      logbrokerSidecarDiskType: item.logbrokerSidecarDiskType,
      logbrokerConfig: item.logbrokerConfig,
      podAgentSidecarDiskType: item.podAgentSidecarDiskType,
      collectPortometricsFromSidecars: item.collectPortometricsFromSidecars,
      nodeSegmentId: item.nodeSegmentId,
      infraComponents: item.infraComponents,
      transmitSystemLogs: item.transmitSystemLogs,
      settings: item.settings,
   };
}

export function formParamsToDeployUnit(params: DeployUnitFormParams, oldValue: DeployUnit | undefined): DeployUnit {
   return {
      boxes: [], // clear children
      resources: oldValue?.resources ?? [],

      anonymousMemoryLimit: params.anonymousMemoryLimit,
      antiaffinity: params.antiaffinity,
      cpu: params.cpu,
      defaultLayerSourceFileStoragePolicy: params.defaultLayerSourceFileStoragePolicy,
      disks: params.disks,
      disruptionBudget: params.disruptionBudget,
      endpointSets: params.endpointSets,
      id: params.id,
      locations: params.locations,
      maxTolerableDowntimePods: params.maxTolerableDowntimePods,
      maxTolerableDowntimeSeconds: params.maxTolerableDowntimeSeconds,
      networkDefaults: params.networkDefaults,
      networkBandwidth: params.networkBandwidth,
      nodeFilters: params.nodeFilters,
      ram: params.ram,
      soxService: false, // clear
      tempDiskIsolation: params.tempDiskIsolation,
      tvm: params.tvm,
      type: params.type,
      yasm: params.yasm,
      sidecars: params.sidecars,
      perLocationSettings: params.perLocationSettings,
      revision: null, // clear,
      patchersRevision: params.patchersRevision,
      initialId: params.initialId,
      logbrokerSidecarDiskType: params.logbrokerSidecarDiskType,
      podAgentSidecarDiskType: params.podAgentSidecarDiskType,
      logbrokerConfig: params.logbrokerConfig,
      collectPortometricsFromSidecars: params.collectPortometricsFromSidecars,
      nodeSegmentId: params.nodeSegmentId,
      infraComponents: params.infraComponents,
      transmitSystemLogs: params.transmitSystemLogs,
      settings: params.settings,
   };
}
