import { BYTES, deepClone } from '@yandex-infracloud-ui/libs';

import {
   ECgroupFsMountMode,
   ELayerSourceFileStoragePolicy,
   EResourceAccessPermissions,
   ETransmitSystemLogs,
   EVolumeMountMode,
   EVolumePersistenceType,
   TDeployUnitSpec,
   TDeployUnitSpec_TMultiClusterReplicaSetDeploy,
   TPodTemplateSpec,
   TTvmConfig_EMode,
} from '../../../../proto-typings';

import { getHexRef, noop } from '../../../../utils';

import {
   DEFAULT_DISK_ID,
   DEFAULT_JUGGLER_PORT,
   DEFAULT_LAYER_ID,
   EMPTY_CHECKSUM,
   HDD_BANDWIDTH_LIMIT_FACTOR,
   SIMPLE_HTTP_SERVER_LAYER_ID,
   SSD_BANDWIDTH_LIMIT_FACTOR,
} from '../../../constants';

import { DeepPartial } from '../../../typeHelpers';
import { SidecarName } from '../../Sidecars';
import { patchList, patchNumber, patchObject, patchString } from '../../utils';

import { getEmptyBox, VolumeMountMode } from '../Box';
import { StageParentNodeIds } from '../StageParentNodeIds';
import { StagePatcherVisitor } from '../StagePatcherVisitor';
import { getEmptyWorkload } from '../Workload';
import {
   DeployUnit,
   DiskLayerType,
   DiskType,
   LayerSourceFileStoragePolicy,
   StaticResourceFiles,
   StaticResourceFileType,
   StaticResourceType,
   StaticResourceUrl,
   TvmBlackbox,
   TvmClientMode,
} from './DeployUnit';
import { DeployUnitConverter } from './DeployUnitConverter';
import { DeployUnitPatcher } from './DeployUnitPatcher';

const NETWORK_ID = {
   Default: '_TEST_NETWORK_',
   New: '_NEW_NETWORK_',
};

const emptyBox = getEmptyBox();
const emptyWorkload = getEmptyWorkload();

const podTemplateSpec = (): DeepPartial<TPodTemplateSpec> => ({
   spec: {
      disk_volume_requests: [
         {
            id: DEFAULT_DISK_ID,
            storage_class: DiskType.SSD,
            quota_policy: {
               capacity: 20 * BYTES.GB,
               bandwidth_guarantee: 20 * BYTES.MB,
               bandwidth_limit: 20 * BYTES.MB * SSD_BANDWIDTH_LIMIT_FACTOR,
            },
            labels: {
               x: 3,
               used_by_infra: true,
            },
         },
      ],
      // TODO убрать, когда перестанем патчить это поле в UI
      ip6_address_requests: [
         {
            network_id: NETWORK_ID.Default,
            vlan_id: 'backbone',
            enable_dns: true,
         },
         {
            network_id: NETWORK_ID.Default,
            vlan_id: 'fastbone',
            enable_dns: true,
         },
      ],
      pod_agent_payload: {
         spec: {
            boxes: [
               // {
               //    id: 'Box1',
               // }
            ],
         },
      },
   },
});

const replicaSet = () => ({
   per_cluster_settings: {
      'man': {
         pod_count: 8,
         deployment_strategy: {
            max_unavailable: 8,
            max_surge: 87,
         },
         unknown_field: 4,
      } as any,
      'iva': {
         pod_count: 8,
      },
   },
   replica_set_template: {
      pod_template_spec: podTemplateSpec(),
   },
});

const multiClusterReplicaSet = (): DeepPartial<TDeployUnitSpec_TMultiClusterReplicaSetDeploy> => ({
   replica_set: {
      pod_template_spec: podTemplateSpec(),
   },
});

const getInitialSpec = (): DeepPartial<TDeployUnitSpec> => ({
   network_defaults: {
      network_id: NETWORK_ID.Default,
   },
   replica_set: replicaSet(),
   // multi_cluster_replica_set: multiClusterReplicaSet(),
   // sidecars
   // pod_agent_sandbox_info: {},
   // dynamic_resource_updater_sandbox_info: {},
   // logbroker_tools_sandbox_info: {},
   // tvm_sandbox_info: {},
});

/**
 * @example
 * it('should xxx', check(
 *    spec => {
 *
 *    },
 *    du => {
 *
 *    },
 *    expected => {
 *
 *    },
 *    spec => {
 *
 *    },
 *  ))
 */
const check = (
   patchSpecBefore: (s: DeepPartial<TDeployUnitSpec>) => void,
   patchModel: (m: DeployUnit) => void,
   patchExpectedSpec: (s: DeepPartial<TDeployUnitSpec>) => void,
   checkOnlyBranch?: (s: TDeployUnitSpec) => any,
) => () => {
   const visitor = new StagePatcherVisitor();
   const spec = getInitialSpec() as TDeployUnitSpec;
   patchSpecBefore(spec);

   const model = DeployUnitConverter.fromApi(
      {
         id: 'test',
         spec,
         status: undefined,
      },
      {},
      {},
      {},
   );
   patchModel(model);

   const expectedPatchedSpec = deepClone(spec);
   patchExpectedSpec(expectedPatchedSpec);

   const patchedSpec = DeployUnitPatcher.patch(visitor, new StageParentNodeIds({ id: 'test-stage' }), spec, model);

   if (checkOnlyBranch) {
      expect(checkOnlyBranch(patchedSpec)).toEqual(checkOnlyBranch(expectedPatchedSpec as TDeployUnitSpec));
   } else {
      expect(patchedSpec).toEqual(expectedPatchedSpec);
   }
};

describe('models/ui|DeployUnitPatcher', () => {
   it(
      'should patch antiaffinity Per Cluster',
      check(
         noop,
         du => {
            du.antiaffinity = {
               perNode: 4,
               perRack: null,
            };
         },
         expected => {
            expected.replica_set!.replica_set_template = {
               ...expected.replica_set!.replica_set_template,
               constraints: {
                  antiaffinity_constraints: [{ key: 'node', max_pods: 4 }],
               },
            };
         },
      ),
   );

   it(
      'should patch locations Per Cluster',
      check(
         noop,
         du => {
            du.locations = {
               sas: {
                  enabled: true,
                  antiaffinity: null,
                  podCount: 3,

                  disruptionBudget: 2,
                  maxTolerableDowntimePods: null,
                  maxTolerableDowntimeSeconds: null,
               },
               man: {
                  enabled: true,

                  antiaffinity: null,
                  podCount: 1,

                  disruptionBudget: 1,
                  maxTolerableDowntimePods: null,
                  maxTolerableDowntimeSeconds: null,
               },
               vla: {
                  enabled: false,

                  antiaffinity: null,
                  podCount: 1,

                  disruptionBudget: 1,
                  maxTolerableDowntimePods: null,
                  maxTolerableDowntimeSeconds: null,
               },
            };
         },
         expected => {
            expected.replica_set!.per_cluster_settings = {
               // create
               'sas': {
                  pod_count: 3,
                  deployment_strategy: {
                     max_unavailable: 2,
                  },
               },
               // update
               'man': {
                  pod_count: 1,
                  unknown_field: 4,
                  deployment_strategy: {
                     max_unavailable: 1,
                     max_surge: 87,
                  },
               } as any,
               // delete (vla)
            };
         },
      ),
   );

   it(
      'should patch locations Multi Cluster',
      check(
         spec => {
            spec.replica_set = undefined;
            spec.multi_cluster_replica_set = multiClusterReplicaSet();
         },
         du => {
            du.locations = {
               sas: {
                  enabled: true,
                  podCount: 3,

                  antiaffinity: { perNode: 4, perRack: null },
                  disruptionBudget: 2,
                  maxTolerableDowntimePods: null,
                  maxTolerableDowntimeSeconds: null,
               },
               man: {
                  enabled: true,
                  podCount: 1,

                  antiaffinity: { perNode: null, perRack: 3 },
                  disruptionBudget: 1,
                  maxTolerableDowntimePods: null,
                  maxTolerableDowntimeSeconds: null,
               },
               vla: {
                  enabled: false,
                  podCount: 1,

                  antiaffinity: { perNode: 4, perRack: null },
                  disruptionBudget: 1,
                  maxTolerableDowntimePods: null,
                  maxTolerableDowntimeSeconds: null,
               },
            };
         },
         expected => {
            expected.multi_cluster_replica_set!.replica_set!.clusters = [
               // create
               {
                  cluster: 'sas',
                  spec: {
                     constraints: {
                        antiaffinity_constraints: [{ key: 'node', max_pods: 4 }],
                     },
                     replica_count: 3,
                  },
               },
               // update
               {
                  cluster: 'man',
                  spec: {
                     constraints: {
                        antiaffinity_constraints: [{ key: 'rack', max_pods: 3 }],
                     },
                     replica_count: 1,
                  },
               },
               // delete (vla)
            ];
         },
      ),
   );

   describe('should patch resource_requests', () => {
      it(
         'should patch CPU',
         check(
            noop,
            du => {
               du.cpu = 123;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec = {
                  spec: {
                     ...expected.replica_set!.replica_set_template!.pod_template_spec!.spec,
                     resource_requests: {
                        vcpu_guarantee: 123,
                        vcpu_limit: 123,
                     },
                  },
               };
            },
         ),
      );

      it(
         'should patch RAM',
         check(
            noop,
            du => {
               du.ram = 123;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec = {
                  spec: {
                     ...expected.replica_set!.replica_set_template!.pod_template_spec!.spec,
                     resource_requests: {
                        memory_guarantee: 123,
                        memory_limit: 123,
                     },
                  },
               };
            },
         ),
      );

      it(
         'should patch network bandwidth guarantee (empty limit)',
         check(
            noop,
            du => {
               du.networkBandwidth.guarantee = 123456;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec = {
                  spec: {
                     ...expected.replica_set!.replica_set_template!.pod_template_spec!.spec,
                     resource_requests: {
                        network_bandwidth_guarantee: 123456,
                        // network_bandwidth_limit: undefined
                     },
                  },
               };
            },
         ),
      );

      it(
         'should patch network bandwidth guarantee and limit',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.resource_requests = {
                  network_bandwidth_guarantee: 123,
                  network_bandwidth_limit: 246,
               };
            },
            du => {
               du.networkBandwidth.guarantee = 56;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec = {
                  spec: {
                     ...expected.replica_set!.replica_set_template!.pod_template_spec!.spec,
                     resource_requests: {
                        network_bandwidth_guarantee: 56,
                        network_bandwidth_limit: 246,
                     },
                  },
               };
            },
         ),
      );
   });

   describe('should patch disks', () => {
      it(
         'add SSD',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [];
            },
            du => {
               du.disks = [
                  {
                     id: 'new-disk',
                     type: DiskType.SSD,
                     bandwidth: {
                        guarantee: 3 * BYTES.MB,
                        limit: {
                           defaultSettings: true,
                           default: 3 * BYTES.MB,
                           custom: 3 * BYTES.MB,
                        },
                     },
                     size: 5 * BYTES.GB,
                     volumes: [],
                     layers: [],
                     staticResources: [],
                  },
               ];
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [
                  {
                     id: 'new-disk',
                     quota_policy: {
                        capacity: 5 * BYTES.GB,
                        bandwidth_limit: 3 * BYTES.MB * SSD_BANDWIDTH_LIMIT_FACTOR,
                        bandwidth_guarantee: 3 * BYTES.MB,
                     },
                     storage_class: DiskType.SSD,
                     labels: {
                        used_by_infra: true,
                     },
                  },
               ];
            },
         ),
      );

      it(
         'add HDD',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [];
            },
            du => {
               du.disks = [
                  {
                     id: 'new-disk',
                     type: DiskType.HDD,
                     bandwidth: {
                        guarantee: 3 * BYTES.MB,
                        limit: {
                           defaultSettings: true,
                           default: HDD_BANDWIDTH_LIMIT_FACTOR * 3 * BYTES.MB,
                           custom: null,
                        },
                     },
                     size: 5 * BYTES.GB,
                     volumes: [],
                     layers: [],
                     staticResources: [],
                  },
               ];
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [
                  {
                     id: 'new-disk',
                     quota_policy: {
                        capacity: 5 * BYTES.GB,
                        bandwidth_limit: 3 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                        bandwidth_guarantee: 3 * BYTES.MB,
                     },
                     storage_class: DiskType.HDD,
                     labels: {
                        used_by_infra: true,
                     },
                  },
               ];
            },
         ),
      );

      it(
         'edit default limit (SSD -> HDD))',
         check(
            noop,
            du => {
               du.disks[0].type = DiskType.HDD;
               du.disks[0].size = 5 * BYTES.GB;
               du.disks[0].bandwidth.guarantee = 3 * BYTES.MB;
               // defaultSettings === true
               du.disks[0].bandwidth.limit.default = 3 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [
                  {
                     id: DEFAULT_DISK_ID,
                     quota_policy: {
                        capacity: 5 * BYTES.GB,
                        bandwidth_guarantee: 3 * BYTES.MB,
                        bandwidth_limit: 3 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
                     },
                     storage_class: DiskType.HDD,
                     labels: {
                        x: 3,
                        used_by_infra: true,
                     }, // unchanged
                  },
               ];
            },
         ),
      );

      it(
         'edit custom limit (HDD -> SSD)',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests![0].storage_class =
                  DiskType.HDD;
            },
            du => {
               du.disks[0].type = DiskType.SSD;
               du.disks[0].size = 5 * BYTES.GB;
               du.disks[0].bandwidth.guarantee = 3 * BYTES.MB;
               du.disks[0].bandwidth.limit.defaultSettings = false;
               du.disks[0].bandwidth.limit.custom = 100 * BYTES.MB;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [
                  {
                     id: DEFAULT_DISK_ID,
                     quota_policy: {
                        capacity: 5 * BYTES.GB,
                        bandwidth_guarantee: 3 * BYTES.MB,
                        bandwidth_limit: 100 * BYTES.MB,
                     },
                     storage_class: DiskType.SSD,
                     labels: {
                        x: 3,
                        used_by_infra: true,
                     }, // unchanged
                  },
               ];
            },
         ),
      );

      it(
         'clear disk',
         check(
            noop,
            du => {
               du.disks = [];
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.disk_volume_requests = [];
            },
         ),
      );
   });

   it(
      'should patch network',
      check(
         noop,
         du => {
            du.networkDefaults = {
               networkId: NETWORK_ID.New,
               customSettings: false,
               virtualServiceIds: [],
               ipv4AddressPoolId: null,
            };
         },
         expected => {
            expected.network_defaults = {
               ...expected.network_defaults,
               network_id: NETWORK_ID.New,
            };

            // должно удаляться, если поменялся network_id
            expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.ip6_address_requests = undefined;
         },
      ),
   );

   it(
      'should patch network with virtual_service_ids',
      check(
         spec => {
            spec.network_defaults = {
               ...spec.network_defaults,
               virtual_service_ids: ['c', 'b', 'e'],
               ip4_address_pool_id: 'abn',
            };
         },
         du => {
            du.networkDefaults.virtualServiceIds = ['', 'a', 'b', '', 'c', '', null, ''];
            du.networkDefaults.ipv4AddressPoolId = 'efg';
         },
         expected => {
            expected.network_defaults!.virtual_service_ids![0] = 'a';
            expected.network_defaults!.virtual_service_ids![1] = 'b';
            expected.network_defaults!.virtual_service_ids![2] = 'c';
            expect(expected.network_defaults!.virtual_service_ids!.length).toEqual(3);

            expected.network_defaults!.ip4_address_pool_id = 'efg';
         },
      ),
   );

   it(
      'should patch network ipv4 address pool id',
      check(
         spec => {
            spec.network_defaults = {
               ...spec.network_defaults,
               ip4_address_pool_id: '1513:1381',
            };
         },
         du => {
            du.networkDefaults = {
               networkId: NETWORK_ID.New,
               customSettings: false,
               virtualServiceIds: [],
               ipv4AddressPoolId: null,
            };
         },
         expected => {
            expected.network_defaults = {
               ...expected.network_defaults,
               network_id: NETWORK_ID.New,
               ip4_address_pool_id: undefined,
            };
         },
         spec => spec.network_defaults,
      ),
   );

   describe('should patch endpoint_sets', () => {
      it(
         'add endpoint sets',
         check(
            spec => {
               spec.endpoint_sets = [];
            },
            du => {
               du.endpointSets = [
                  { id: null, port: 345 },
                  { id: 'es', port: null },
               ];
            },
            expected => {
               expected.endpoint_sets = [{ port: 345 }, { id: 'es' }];
            },
         ),
      );

      it(
         'edit endpoint sets',
         check(
            spec => {
               spec.endpoint_sets = [{ port: 80 }];
            },
            du => {
               du.endpointSets = [...du.endpointSets, { id: null, port: null }, { id: 'es', port: 8080 }];
            },
            expected => {
               expected.endpoint_sets = [{ port: 80 }, {}, { id: 'es', port: 8080 }];
            },
         ),
      );

      it(
         'remove endpoint sets',
         check(
            spec => {
               spec.endpoint_sets = [{ port: 80 }, { id: 'es' }];
            },
            du => {
               du.endpointSets = [
                  { ...du.endpointSets[0], removed: true },
                  { ...du.endpointSets[1], removed: true },
               ];
            },
            expected => {
               expected.endpoint_sets = [];
            },
         ),
      );
   });

   describe('should patch logbroker config', () => {
      describe('should patch destroy policy', () => {
         it(
            'should patch max tries: add',
            check(
               noop,
               du => {
                  du.logbrokerConfig.destroyPolicy = {
                     maxTries: 4,
                     restartPeriodMs: 12000,
                  };
               },
               expected => {
                  expected.logbroker_config = {
                     destroy_policy: {
                        max_tries: 4,
                        restart_period_ms: 12000,
                     },
                  };
               },
            ),
         );

         it(
            'should patch max tries and restart period: remove',
            check(
               spec => {
                  spec.logbroker_config = {
                     destroy_policy: {
                        max_tries: 5,
                        restart_period_ms: 11000,
                     },
                  };
               },
               du => {
                  du.logbrokerConfig.destroyPolicy = {
                     maxTries: null,
                     restartPeriodMs: null,
                  };
               },
               expected => {
                  expected.logbroker_config = {};
               },
               spec => spec.logbroker_config,
            ),
         );
      });

      describe('should patch additional resources request per pod', () => {
         it(
            'should not patch: request was missed, zero cpu is disabled',
            check(
               noop,
               du => {
                  du.logbrokerConfig.podAdditionalResourcesRequest = {
                     setCpuToZero: false,
                  };
               },
               expected => {
                  expected.logbroker_config = undefined;
               },
            ),
         );

         it(
            'should patch: request was missed, zero cpu is enabled',
            check(
               noop,
               du => {
                  du.logbrokerConfig.podAdditionalResourcesRequest = {
                     setCpuToZero: true,
                  };
               },
               expected => {
                  expected.logbroker_config = {
                     pod_additional_resources_request: {
                        vcpu_guarantee: 0,
                        vcpu_limit: 0,
                     },
                  };
               },
               spec => spec.logbroker_config,
            ),
         );

         it(
            'should not patch: request was defined, zero cpu is disabled',
            check(
               spec => {
                  spec.logbroker_config = {
                     pod_additional_resources_request: {
                        vcpu_guarantee: 100,
                        vcpu_limit: 200,
                     },
                  };
               },
               du => {
                  du.logbrokerConfig.podAdditionalResourcesRequest = {
                     setCpuToZero: false,
                  };
               },
               expected => {
                  expected.logbroker_config = {
                     pod_additional_resources_request: {
                        vcpu_guarantee: 100,
                        vcpu_limit: 200,
                     },
                  };
               },
               spec => spec.logbroker_config,
            ),
         );

         it(
            'should patch: request was defined, zero cpu is enabled',
            check(
               spec => {
                  spec.logbroker_config = {
                     pod_additional_resources_request: {
                        vcpu_guarantee: 100,
                        vcpu_limit: 200,
                     },
                  };
               },
               du => {
                  du.logbrokerConfig.podAdditionalResourcesRequest = {
                     setCpuToZero: true,
                  };
               },
               expected => {
                  expected.logbroker_config = {
                     pod_additional_resources_request: {
                        vcpu_guarantee: 0,
                        vcpu_limit: 0,
                     },
                  };
               },
               spec => spec.logbroker_config,
            ),
         );

         it(
            'should remove: request was zero, zero cpu is disabled',
            check(
               spec => {
                  spec.logbroker_config = {
                     pod_additional_resources_request: {
                        vcpu_guarantee: 0,
                        vcpu_limit: 0,
                     },
                  };
               },
               du => {
                  du.logbrokerConfig.podAdditionalResourcesRequest = {
                     setCpuToZero: false,
                  };
               },
               expected => {
                  expected.logbroker_config = {};
               },
               spec => spec.logbroker_config,
            ),
         );
      });
   });

   it(
      'should patch images_for_boxes: add',
      check(
         spec => {
            spec.images_for_boxes = {};
         },
         du => {
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  dockerImage: {
                     enabled: true,
                     name: 'dockerName1',
                     tag: 'dockerTag1',
                  },
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  dockerImage: {
                     enabled: true,
                     name: ' dockerName2 ', // whitespace
                     tag: ' dockerTag2 ', // whitespace
                  },
               },
               {
                  ...emptyBox,
                  id: 'Box3',
                  dockerImage: {
                     enabled: false, // disabled
                     name: 'dockerName3',
                     tag: 'dockerTag3',
                  },
               },
            ];
         },
         expected => {
            expected.images_for_boxes = {
               Box1: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName1',
                  tag: 'dockerTag1',
               },
               Box2: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName2',
                  tag: 'dockerTag2',
               },
            };
         },
         spec => spec.images_for_boxes,
      ),
   );

   it(
      'should patch images_for_boxes: edit',
      check(
         spec => {
            spec.images_for_boxes = {
               Box1: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName1',
                  tag: 'dockerTag1',
               },
               Box2: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName2',
                  tag: 'dockerTag2',
               },
            };

            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
               {
                  id: 'Box1',
                  // ...
               },
               {
                  id: 'Box2',
                  // ...
               },
            ];
         },
         du => {
            du.boxes[0].dockerImage = {
               enabled: false,
               name: 'dockerName1_new',
               tag: 'dockerTag1_new',
            };

            du.boxes[1].dockerImage = {
               enabled: true,
               name: 'dockerName2_new',
               tag: 'dockerTag2_new',
            };

            du.boxes.push({
               ...emptyBox,
               id: 'Box3',
               dockerImage: { enabled: true, name: 'dockerName3', tag: 'dockerTag3' },
            });
         },
         expected => {
            expected.images_for_boxes = {
               // disabled
               // Box1: {
               //    registry_host: 'registry.yandex.net',
               //    name: 'dockerName1_new',
               //    tag: 'dockerTag1_new',
               // },
               Box2: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName2_new',
                  tag: 'dockerTag2_new',
               },
               // added
               Box3: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName3',
                  tag: 'dockerTag3',
               },
            };
         },
         spec => spec.images_for_boxes,
      ),
   );

   it(
      'should patch images_for_boxes: remove',
      check(
         spec => {
            spec.images_for_boxes = {
               Box1: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName1',
                  tag: 'dockerTag1',
               },
               Box2: {
                  registry_host: 'registry.yandex.net',
                  name: 'dockerName2',
                  tag: 'dockerTag2',
               },
            };

            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
               {
                  id: 'Box1',
                  // ...
               },
               {
                  id: 'Box2',
                  // ...
               },
            ];
         },
         du => {
            du.boxes = [];
         },
         expected => {
            expected.images_for_boxes = {};
         },
         spec => spec.images_for_boxes,
      ),
   );

   it(
      'should patch box_juggler_configs: add',
      check(
         noop,
         du => {
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  juggler: {
                     enabled: true,
                     port: DEFAULT_JUGGLER_PORT,
                     bundles: [],
                  },
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  juggler: {
                     enabled: true,
                     port: DEFAULT_JUGGLER_PORT,
                     bundles: [{ url: 'rbtorrent1' }, { url: 'rbtorrent2' }],
                  },
               },
            ];
         },
         expected => {
            expected.box_juggler_configs = {
               Box1: {
                  port: DEFAULT_JUGGLER_PORT,
               },
               Box2: {
                  port: DEFAULT_JUGGLER_PORT,
                  archived_checks: [{ url: 'rbtorrent1' }, { url: 'rbtorrent2' }],
               },
            };
         },
         spec => spec.box_juggler_configs,
      ),
   );

   it(
      'should patch box_juggler_configs: edit',
      check(
         spec => {
            spec.box_juggler_configs = {
               Box1: {
                  port: DEFAULT_JUGGLER_PORT,
               },
               Box2: {
                  port: DEFAULT_JUGGLER_PORT,
                  archived_checks: [{ url: 'rbtorrent1' }, { url: 'rbtorrent2' }],
               },
            };
         },
         du => {
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  juggler: {
                     enabled: true,
                     port: 123,
                     bundles: [{ url: 'rbtorrent1' }, { url: 'rbtorrent2' }, { url: 'rbtorrent3' }],
                  },
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  juggler: {
                     enabled: true,
                     port: 1234,
                     bundles: [],
                  },
               },
               {
                  ...emptyBox,
                  id: 'Box3',
                  juggler: {
                     enabled: true,
                     port: 12345,
                     bundles: [{ url: 'rbtorrent4' }],
                  },
               },
            ];
         },
         expected => {
            expected.box_juggler_configs = {
               Box1: {
                  port: 123,
                  archived_checks: [{ url: 'rbtorrent1' }, { url: 'rbtorrent2' }, { url: 'rbtorrent3' }],
               },
               Box2: {
                  port: 1234,
               },
               Box3: {
                  port: 12345,
                  archived_checks: [{ url: 'rbtorrent4' }],
               },
            };
         },
         spec => spec.box_juggler_configs,
      ),
   );

   it(
      'should patch box_juggler_configs: clear',
      check(
         spec => {
            spec.box_juggler_configs = {
               Box1: {
                  port: DEFAULT_JUGGLER_PORT,
               },
               Box2: {
                  port: DEFAULT_JUGGLER_PORT,
                  archived_checks: [{ url: 'rbtorrent1' }, { url: 'rbtorrent2' }],
               },
            };
         },
         du => {
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  juggler: {
                     enabled: false,
                     port: null,
                     bundles: [],
                  },
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  juggler: {
                     enabled: false,
                     port: null,
                     bundles: [],
                  },
               },
            ];
         },
         expected => {
            expected.box_juggler_configs = {};
         },
         spec => spec.box_juggler_configs,
      ),
   );

   describe('edit transmit_system_logs_policy', () => {
      it(
         'enable',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.transmit_system_logs_policy = {};
            },
            du => {
               du.transmitSystemLogs = ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.transmit_system_logs_policy = {
                  transmit_system_logs: ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED,
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );

      it(
         'disable',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.transmit_system_logs_policy = {
                  transmit_system_logs: ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED,
               };
            },
            du => {
               du.transmitSystemLogs = ETransmitSystemLogs.ETransmitSystemLogsPolicy_DISABLED;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.transmit_system_logs_policy = {
                  transmit_system_logs: ETransmitSystemLogs.ETransmitSystemLogsPolicy_DISABLED,
               };
            },
            spec =>
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!
                  .transmit_system_logs_policy,
         ),
      );

      it(
         'clear',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.transmit_system_logs_policy = {
                  transmit_system_logs: ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED,
               };
            },
            du => {
               du.transmitSystemLogs = ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.transmit_system_logs_policy = undefined;
            },
            spec =>
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!
                  .transmit_system_logs_policy,
         ),
      );
   });

   describe('should patch layers', () => {
      it(
         'add disk layers',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {};
            },
            du => {
               du.disks[0].layers.push({
                  type: DiskLayerType.Url,
                  id: DEFAULT_LAYER_ID,
                  url: 'rbtorrent:default',
                  checksum: 'EMPTY:',
                  _ref: getHexRef(),
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
               });

               du.disks[0].layers.push({
                  type: DiskLayerType.Url,
                  id: 'removed',
                  url: 'removed',
                  checksum: 'EMPTY:',
                  _ref: getHexRef(),
                  removed: true,
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
               });

               du.disks[0].layers.push({
                  type: DiskLayerType.Url,
                  id: 'unused1',
                  url: 'unused1',
                  checksum: 'EMPTY:',
                  _ref: getHexRef(),
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.Keep,
               });

               du.disks[0].layers.push({
                  type: DiskLayerType.Url,
                  id: 'unused2',
                  url: 'unused2',
                  checksum: 'EMPTY:',
                  _ref: getHexRef(),
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.Remove,
               });
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  layers: [
                     {
                        id: DEFAULT_LAYER_ID,
                        url: 'rbtorrent:default',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'unused1',
                        url: 'unused1',
                        checksum: 'EMPTY:',
                        layer_source_file_storage_policy:
                           ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP,
                     },
                     {
                        id: 'unused2',
                        url: 'unused2',
                        checksum: 'EMPTY:',
                        layer_source_file_storage_policy:
                           ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_REMOVE,
                     },
                  ],
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );

      it(
         'edit layers',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  layers: [
                     {
                        id: 'base-layer-0',
                        url: 'rbtorrent:default',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer-meta-1',
                        meta: {
                           sandbox_resource: {
                              task_type: 'task_type',
                              task_id: '123456789',
                              resource_type: 'SAMPLE_RELEASE_INTEGRATION_LAYER',
                              resource_id: '1414625542',
                           },
                        },
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer-meta-2',
                        meta: {
                           sandbox_resource: {
                              task_type: 'task_type',
                              task_id: '123456789',
                              resource_type: 'SAMPLE_RELEASE_INTEGRATION_LAYER',
                              resource_id: '1414625542',
                           },
                        },
                        checksum: 'MD5:md5',
                        url: 'rbtorrent:mylayer',
                     },
                     {
                        id: 'base-layer-1',
                        url: 'rbtorrent:custom',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer3',
                        url: 'rbtorrent:layer3',
                        checksum: 'EMPTY:',
                     },
                  ],
               };
            },
            du => {
               du.defaultLayerSourceFileStoragePolicy = LayerSourceFileStoragePolicy.Keep;

               du.disks[0].layers[0].id = 'base-layer-custom';
               du.disks[0].layers[0].url = 'rbtorrent:custom';
               du.disks[0].layers[0].checksum = 'md5:...';

               du.disks[0].layers[1].url = 'new-meta-url';

               du.disks[0].layers[3].removed = true;

               du.disks[0].layers.push({
                  id: 'new-layer',
                  type: DiskLayerType.Url,
                  url: 'rbtorrent:new',
                  checksum: 'EMPTY:',
                  _ref: getHexRef(),
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
               });
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  default_layer_source_file_storage_policy:
                     ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP,
                  layers: [
                     {
                        id: 'base-layer-custom',
                        url: 'rbtorrent:custom',
                        checksum: 'md5:...',
                     },
                     {
                        id: 'layer-meta-1',
                        // removed meta
                        // meta: {
                        //    sandbox_resource: {
                        //       task_type: 'task_type',
                        //       task_id: '123456789',
                        //       resource_type: 'SAMPLE_RELEASE_INTEGRATION_LAYER',
                        //       resource_id: '1414625542',
                        //    },
                        // },
                        url: 'new-meta-url',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer-meta-2',
                        meta: {
                           sandbox_resource: {
                              task_type: 'task_type',
                              task_id: '123456789',
                              resource_type: 'SAMPLE_RELEASE_INTEGRATION_LAYER',
                              resource_id: '1414625542',
                           },
                        },
                        checksum: 'MD5:md5',
                        url: 'rbtorrent:mylayer',
                     },
                     // removed
                     // {
                     //    id: 'base-layer-1',
                     //    url: 'rbtorrent:custom',
                     //    checksum: 'EMPTY:',
                     // },
                     {
                        id: 'layer3',
                        url: 'rbtorrent:layer3',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'new-layer',
                        url: 'rbtorrent:new',
                        checksum: 'EMPTY:',
                     },
                  ],
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );

      it(
         'clear layers',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  layers: [
                     {
                        id: 'base-layer-0',
                        url: 'rbtorrent:default',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer1',
                        url: 'rbtorrent:layer1',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer2',
                        url: 'rbtorrent:layer2',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'base-layer-1',
                        url: 'rbtorrent:custom',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer3',
                        url: 'rbtorrent:layer3',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'layer4',
                        url: 'rbtorrent:layer4',
                        checksum: 'EMPTY:',
                     },
                     {
                        id: 'unused',
                        url: 'rbtorrent:unused',
                        checksum: 'EMPTY:',
                     },
                  ],
               };
            },
            du => {
               du.defaultLayerSourceFileStoragePolicy = LayerSourceFileStoragePolicy.Remove;

               du.disks[0].layers[0].removed = true;
               du.disks[0].layers[1].removed = true;
               du.disks[0].layers[2].removed = true;
               du.disks[0].layers[3].removed = true;
               du.disks[0].layers[4].removed = true;
               du.disks[0].layers[5].removed = true;
               du.disks[0].layers[6].removed = true;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  default_layer_source_file_storage_policy:
                     ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_REMOVE,
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );
   });

   describe('should patch static_resources', () => {
      it(
         'add disk static resources',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {};
            },
            du => {
               du.disks[0].staticResources.push({
                  id: 'Resource1',
                  type: StaticResourceType.Url,
                  url: 'url1',
                  accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                  verification: { enabled: false, checksum: 'checksum1' },
                  _ref: getHexRef(),
               });

               du.disks[0].staticResources.push({
                  id: 'Resource2',
                  type: StaticResourceType.Url,
                  url: 'url2',
                  accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_600,
                  verification: { enabled: true, checksum: 'checksum2' },
                  _ref: getHexRef(),
               });

               du.disks[0].staticResources.push({
                  id: 'Resource3',
                  type: StaticResourceType.Files,
                  files: [
                     {
                        name: 'file1',
                        type: StaticResourceFileType.Raw,
                        raw: 'raw data',
                     },
                     {
                        name: 'file2',
                        type: StaticResourceFileType.Secret,
                        secret: {
                           alias: 'alias',
                           key: 'key',
                        },
                     },
                  ],
                  accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_660,
                  verification: { enabled: true, checksum: 'checksum3' },
                  _ref: getHexRef(),
               });
            },
            expected => {
               // noinspection SpellCheckingInspection
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     {
                        id: 'Resource1',
                        url: 'url1',
                        verification: {
                           disabled: true,
                           checksum: 'checksum1',
                        },
                     },
                     {
                        id: 'Resource2',
                        url: 'url2',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum2',
                        },
                        access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_600,
                     },
                     {
                        id: 'Resource3',
                        files: {
                           files: [
                              {
                                 file_name: 'file1',
                                 raw_data: 'raw data',
                              },
                              {
                                 file_name: 'file2',
                                 secret_data: {
                                    alias: 'alias',
                                    id: 'key',
                                 },
                              },
                           ],
                        },
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum3',
                        },
                        access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_660,
                     },
                  ],
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );

      it(
         'edit static_resources',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     {
                        id: 'Resource1',
                        url: 'url1',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum1',
                        },
                     },
                     {
                        id: 'my-static1',
                        meta: {
                           sandbox_resource: {
                              task_type: 'task_type',
                              task_id: '123456789',
                              resource_type: 'SAMPLE_RELEASE_INTEGRATION_STATIC',
                              resource_id: '987654321',
                           },
                        },
                        verification: {
                           checksum: 'MD5:md5',
                           check_period_ms: 180000,
                        },
                        url: 'old_url',
                     },
                     {
                        id: 'my-static2',
                        url: 'url2',
                        meta: {
                           sandbox_resource: {
                              task_type: 'task_type',
                              task_id: '123456789',
                              resource_type: 'SAMPLE_RELEASE_INTEGRATION_STATIC',
                              resource_id: '987654321',
                           },
                        },
                        verification: {
                           checksum: 'MD5:md5',
                           check_period_ms: 180000,
                        },
                     },
                     {
                        id: 'Resource2',
                        url: 'url2',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum2',
                        },
                     },
                     {
                        id: 'Resource3',
                        url: 'url3',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum3',
                        },
                     },
                     {
                        id: 'Files',
                        files: {
                           files: [
                              {
                                 file_name: 'filename',
                                 multi_secret_data: {
                                    secret_alias: 'sec-123:ver-123',
                                 },
                              },
                              {
                                 file_name: 'old_name',
                                 raw_data: 'raw',
                              },
                           ],
                        },
                        // добавляем, если нет значения
                        // verification: {
                        //    checksum: 'EMPTY:',
                        // },
                     },
                  ],
               };
            },
            du => {
               (du.disks[0].staticResources[1] as StaticResourceUrl).url = 'new_url';
               (du.disks[0].staticResources[1] as StaticResourceUrl).verification.checksum = EMPTY_CHECKSUM;

               (du.disks[0].staticResources[5] as StaticResourceFiles).files[1].name = 'new_name';

               du.disks[0].staticResources[3].removed = true;
            },
            expected => {
               // noinspection SpellCheckingInspection
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     {
                        id: 'Resource1',
                        url: 'url1',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum1',
                        },
                     },
                     {
                        id: 'my-static1',
                        // удаляем, если url изменился
                        // meta: {
                        //    sandbox_resource: {
                        //       task_type: 'task_type',
                        //       task_id: '123456789',
                        //       resource_type: 'SAMPLE_RELEASE_INTEGRATION_STATIC',
                        //       resource_id: '987654321',
                        //    },
                        // },
                        verification: {
                           checksum: EMPTY_CHECKSUM,
                           check_period_ms: 180000,
                        },
                        // новый url
                        url: 'new_url',
                     },
                     {
                        id: 'my-static2',
                        url: 'url2',
                        meta: {
                           sandbox_resource: {
                              task_type: 'task_type',
                              task_id: '123456789',
                              resource_type: 'SAMPLE_RELEASE_INTEGRATION_STATIC',
                              resource_id: '987654321',
                           },
                        },
                        verification: {
                           checksum: 'MD5:md5',
                           check_period_ms: 180000,
                        },
                     },
                     // removed
                     // {
                     //    id: 'Resource2',
                     //    url: 'url2',
                     //    verification: {
                     //       check_period_ms: 180000,
                     //       checksum: 'checksum2',
                     //    },
                     // },
                     {
                        id: 'Resource3',
                        url: 'url3',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum3',
                        },
                     },
                     {
                        id: 'Files',
                        files: {
                           files: [
                              {
                                 file_name: 'filename',
                                 multi_secret_data: {
                                    secret_alias: 'sec-123:ver-123',
                                 },
                              },
                              {
                                 // переименовали
                                 file_name: 'new_name',
                                 raw_data: 'raw',
                              },
                           ],
                        },
                        verification: {
                           // добавляем 'EMPTY:', если нет значения
                           checksum: 'EMPTY:',
                        },
                     },
                  ],
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );

      it(
         'clear static resources',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     {
                        id: 'Resource1',
                        url: 'url1',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum1',
                        },
                     },
                     {
                        id: 'Resource2',
                        url: 'url2',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum2',
                        },
                        access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_660,
                     },
                     {
                        id: 'Resource3',
                        url: 'url3',
                        verification: {
                           check_period_ms: 180000,
                           checksum: 'checksum3',
                        },
                        access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_600,
                     },
                  ],
               };
            },
            du => {
               du.disks[0].staticResources[0].removed = true;
               du.disks[0].staticResources[1].removed = true;
               du.disks[0].staticResources[2].removed = true;
            },
            expected => {
               // noinspection SpellCheckingInspection
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {};
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );

      it(
         'should patch static_resources.check_period_ms: keep untouched',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     {
                        id: 'Resource1',
                        url: 'url1',
                        verification: {
                           checksum: 'checksum1',
                        },
                     },
                  ],
               };

               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'box',
                     static_resources: [
                        {
                           resource_ref: 'Resource1',
                           mount_point: 'mount',
                        },
                     ],
                  },
               ];
            },
            du => {
               du.boxes[0].staticResources[0].mountPoint = 'mount2_new';
            },
            expected => {
               // noinspection SpellCheckingInspection
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     {
                        id: 'Resource1',
                        url: 'url1',
                        verification: {
                           checksum: 'checksum1',
                        },
                     },
                  ],
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources,
         ),
      );
   });

   describe('should patch volumes', () => {
      // TODO volumes

      it(
         'edit volume layers',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  layers: [
                     {
                        id: 'TestVolumeLayer',
                        url: 'rbtorrent:custom',
                        checksum: 'EMPTY:',
                     },
                  ],
               };

               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.volumes = [
                  {
                     generic: {
                        layer_refs: ['TestVolumeLayer'],
                     },
                     id: 'TestVolume',
                  },
               ];
            },
            du => {
               const ref = getHexRef();

               du.disks[0].layers.push({
                  id: 'new_layer',
                  url: 'new_layer_url',
                  type: DiskLayerType.Url,
                  checksum: 'EMPTY:',
                  _ref: ref,
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.None,
               });

               du.disks[0].volumes[0].layers.push({ _layerRef: ref });
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.volumes = [
                  {
                     generic: {
                        layer_refs: ['TestVolumeLayer', 'new_layer'],
                     },
                     id: 'TestVolume',
                  },
               ];

               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources!.layers!.push(
                  {
                     id: 'new_layer',
                     url: 'new_layer_url',
                     checksum: 'EMPTY:',
                  },
               );
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec,
         ),
      );

      // TODO volume static resources
   });

   it(
      'should patch mutable_workloads: edit',
      check(
         noop,
         du => {
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload1',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload2',
                     },
                  ],
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload1',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload2',
                     },
                  ],
               },
            ];
         },
         expected => {
            expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.mutable_workloads = [
               {
                  workload_ref: 'Box1-Workload1',
               },
               {
                  workload_ref: 'Box1-Workload2',
               },
               {
                  workload_ref: 'Box2-Workload1',
               },
               {
                  workload_ref: 'Box2-Workload2',
               },
            ];
         },
         spec =>
            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.mutable_workloads,
      ),
   );

   describe('should patch logrotate_configs', () => {
      it(
         'add',
         check(
            noop,
            du => {
               du.boxes = [
                  {
                     ...emptyBox,
                     id: 'Box',
                     logrotateConfig: {
                        rawConfig: 'raw',
                        runPeriodMillisecond: 3600000,
                     },
                  },
                  {
                     ...emptyBox,
                     id: 'Box1',
                     logrotateConfig: {
                        rawConfig: 'raw1',
                        runPeriodMillisecond: null,
                     },
                  },
                  {
                     ...emptyBox,
                     id: 'Box2',
                     logrotateConfig: {
                        rawConfig: '',
                        runPeriodMillisecond: 7200000,
                     },
                  },
                  {
                     ...emptyBox,
                     id: 'Box3',
                     logrotateConfig: {
                        rawConfig: null,
                        runPeriodMillisecond: null,
                     },
                  },
               ];
            },
            expected => {
               expected.logrotate_configs = {
                  'Box': {
                     raw_config: 'raw',
                     run_period_millisecond: 3600000,
                  },
                  'Box1': {
                     raw_config: 'raw1',
                  },
                  'Box2': {
                     run_period_millisecond: 7200000,
                  },
               };
            },
            spec => spec.logrotate_configs,
         ),
      );
   });

   it(
      'should patch tvm_config: add',
      check(
         noop,
         du => {
            du.tvm = {
               enabled: true,
               clientPort: 2,
               blackbox: TvmBlackbox.ProdYaTeam,
               cpuLimit: null,
               memoryLimit: null,
               diskType: null,
               clients: [
                  {
                     mode: TvmClientMode.CheckOnly,
                     source: {
                        app: 11,
                        alias: 's11',
                     },
                     secret: null,
                  },
                  {
                     mode: TvmClientMode.GetCheck,
                     source: {
                        app: 12,
                        alias: 's12',
                     },
                     destinations: [
                        {
                           app: 1211,
                           alias: 'd1211',
                        },
                     ],
                     secret: {
                        alias: 'sec-s12:ver-s12',
                        key: 'key',
                     },
                  },
               ],
            };
         },
         expected => {
            expected.tvm_config = {
               mode: TTvmConfig_EMode.ENABLED,
               client_port: 2,
               blackbox_environment: TvmBlackbox.ProdYaTeam,
               clients: [
                  {
                     source: {
                        app_id: 11,
                        alias: 's11',
                     },
                  },
                  {
                     source: {
                        app_id: 12,
                        alias: 's12',
                     },
                     destinations: [
                        {
                           app_id: 1211,
                           alias: 'd1211',
                        },
                     ],
                     secret_selector: {
                        alias: 'sec-s12:ver-s12',
                        id: 'key',
                     },
                  },
               ],
            };
         },
         spec => spec.tvm_config,
      ),
   );

   it(
      'should patch tvm_config: edit',
      check(
         spec => {
            spec.tvm_config = {
               mode: TTvmConfig_EMode.ENABLED,
               client_port: 2,
               blackbox_environment: TvmBlackbox.ProdYaTeam,
               clients: [
                  {
                     source: {
                        app_id: 10,
                        abc_service_id: '10',
                        alias: 's10',
                     },
                  },
                  {
                     source: {
                        app_id: 11,
                        abc_service_id: '11',
                        alias: 's11',
                     },
                  },
                  {
                     source: {
                        app_id: 12,
                        abc_service_id: '12',
                        alias: 's12',
                     },
                     destinations: [
                        {
                           app_id: 1211,
                           abc_service_id: '1211',
                           alias: 'd1211',
                        },
                        {
                           app_id: 1212,
                           abc_service_id: '1212',
                           alias: 'd1212',
                        },
                     ],
                     secret_selector: {
                        alias: 'sec-s12:ver-s12',
                        id: 'key',
                     },
                  },
                  {
                     source: {
                        app_id: 13,
                        alias: 's13',
                     },
                     destinations: [
                        {
                           app_id: 1311,
                           abc_service_id: '1311',
                           alias: 'd1311',
                        },
                        {
                           app_id: 1312,
                           abc_service_id: '1312',
                           alias: 'd1312',
                        },
                     ],
                     secret_selector: {
                        alias: 'sec-s13:ver-s13',
                        id: 'key',
                     },
                  },
                  {
                     source: {
                        app_id: 14,
                        abc_service_id: '14',
                        alias: 's14',
                     },
                  },
               ],
            };
         },
         du => {
            du.tvm = {
               enabled: true,
               clientPort: 211,
               blackbox: TvmBlackbox.Stress,
               cpuLimit: null,
               memoryLimit: null,
               diskType: null,
               clients: [
                  {
                     _order: 0,
                     mode: TvmClientMode.GetCheck,
                     source: {
                        app: 10,
                        alias: 's10',
                     },
                     destinations: [
                        {
                           app: 1011,
                           alias: 'd1011',
                        },
                        {
                           app: 1012,
                           alias: 'd1012',
                        },
                     ],
                     secret: {
                        alias: 'sec-s10:ver-s10',
                        key: 'key',
                     },
                  },
                  {
                     mode: TvmClientMode.CheckOnly,
                     source: {
                        app: 13,
                        alias: 's13',
                     },
                     secret: null,
                  },
                  {
                     _order: 2,
                     mode: TvmClientMode.GetCheck,
                     source: {
                        app: 120,
                        alias: 's120',
                     },
                     destinations: [
                        {
                           app: 1213,
                           alias: 'd1213',
                        },
                        {
                           _order: 1,
                           app: 1212,
                           alias: 'd1212_new',
                        },
                     ],
                     secret: {
                        alias: 'sec-s12:ver-s12',
                        key: 'key-new',
                     },
                  },
                  {
                     _order: 4,
                     mode: TvmClientMode.GetCheck,
                     source: {
                        app: 14,
                        alias: 's14_new',
                     },
                     destinations: [
                        {
                           _order: 0,
                           app: 1411,
                           alias: 'd1411',
                        },
                     ],
                     secret: {
                        alias: 'sec-s14:ver-s14',
                        key: 'key',
                     },
                  },
               ],
            };
         },
         expected => {
            expected.tvm_config = {
               mode: TTvmConfig_EMode.ENABLED,
               client_port: 211,
               blackbox_environment: TvmBlackbox.Stress,
               clients: [
                  {
                     source: {
                        app_id: 10,
                        abc_service_id: '10',
                        alias: 's10',
                     },
                     destinations: [
                        {
                           app_id: 1011,
                           alias: 'd1011',
                        },
                        {
                           app_id: 1012,
                           alias: 'd1012',
                        },
                     ],
                     secret_selector: {
                        alias: 'sec-s10:ver-s10',
                        id: 'key',
                     },
                  },
                  {
                     source: {
                        app_id: 120,
                        alias: 's120',
                     },
                     destinations: [
                        {
                           app_id: 1212,
                           abc_service_id: '1212',
                           alias: 'd1212_new',
                        },
                        {
                           app_id: 1213,
                           alias: 'd1213',
                        },
                     ],
                     secret_selector: {
                        alias: 'sec-s12:ver-s12',
                        id: 'key-new',
                     },
                  },
                  {
                     source: {
                        app_id: 14,
                        abc_service_id: '14',
                        alias: 's14_new',
                     },
                     destinations: [
                        {
                           app_id: 1411,
                           alias: 'd1411',
                        },
                     ],
                     secret_selector: {
                        alias: 'sec-s14:ver-s14',
                        id: 'key',
                     },
                  },
                  {
                     source: {
                        app_id: 13,
                        alias: 's13',
                     },
                  },
               ],
            };
         },
         spec => spec.tvm_config,
      ),
   );

   it(
      'should patch pod/workload yasm monitoring: add', ///
      check(
         spec => {
            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra = {};
         },
         du => {
            du.yasm = {
               yasmTags: {
                  itype: 'pod-itype',
                  tags: [
                     {
                        key: 'key1',
                        value: 'pod-value1',
                     },
                     {
                        key: 'key2',
                        value: 'pod-value2',
                     },
                  ],
               },
               podAgent: {
                  addPodAgentUserSignals: false,
               },
            };
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload1',
                        yasm: {
                           unistats: [
                              {
                                 port: 1,
                                 url: 'path1',
                                 yasmTags: {
                                    itype: 'itype1',
                                    tags: [
                                       {
                                          key: 'key1',
                                          value: 'value1',
                                       },
                                       {
                                          key: 'key2',
                                          value: 'value2',
                                       },
                                    ],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload2',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload3',
                        yasm: {
                           unistats: [
                              {
                                 port: null,
                                 url: null,
                                 yasmTags: {
                                    itype: 'itype3',
                                    tags: [],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload4',
                        yasm: {
                           unistats: [
                              {
                                 port: null,
                                 url: null,
                                 yasmTags: {
                                    itype: null,
                                    tags: [
                                       {
                                          key: 'key1',
                                          value: 'value1',
                                       },
                                       {
                                          key: 'key2',
                                          value: 'value2',
                                       },
                                    ],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                  ],
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload1',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload2',
                        yasm: {
                           unistats: [
                              {
                                 port: 2,
                                 url: 'path2',
                                 yasmTags: {
                                    itype: null,
                                    tags: [],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                  ],
               },
            ];
         },
         expected => {
            expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra = {
               monitoring: {
                  labels: {
                     itype: 'pod-itype',
                     key1: 'pod-value1',
                     key2: 'pod-value2',
                  },
                  unistats: [
                     {
                        workload_id: 'Box1-Workload1',
                        port: 1,
                        path: 'path1',
                        labels: {
                           itype: 'itype1',
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload3',
                        labels: {
                           itype: 'itype3',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload4',
                        labels: {
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box2-Workload2',
                        port: 2,
                        path: 'path2',
                     },
                  ],
               },
            };
         },
         spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra,
      ),
   );

   it(
      'should patch pod/workload yasm monitoring: edit',
      check(
         spec => {
            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra = {
               monitoring: {
                  labels: {
                     itype: 'pod-itype',
                     key1: 'pod-value1',
                     key2: 'pod-value2',
                  },
                  unistats: [
                     {
                        workload_id: 'Box1-Workload1',
                        port: 1,
                        path: 'path1',
                        labels: {
                           itype: 'itype1',
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload2',
                        port: 2,
                        path: 'path2',
                        labels: {
                           itype: 'itype2',
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload3',
                        port: 3,
                        path: 'path3',
                        labels: {
                           itype: 'itype3',
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box2-Workload1',
                        labels: {
                           itype: 'itype1',
                        },
                     },
                     {
                        workload_id: 'Box2-Workload2',
                        labels: {
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box2-Workload3',
                        port: 2,
                        path: 'path2',
                     },
                  ],
               },
            };
         },
         du => {
            du.yasm = {
               yasmTags: {
                  itype: 'pod-itype-new',
                  tags: [
                     {
                        key: 'key2',
                        value: 'pod-value2-new',
                     },
                     {
                        key: 'key3',
                        value: 'pod-value3',
                     },
                  ],
               },
               podAgent: {
                  addPodAgentUserSignals: false,
               },
            };
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload1',
                        yasm: {
                           unistats: [
                              {
                                 port: 11,
                                 url: 'path11',
                                 yasmTags: {
                                    itype: 'itype1-new',
                                    tags: [
                                       {
                                          key: 'key2',
                                          value: 'value2-new',
                                       },
                                       {
                                          key: 'key3',
                                          value: 'value3',
                                       },
                                    ],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload2',
                        yasm: {
                           unistats: [
                              {
                                 port: 22,
                                 url: null,
                                 yasmTags: {
                                    itype: 'itype2_new',
                                    tags: [],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload3',
                        yasm: {
                           unistats: [
                              {
                                 port: null,
                                 url: 'path33',
                                 yasmTags: {
                                    itype: null,
                                    tags: [
                                       {
                                          key: 'key1',
                                          value: 'value1-new',
                                       },
                                       {
                                          key: 'key2',
                                          value: 'value2',
                                       },
                                       {
                                          key: 'key3',
                                          value: 'value3',
                                       },
                                    ],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                  ],
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload1',
                        yasm: {
                           unistats: [
                              {
                                 port: null,
                                 url: null,
                                 yasmTags: {
                                    itype: null,
                                    tags: [
                                       {
                                          key: 'key1',
                                          value: 'value1',
                                       },
                                    ],
                                 },
                                 inheritMissedLabels: false,
                                 outputFormat: null,
                                 prefix: null,
                              },
                           ],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                  ],
               },
            ];
         },
         expected => {
            expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra = {
               monitoring: {
                  labels: {
                     itype: 'pod-itype-new',
                     key2: 'pod-value2-new',
                     key3: 'pod-value3',
                  },
                  unistats: [
                     {
                        workload_id: 'Box2-Workload1',
                        labels: {
                           key1: 'value1',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload1',
                        port: 11,
                        path: 'path11',
                        labels: {
                           itype: 'itype1-new',
                           key2: 'value2-new',
                           key3: 'value3',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload2',
                        port: 22,
                        labels: {
                           itype: 'itype2_new',
                        },
                     },
                     {
                        workload_id: 'Box1-Workload3',
                        path: 'path33',
                        labels: {
                           key1: 'value1-new',
                           key2: 'value2',
                           key3: 'value3',
                        },
                     },
                  ],
               },
            };
         },
         spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra,
      ),
   );

   it(
      'should patch pod/workload yasm monitoring: clear',
      check(
         spec => {
            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra = {
               monitoring: {
                  labels: {
                     itype: 'pod-itype',
                     key1: 'pod-value1',
                     key2: 'pod-value2',
                  },
                  unistats: [
                     {
                        workload_id: 'Box1-Workload1',
                        port: 1,
                        path: 'path1',
                        labels: {
                           itype: 'itype1',
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                     {
                        workload_id: 'Box2-Workload1',
                        port: 2,
                        path: 'path2',
                        labels: {
                           itype: 'itype2',
                           key1: 'value1',
                           key2: 'value2',
                        },
                     },
                  ],
               },
            };
         },
         du => {
            du.yasm = {
               yasmTags: {
                  itype: null,
                  tags: [
                     {
                        key: null,
                        value: null,
                     },
                  ],
               },
               podAgent: {
                  addPodAgentUserSignals: false,
               },
            };
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload1',
                        yasm: {
                           unistats: [],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                  ],
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload1',
                        yasm: {
                           unistats: [],
                           porto: {
                              usePortoMetrics: false,
                              inheritMissedLabels: false,
                              yasmTags: {
                                 itype: null,
                                 tags: [],
                              },
                           },
                        },
                     },
                  ],
               },
            ];
         },
         expected => {
            expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra = {
               monitoring: {},
            };
         },
         spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.host_infra,
      ),
   );

   describe('should patch boxes', () => {
      it(
         'should patch box layers',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  layers: [
                     { id: 'unused', url: 'unused', checksum: EMPTY_CHECKSUM },
                     { id: 'base1', url: 'base1', checksum: EMPTY_CHECKSUM },
                     { id: 'base2', url: 'base2', checksum: EMPTY_CHECKSUM },
                  ],
               };

               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'Box1',
                     rootfs: {
                        layer_refs: ['base1'],
                     },
                  },
                  {
                     id: 'Box2',
                     rootfs: {
                        layer_refs: ['base2'],
                     },
                  },
                  {
                     id: 'Box3',
                     rootfs: {
                        layer_refs: ['base1'],
                     },
                  },
               ];
            },
            du => {
               const ref1 = getHexRef();
               const ref2 = getHexRef();

               du.disks[0].layers[1].id = 'new_id';
               du.disks[0].layers[1].url = 'new_url';

               du.disks[0].layers[0].removed = true;

               // новый слой
               du.disks[0].layers.push({
                  id: DEFAULT_LAYER_ID,
                  type: DiskLayerType.Url,
                  url: 'default',
                  checksum: EMPTY_CHECKSUM,
                  _ref: ref1,
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.Keep,
               });

               // новый слой
               du.disks[0].layers.push({
                  id: SIMPLE_HTTP_SERVER_LAYER_ID,
                  type: DiskLayerType.Url,
                  url: 'simple',
                  checksum: EMPTY_CHECKSUM,
                  _ref: ref2,
                  layerSourceFileStoragePolicy: LayerSourceFileStoragePolicy.Remove,
               });

               // меняем слой в боксе на другой
               // eslint-disable-next-line no-underscore-dangle
               du.boxes[0].layers[0]._layerRef = du.disks[0].layers[2]._ref;

               // удаляем слои
               du.boxes[1].layers = [];

               // new box
               du.boxes.push({
                  ...emptyBox,
                  id: 'Box4',
                  workloads: [],
                  // layers: [],
                  // virtualDiskIdRef: null,
               });

               // new box
               du.boxes.push({
                  ...emptyBox,
                  id: 'Box5',
                  workloads: [],
                  layers: [
                     {
                        _layerRef: ref1,
                     },
                     {
                        _layerRef: ref2,
                     },
                  ],
                  virtualDiskIdRef: du.disks[0].id,
               });

               // new box
               du.boxes.push({
                  ...emptyBox,
                  id: 'Box6',
                  workloads: [],
                  layers: [
                     {
                        _layerRef: ref1,
                     },
                     {
                        _layerRef: ref2,
                     },
                  ],
                  // новый box без virtualDiskIdRef, по умолчанию первый диск считаем дефолтным (du.disks[0].id)
                  // virtualDiskIdRef: null,
               });
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'Box1',
                     rootfs: {
                        layer_refs: ['base2'],
                     },
                  },
                  {
                     id: 'Box2',
                     rootfs: {
                        // удаляем слои в старом box
                        // layer_refs: [],
                     },
                  },
                  {
                     id: 'Box3',
                     rootfs: {
                        layer_refs: ['new_id'],
                     },
                  },
                  {
                     id: 'Box4',
                     // новый box без слоёв
                     // rootfs: {
                     //    layer_refs: [],
                     // },
                     cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  },
                  {
                     id: 'Box5',
                     rootfs: {
                        layer_refs: [DEFAULT_LAYER_ID, SIMPLE_HTTP_SERVER_LAYER_ID],
                     },
                     cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  },
                  {
                     id: 'Box6',
                     rootfs: {
                        layer_refs: [DEFAULT_LAYER_ID, SIMPLE_HTTP_SERVER_LAYER_ID],
                     },
                     cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  },
               ];

               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  layers: [
                     // { id: 'unused', url: 'unused', checksum: EMPTY_CHECKSUM }, // removed
                     { id: 'new_id', url: 'new_url', checksum: EMPTY_CHECKSUM },
                     { id: 'base2', url: 'base2', checksum: EMPTY_CHECKSUM },
                     {
                        id: DEFAULT_LAYER_ID,
                        url: 'default',
                        checksum: EMPTY_CHECKSUM,
                        layer_source_file_storage_policy:
                           ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP,
                     },
                     {
                        id: SIMPLE_HTTP_SERVER_LAYER_ID,
                        url: 'simple',
                        checksum: EMPTY_CHECKSUM,
                        layer_source_file_storage_policy:
                           ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_REMOVE,
                     },
                  ],
               };
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec,
         ),
      );

      it(
         'should patch box static resources',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     { id: 'unused', url: 'unused', verification: { checksum: EMPTY_CHECKSUM } },
                     { id: 'base1', url: 'base1', verification: { checksum: EMPTY_CHECKSUM } },
                     { id: 'base2', url: 'base2', verification: { checksum: EMPTY_CHECKSUM } },
                  ],
               };

               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'Box1',
                     static_resources: [
                        {
                           resource_ref: 'base1',
                           mount_point: 'base_directory1',
                        },
                     ],
                  },
                  {
                     id: 'Box2',
                     static_resources: [
                        {
                           resource_ref: 'base2',
                           mount_point: 'base_directory2',
                        },
                     ],
                  },
               ];
            },
            du => {
               const ref1 = getHexRef();
               const ref2 = getHexRef();

               du.disks[0].staticResources[1].id = 'new_id';
               (du.disks[0].staticResources[1] as StaticResourceUrl).url = 'new_url';

               du.disks[0].staticResources[0].removed = true;

               // новый слой
               du.disks[0].staticResources.push({
                  id: DEFAULT_LAYER_ID,
                  type: StaticResourceType.Url,
                  url: 'default',
                  verification: { enabled: true, checksum: EMPTY_CHECKSUM },
                  accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                  _ref: ref1,
               });

               // новый слой
               du.disks[0].staticResources.push({
                  id: SIMPLE_HTTP_SERVER_LAYER_ID,
                  type: StaticResourceType.Url,
                  url: 'simple',
                  verification: { enabled: true, checksum: EMPTY_CHECKSUM },
                  accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
                  _ref: ref2,
               });

               // меняем ресурс в боксе на другой
               // eslint-disable-next-line no-underscore-dangle
               du.boxes[0].staticResources[0]._staticResourceRef = du.disks[0].staticResources[2]._ref;
               du.boxes[0].staticResources[0].mountPoint = 'new_directory';

               // удаляем ресурс
               du.boxes[1].staticResources = [];

               // new box
               du.boxes.push({
                  ...emptyBox,
                  // },
                  // {
                  //    id: 'Box2',
                  //    // rootfs: {
                  //    //    layer_refs: [],
                  //    // },
                  //    cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  // },
                  // {
                  id: 'Box3',
                  workloads: [],
                  // layers: [],
                  // staticResources: [],
               });

               // new box
               du.boxes.push({
                  ...emptyBox,
                  id: 'Box4',
                  workloads: [],
                  // layers: [],
                  staticResources: [
                     {
                        _staticResourceRef: ref1,
                        mountPoint: 'default',
                     },
                     {
                        _staticResourceRef: ref2,
                        mountPoint: 'simple',
                     },
                  ],
                  virtualDiskIdRef: du.disks[0].id,
               });

               // new box
               du.boxes.push({
                  ...emptyBox,
                  id: 'Box5',
                  workloads: [],
                  // layers: [],
                  staticResources: [
                     {
                        _staticResourceRef: ref1,
                        mountPoint: 'default',
                     },
                     {
                        _staticResourceRef: ref2,
                        mountPoint: 'simple',
                     },
                  ],
                  // новый box без virtualDiskIdRef, по умолчанию первый диск считаем дефолтным (du.disks[0].id)
                  // virtualDiskIdRef: null,
               });
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.resources = {
                  static_resources: [
                     // { id: 'unused', url: 'unused', verification: { checksum: EMPTY_CHECKSUM } }, // removed
                     {
                        id: 'new_id',
                        url: 'new_url',
                        verification: { checksum: EMPTY_CHECKSUM },
                     },
                     { id: 'base2', url: 'base2', verification: { checksum: EMPTY_CHECKSUM } },
                     {
                        id: DEFAULT_LAYER_ID,
                        url: 'default',
                        verification: { checksum: EMPTY_CHECKSUM, check_period_ms: 180000 },
                     },
                     {
                        id: SIMPLE_HTTP_SERVER_LAYER_ID,
                        url: 'simple',
                        verification: { checksum: EMPTY_CHECKSUM, check_period_ms: 180000 },
                     },
                  ],
               };

               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'Box1',
                     static_resources: [
                        {
                           resource_ref: 'base2',
                           mount_point: 'new_directory',
                        },
                     ],
                  },
                  {
                     id: 'Box2',
                     static_resources: [],
                  },
                  {
                     id: 'Box3',
                     // static_resources: [], // новый box без ресурсов
                     cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  },
                  {
                     id: 'Box4',
                     static_resources: [
                        {
                           resource_ref: DEFAULT_LAYER_ID,
                           mount_point: 'default',
                        },
                        {
                           resource_ref: SIMPLE_HTTP_SERVER_LAYER_ID,
                           mount_point: 'simple',
                        },
                     ],
                     cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  },
                  {
                     id: 'Box5',
                     static_resources: [
                        {
                           resource_ref: DEFAULT_LAYER_ID,
                           mount_point: 'default',
                        },
                        {
                           resource_ref: SIMPLE_HTTP_SERVER_LAYER_ID,
                           mount_point: 'simple',
                        },
                     ],
                     cgroup_fs_mount_mode: ECgroupFsMountMode.ECgroupFsMountMode_RO, // RO для новых боксов
                  },
               ];
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec,
         ),
      );

      it(
         'should patch box volumes',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.volumes = [
                  {
                     id: 'volume1',
                     // generic
                     // virtual_disk_id_ref
                  },
               ];

               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'Box1',
                     volumes: [
                        {
                           mount_point: 'volumes/volume1',
                           volume_ref: 'volume1',
                        },
                     ],
                  },
                  {
                     id: 'Box2',
                     volumes: [],
                  },
               ];
            },
            du => {
               const ref = getHexRef();

               du.disks[0].volumes.push({
                  _ref: ref,
                  id: 'volume2',
                  layers: [],
                  staticResources: [],
                  persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
               });

               du.boxes[1].volumes = [...du.boxes[0].volumes];
               du.boxes[0].volumes = [];

               du.boxes[1].volumes.push({
                  _volumeRef: ref,
                  mode: VolumeMountMode.ReadWrite,
                  mountPoint: 'volumes/volume2',
               });
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.volumes = [
                  {
                     id: 'volume1',
                     // generic
                     // virtual_disk_id_ref
                  },
                  {
                     id: 'volume2',
                     // generic
                     // virtual_disk_id_ref
                  },
               ];

               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.boxes = [
                  {
                     id: 'Box1',
                     // volumes: [],
                  },
                  {
                     id: 'Box2',
                     volumes: [
                        {
                           // mode: undefined,
                           mount_point: 'volumes/volume1',
                           volume_ref: 'volume1',
                        },
                        {
                           mode: EVolumeMountMode.EVolumeMountMode_READ_WRITE,
                           mount_point: 'volumes/volume2',
                           volume_ref: 'volume2',
                        },
                     ],
                  },
               ];
            },
            spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec,
         ),
      );
   });

   it(
      'should patch workloads',
      check(
         spec => {
            spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.workloads = [
               {
                  id: 'Workload',
                  box_ref: 'Box1',
               },
               {
                  id: 'Deleted-Workload',
               },
            ];
         },
         du => {
            du.boxes = [
               {
                  ...emptyBox,
                  id: 'Box1',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Workload',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload2',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box1-Workload3',
                     },
                  ],
               },
               {
                  ...emptyBox,
                  id: 'Box2',
                  workloads: [
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload1',
                     },
                     {
                        ...emptyWorkload,
                        id: 'Box2-Workload2',
                     },
                  ],
               },
            ];
         },
         expected => {
            expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.workloads = [
               {
                  id: 'Workload',
                  box_ref: 'Box1',
                  transmit_logs: true,
                  start: {
                     command_line: 'bash -c "echo hello world"',
                  },
                  readiness_check: {
                     tcp_check: {
                        port: 80,
                     },
                  },
               },
               {
                  id: 'Box1-Workload2',
                  box_ref: 'Box1',
                  transmit_logs: true,
                  start: {
                     command_line: 'bash -c "echo hello world"',
                  },
                  readiness_check: {
                     tcp_check: {
                        port: 80,
                     },
                  },
               },
               {
                  id: 'Box1-Workload3',
                  box_ref: 'Box1',
                  transmit_logs: true,
                  start: {
                     command_line: 'bash -c "echo hello world"',
                  },
                  readiness_check: {
                     tcp_check: {
                        port: 80,
                     },
                  },
               },
               {
                  id: 'Box2-Workload1',
                  box_ref: 'Box2',
                  transmit_logs: true,
                  start: {
                     command_line: 'bash -c "echo hello world"',
                  },
                  readiness_check: {
                     tcp_check: {
                        port: 80,
                     },
                  },
               },
               {
                  id: 'Box2-Workload2',
                  box_ref: 'Box2',
                  transmit_logs: true,
                  start: {
                     command_line: 'bash -c "echo hello world"',
                  },
                  readiness_check: {
                     tcp_check: {
                        port: 80,
                     },
                  },
               },
            ];
         },
         spec => spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.pod_agent_payload!.spec!.workloads,
      ),
   );

   describe('should patch sidecars', () => {
      it(
         'add sidecars',
         check(
            noop,
            du => {
               du.sidecars[SidecarName.PodAgent].resourceRevision = 1000000001;
               du.sidecars[SidecarName.DynamicResource].resourceRevision = 1000000002;
               du.sidecars[SidecarName.Logbroker].resourceRevision = 1000000003;
               du.sidecars[SidecarName.TVM].resourceRevision = 1000000004;
            },
            expected => {
               expected.pod_agent_sandbox_info = { revision: 1000000001 };
               expected.dynamic_resource_updater_sandbox_info = { revision: 1000000002 };
               expected.logbroker_tools_sandbox_info = { revision: 1000000003 };
               expected.tvm_sandbox_info = { revision: 1000000004 };
            },
         ),
      );

      it(
         'edit sidecars',
         check(
            spec => {
               spec.pod_agent_sandbox_info = { revision: 1234567890 };
               spec.dynamic_resource_updater_sandbox_info = { revision: 2345678901 };
               spec.logbroker_tools_sandbox_info = { revision: 3456789012 };
               spec.tvm_sandbox_info = { revision: 4567890123 };
            },
            du => {
               du.sidecars[SidecarName.PodAgent].resourceRevision = 1000000001;
               du.sidecars[SidecarName.DynamicResource].resourceRevision = 1000000002;
               du.sidecars[SidecarName.Logbroker].resourceRevision = 1000000003;
               du.sidecars[SidecarName.TVM].resourceRevision = 1000000004;
            },
            expected => {
               expected.pod_agent_sandbox_info = { revision: 1000000001 };
               expected.dynamic_resource_updater_sandbox_info = { revision: 1000000002 };
               expected.logbroker_tools_sandbox_info = { revision: 1000000003 };
               expected.tvm_sandbox_info = { revision: 1000000004 };
            },
         ),
      );

      it(
         'clear sidecars',
         check(
            spec => {
               spec.pod_agent_sandbox_info = { revision: 1234567890 };
               spec.dynamic_resource_updater_sandbox_info = { revision: 2345678901 };
               spec.logbroker_tools_sandbox_info = { revision: 3456789012 };
               spec.tvm_sandbox_info = { revision: 4567890123 };
            },
            du => {
               du.sidecars[SidecarName.PodAgent].resourceRevision = null;
               du.sidecars[SidecarName.DynamicResource].resourceRevision = null;
               du.sidecars[SidecarName.Logbroker].resourceRevision = null;
               du.sidecars[SidecarName.TVM].resourceRevision = null;
            },
            expected => {
               expected.pod_agent_sandbox_info = {};
               expected.dynamic_resource_updater_sandbox_info = {};
               expected.logbroker_tools_sandbox_info = {};
               expected.tvm_sandbox_info = {};
            },
         ),
      );
   });

   describe('patch node_filters', () => {
      it(
         'should fill empty filter (undefined)',
         check(
            spec => {
               delete spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter;
            },
            du => {
               du.nodeFilters.requireAvx = true;
               du.nodeFilters.requireAvx2 = true;
               du.nodeFilters.requireBandwidth10G = true;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/cpu_flags/avx] = true AND [/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
         ),
      );

      it(
         'should fill empty filter (empty string)',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter = '';
            },
            du => {
               du.nodeFilters.requireAvx = true;
               du.nodeFilters.requireAvx2 = true;
               du.nodeFilters.requireBandwidth10G = true;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/cpu_flags/avx] = true AND [/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
         ),
      );

      it(
         'should remove predicate',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/cpu_flags/avx] = true AND [/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
            du => {
               du.nodeFilters.requireAvx = false;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
         ),
      );

      it(
         'should keep unknown predicates untouched',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/bla_bla] = true AND [/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
            du => {
               du.nodeFilters.requireAvx2 = false;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/bla_bla] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
         ),
      );

      it(
         'should clear node_filters if empty',
         check(
            spec => {
               spec.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter =
                  '[/labels/cpu_flags/avx2] = true AND [/labels/extras/network/bandwidth_10G] = true';
            },
            du => {
               du.nodeFilters.requireAvx2 = false;
               du.nodeFilters.requireBandwidth10G = false;
            },
            expected => {
               expected.replica_set!.replica_set_template!.pod_template_spec!.spec!.node_filter = undefined;
            },
         ),
      );
   });
});

// region Check TypeScript typings
interface B {
   name: string;
}

interface A {
   age: number;
   b: B;
   list: string[];
   name: string;
}

const a: A = {
   age: 1,
   name: 'x',
   b: { name: 'x' },
   list: [],
};

patchNumber(a, 'age', () => 5);
patchString(a, 'name', () => 'ss');
patchList(a, 'list', () => []);
patchObject(a, 'b', () => ({ name: 'x' }));
