import { deepClone, getSetDifference, isEmpty, isEqual } from '@yandex-infracloud-ui/libs';

import {
   ELayerSourceFileStoragePolicy,
   EResourceAccessPermissions,
   ETransmitSystemLogs,
   EVolumePersistenceType,
   TAntiaffinityConstraint,
   TBoxJugglerConfig,
   TCoredumpPolicy,
   TCoredumpProcessor,
   TDeployUnitSpec,
   TDeployUnitSpec_TClusterSettings,
   TDeployUnitSpec_TDeploySettings_EDeployStrategy,
   TDeployUnitSpec_TEndpointSetTemplate,
   TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings,
   TDockerImageDescription,
   TDownloadableResource,
   TFile,
   TLayer,
   TLogbrokerConfig,
   TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences,
   TMutableWorkload,
   TPodAgentSpec,
   TPodResource,
   TPodSpec,
   TPodSpec_TDiskVolumeRequest,
   TPodTemplateSpec,
   TTvmApp,
   TTvmClient,
   TTvmConfig,
   TTvmConfig_EMode,
   TVolume,
   TVolumeMountedStaticResource,
   TWorkload,
} from '../../../../proto-typings';

import { createKey, restoreObjectFromKey } from '../../../../utils';
import {
   DEFAULT_HDD_BANDWIDTH_GUARANTEE,
   DEFAULT_NETWORK_BANDWIDTH_GUARANTEE,
   DEFAULT_SSD_BANDWIDTH_GUARANTEE,
   DEFAULT_STATIC_RESOURCE_CHECK_PERIOD,
   EMPTY_CHECKSUM,
   HDD_BANDWIDTH_LIMIT_FACTOR,
   SSD_BANDWIDTH_LIMIT_FACTOR,
   YANDEX_DOCKER_REGISTRY_HOST,
} from '../../../constants';
import { ConfirmationType } from '../../Confirmations';
import { extractFormat, isMultisecret, patchSecret } from '../../secrets';
import { extractActualAlias } from '../../secrets/extractActualAlias';
import { SidecarsForUpdating, sidecarsUpdateConfig } from '../../Sidecars';
import {
   patchBoolean,
   patchList,
   patchNumber,
   patchObject,
   patchObjectListById,
   patchString,
   skipEmpty,
} from '../../utils';

import { Box } from '../Box';
import { BoxPatcher } from '../Box/BoxPatcher';
import { StageParentNodeIds } from '../StageParentNodeIds';
import { StagePatcherVisitor } from '../StagePatcherVisitor';
import { Workload, WorkloadPatcher, WorkloadPortoYasm, WorkloadUnistatYasm } from '../Workload';
import { patchYasmLabels } from '../yasm';

import {
   AntiaffinityRecord,
   AntiaffinityType,
   DeployUnit,
   DeployUnitDisk,
   DeployUnitLocationMap,
   DeployUnitTvm,
   DeployUnitType,
   DiskLayer,
   DiskStaticResource,
   DiskType,
   DiskVolume,
   getDefaultAntiaffinity,
   getDefaultTvm,
   LayerSourceFileStoragePolicy,
   PerLocationSettings,
   PerLocationStrategy,
   PodNodeFilters,
   podNodeFiltersMap,
   StaticResourceFile,
   StaticResourceFileType,
   StaticResourceType,
   TvmClientMode,
   VolumeStaticResource,
} from './DeployUnit';
import { DeployUnitConverter } from './DeployUnitConverter';

const layerSourceFileStoragePolicies: Record<LayerSourceFileStoragePolicy, ELayerSourceFileStoragePolicy> = {
   [LayerSourceFileStoragePolicy.None]: ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_NONE,
   [LayerSourceFileStoragePolicy.Keep]: ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP,
   [LayerSourceFileStoragePolicy.Remove]: ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_REMOVE,
};

export class DeployUnitPatcher {
   public static patch(
      visitor: StagePatcherVisitor,
      parentNodes: StageParentNodeIds,
      raw: TDeployUnitSpec | undefined,
      du: DeployUnit,
   ): TDeployUnitSpec {
      return new DeployUnitPatcher(visitor, parentNodes, raw, du).toValue();
   }

   constructor(
      private visitor: StagePatcherVisitor,
      private parentNodes: StageParentNodeIds,
      private raw: TDeployUnitSpec | undefined,
      private du: DeployUnit,
   ) {}

   private getEnabledClusters(locations: DeployUnitLocationMap) {
      return Object.keys(locations).filter(cluster => locations[cluster].enabled);
   }

   private patchAntiaffinityConstrains(
      antiaffinity: AntiaffinityRecord | null,
      antiaffinityConstraints: TAntiaffinityConstraint[],
   ) {
      if (!antiaffinity) {
         return [];
      }

      function p(key: AntiaffinityType, max_pods: number | null) {
         if (max_pods === null) {
            antiaffinityConstraints = antiaffinityConstraints.filter(c => c.key !== key);
            return;
         }

         const exists = antiaffinityConstraints.find(c => c.key === key);
         if (exists) {
            exists.max_pods = max_pods;
            return;
         }

         antiaffinityConstraints.push(({
            key,
            max_pods,
         } as Partial<TAntiaffinityConstraint>) as TAntiaffinityConstraint);
      }

      p(AntiaffinityType.Node, antiaffinity.perNode);
      p(AntiaffinityType.Rack, antiaffinity.perRack);

      return antiaffinityConstraints;
   }

   private patchBoxes(boxes: Box[], podAgentSpec: TPodAgentSpec) {
      patchList(podAgentSpec, 'boxes', specBoxes => {
         for (const box of boxes) {
            const index = specBoxes.some(v => v.id === box.id)
               ? specBoxes.findIndex(v => v.id === box.id)
               : specBoxes.length; // new in the end;
            const boxSpec = specBoxes[index] ?? { id: box.id };

            specBoxes[index] = BoxPatcher.patch(
               this.visitor,
               this.parentNodes.withDuId(this.du),
               boxSpec,
               box,
               this.du.disks,
            );
         }

         // handle removed
         const { removed } = getSetDifference(new Set(specBoxes.map(v => v.id)), new Set(boxes.map(v => v.id)));

         removed.forEach(boxId => {
            const i = specBoxes.findIndex(v => v.id === boxId);
            specBoxes.splice(i, 1);
         });

         return specBoxes;
      });
   }

   private patchCoredump(coredump: Record<string, TCoredumpPolicy>, boxes: Box[]) {
      const workloads = boxes
         .map(box => box.workloads || [])
         .flat()
         .reduce((obj, workload) => {
            obj[workload.id] = workload;
            return obj;
         }, {} as { [id: string]: Workload });

      const newCoredumpConfig: Record<string, TCoredumpPolicy> = {};

      const newWorkloadIds = Object.values(workloads)
         .filter(workload => Boolean(workload.coredumpPolicy))
         .map(e => e.id);

      for (const id of newWorkloadIds) {
         const newCoredumpPolicy = workloads[id].coredumpPolicy as TCoredumpPolicy;
         const coredumpPolicy = coredump[id] ?? newCoredumpPolicy;

         patchObject(coredumpPolicy, 'coredump_processor', coredumpProcessor => {
            const newCoredumpProcessor = newCoredumpPolicy.coredump_processor as TCoredumpProcessor;

            patchNumber(coredumpProcessor, 'count_limit', () => newCoredumpProcessor.count_limit);

            patchNumber(
               coredumpProcessor,
               'total_size_limit_megabytes',
               () => newCoredumpProcessor.total_size_limit_megabytes,
            );

            patchNumber(coredumpProcessor, 'probability', () => newCoredumpProcessor.probability);

            patchNumber(coredumpProcessor, 'cleanup_ttl_seconds', () => newCoredumpProcessor.cleanup_ttl_seconds);

            const newCoredumpAggregator = newCoredumpProcessor.aggregator;
            if (newCoredumpAggregator) {
               patchObject(coredumpProcessor, 'aggregator', aggregator => {
                  patchBoolean(aggregator, 'enabled', () => skipEmpty(newCoredumpAggregator.enabled));

                  patchString(aggregator, 'url', () => newCoredumpAggregator.url);

                  patchString(aggregator, 'service_name', () => newCoredumpAggregator.service_name);

                  patchString(aggregator, 'ctype', () => newCoredumpAggregator.ctype);

                  return aggregator;
               });
            }

            return coredumpProcessor;
         });

         newCoredumpConfig[id] = coredumpPolicy;
      }

      return newCoredumpConfig;
   }

   private patchDisks(disks: DeployUnitDisk[], podSpec: TPodSpec, duId: string) {
      patchList(podSpec, 'disk_volume_requests', diskVolumeRequests => {
         const countDisks = disks?.length ?? 0;

         // add new and update existed
         for (const disk of disks) {
            const exist = diskVolumeRequests.find(v => v.id === disk.id);
            const diskItem: Partial<TPodSpec_TDiskVolumeRequest> = exist ?? {};

            if (countDisks < 2) {
               patchObject(diskItem, 'labels', labels => {
                  // Если в DU находится ровно один диск, надо обязательно указать этот лейбл:
                  // https://deploy.yandex-team.ru/docs/concepts/pod/diskvolumerequests#dctl
                  patchBoolean(labels, 'used_by_infra', () => true);

                  return labels;
               });
            }

            patchString(diskItem, 'storage_class', () => disk.type);

            patchObject(diskItem, 'quota_policy', quotaPolicy => {
               patchNumber(quotaPolicy, 'capacity', () => disk.size);

               const hddBandwidthModalConfirmed =
                  disk.type === DiskType.HDD && this.visitor.isDuConfirmed(duId, ConfirmationType.HddBandwidth);

               const ssdBandwidthModalConfirmed =
                  disk.type === DiskType.SSD && this.visitor.isDuConfirmed(duId, ConfirmationType.SsdBandwidth);

               patchNumber(quotaPolicy, 'bandwidth_guarantee', () => {
                  const hddBandwidthGuarantee = hddBandwidthModalConfirmed
                     ? DEFAULT_HDD_BANDWIDTH_GUARANTEE
                     : disk.bandwidth.guarantee;

                  const ssdBandwidthGuarantee = ssdBandwidthModalConfirmed
                     ? DEFAULT_SSD_BANDWIDTH_GUARANTEE
                     : disk.bandwidth.guarantee;

                  return disk.type === DiskType.SSD ? ssdBandwidthGuarantee : hddBandwidthGuarantee;
               });

               patchNumber(quotaPolicy, 'bandwidth_limit', () => {
                  const bandwidthLimit = disk.bandwidth.limit.defaultSettings
                     ? disk.bandwidth.limit.default
                     : disk.bandwidth.limit.custom;

                  const hddBandwidthLimit = hddBandwidthModalConfirmed
                     ? DEFAULT_HDD_BANDWIDTH_GUARANTEE * HDD_BANDWIDTH_LIMIT_FACTOR
                     : bandwidthLimit;

                  const ssdBandwidthLimit = ssdBandwidthModalConfirmed
                     ? DEFAULT_SSD_BANDWIDTH_GUARANTEE * SSD_BANDWIDTH_LIMIT_FACTOR
                     : bandwidthLimit;

                  return disk.type === DiskType.SSD ? ssdBandwidthLimit : hddBandwidthLimit;
               });

               return quotaPolicy;
            });

            if (!exist) {
               patchString(diskItem, 'id', () => disk.id);
               diskVolumeRequests.push(diskItem as TPodSpec_TDiskVolumeRequest);
            }
         }

         // handle removed
         const { removed } = getSetDifference(
            new Set(diskVolumeRequests.map(v => v.id)),
            new Set(disks.map(v => v.id)),
         );

         removed.forEach(id => {
            const i = diskVolumeRequests.findIndex(v => v.id === id);
            diskVolumeRequests.splice(i, 1);
         });

         return diskVolumeRequests;
      });
   }

   private patchImagesForBoxes(boxes: Box[], specImages: Record<string, TDockerImageDescription>) {
      for (const box of boxes) {
         patchObject(specImages, box.id, boxImage => {
            if (!box.dockerImage.enabled) {
               return undefined;
            }

            patchString(boxImage, 'registry_host', () => YANDEX_DOCKER_REGISTRY_HOST);
            patchString(boxImage, 'name', () => box.dockerImage?.name?.replace(/\s/g, ''));
            patchString(boxImage, 'tag', () => box.dockerImage?.tag?.replace(/\s/g, ''));

            return boxImage;
         });
      }

      // handle removed
      const { removed } = getSetDifference(new Set(Object.keys(specImages)), new Set(boxes.map(b => b.id)));

      removed.forEach(boxId => {
         delete specImages[boxId];
      });

      return specImages;
   }

   private patchLogrotateConfigs(boxes: Box[], specLogrotateConfigs: TDeployUnitSpec['logrotate_configs']) {
      for (const box of boxes) {
         patchObject(specLogrotateConfigs, box.id, logrotateConfig => {
            patchString(logrotateConfig, 'raw_config', () => box.logrotateConfig?.rawConfig);
            patchNumber(logrotateConfig, 'run_period_millisecond', () => box.logrotateConfig?.runPeriodMillisecond);

            return logrotateConfig;
         });
      }

      // handle removed
      const { removed } = getSetDifference(new Set(Object.keys(specLogrotateConfigs)), new Set(boxes.map(b => b.id)));

      removed.forEach(boxId => {
         delete specLogrotateConfigs[boxId];
      });

      return specLogrotateConfigs;
   }

   private patchJuggler(boxes: Box[], boxJugglerConfigs: Record<string, TBoxJugglerConfig>) {
      for (const box of boxes) {
         patchObject(boxJugglerConfigs, box.id, boxJuggler => {
            if (!box.juggler?.enabled) {
               return undefined;
            }

            patchNumber(boxJuggler, 'port', () => box.juggler.port);
            patchList(boxJuggler, 'archived_checks', archivedChecks => {
               const bundles = box.juggler.bundles.filter(v => !isEmpty(v.url));

               if (isEmpty(bundles)) {
                  return undefined;
               }

               for (const bundle of bundles) {
                  const exist = archivedChecks.find(v => v.url === bundle.url);

                  if (!exist) {
                     archivedChecks.push({ url: bundle.url } as TDownloadableResource);
                  } else {
                     const i = archivedChecks.findIndex(v => v.url === bundle.url);

                     archivedChecks.splice(i, 1);
                     archivedChecks.push(exist);
                  }
               }

               // handle removed
               const { removed } = getSetDifference(
                  new Set(archivedChecks.map(v => v.url)),
                  new Set(bundles.map(v => v.url)),
               );

               removed.forEach(url => {
                  const i = archivedChecks.findIndex(v => v.url === url);
                  archivedChecks.splice(i, 1);
               });

               return skipEmpty(archivedChecks);
            });

            return boxJuggler;
         });
      }

      // handle removed
      const { removed } = getSetDifference(
         new Set(Object.keys(boxJugglerConfigs)),
         new Set(boxes.filter(b => b.juggler?.enabled).map(b => b.id)),
      );

      removed.forEach(boxId => {
         delete boxJugglerConfigs[boxId];
      });

      return boxJugglerConfigs;
   }

   private patchMultiClusterLocations(
      du: DeployUnit,
      clusters: TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences[],
   ) {
      // add new and update existed
      for (const cluster in du.locations) {
         if (du.locations.hasOwnProperty(cluster) && du.locations[cluster].enabled) {
            const location = du.locations[cluster];
            const exist = clusters.find(c => c.cluster === cluster);
            const clusterSetting: Partial<TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences> = exist ?? {};

            patchObject(clusterSetting, 'spec', clusterSpec => {
               patchNumber(clusterSpec, 'replica_count', () => location.podCount);

               patchObject(clusterSpec, 'constraints', constraints => {
                  patchList(constraints, 'antiaffinity_constraints', antiaffinityConstraints =>
                     this.patchAntiaffinityConstrains(location.antiaffinity, antiaffinityConstraints),
                  );

                  return constraints;
               });

               return clusterSpec;
            });

            if (!exist) {
               patchString(clusterSetting, 'cluster', () => cluster);
               clusters.push(clusterSetting as TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences);
            }
         }
      }

      // handle removed
      const { removed } = getSetDifference(
         new Set(clusters.map(c => c.cluster)),
         new Set(this.getEnabledClusters(du.locations)),
      );

      removed.forEach(cluster => {
         const i = clusters.findIndex(v => v.cluster === cluster);

         clusters.splice(i, 1);
      });
   }

   private patchNodeFilters(du: DeployUnit, podSpec: TPodSpec) {
      patchString(podSpec, 'node_filter', filters => {
         const separator = ' AND ';
         const previousValue = DeployUnitConverter.getNodeFilters(filters ?? '');
         const predicates = new Set((filters ?? '').split(separator).filter(part => !isEmpty(part)));

         function getEnabledOptions(v: PodNodeFilters): Set<keyof PodNodeFilters> {
            return new Set((Object.keys(v) as (keyof PodNodeFilters)[]).sort().filter(k => v[k]));
         }

         const { added, removed } = getSetDifference(
            getEnabledOptions(previousValue),
            getEnabledOptions(du.nodeFilters),
         );

         added.forEach(k => {
            predicates.add(podNodeFiltersMap[k]);
         });

         removed.forEach(k => {
            predicates.delete(podNodeFiltersMap[k]);
         });

         return skipEmpty(Array.from(predicates).join(separator));
      });
   }

   private patchPerClusterLocations(
      du: DeployUnit,
      clusterHashObject: Record<string, TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings>,
   ) {
      // add new and update existed
      for (const cluster in du.locations) {
         if (du.locations.hasOwnProperty(cluster) && du.locations[cluster].enabled) {
            const location = du.locations[cluster];
            patchObject(clusterHashObject, cluster, clusterSetting => {
               patchNumber(clusterSetting, 'pod_count', () => location.podCount);

               patchObject(clusterSetting, 'deployment_strategy', deploymentStrategy => {
                  patchNumber(deploymentStrategy, 'max_unavailable', () => location.disruptionBudget);
                  patchNumber(
                     deploymentStrategy,
                     'max_tolerable_downtime_pods',
                     () => location.maxTolerableDowntimePods,
                  );
                  patchNumber(
                     deploymentStrategy,
                     'max_tolerable_downtime_seconds',
                     () => location.maxTolerableDowntimeSeconds,
                  );

                  return deploymentStrategy;
               });

               return clusterSetting;
            });
         }
      }

      // handle removed
      const { removed } = getSetDifference(
         new Set(Object.keys(clusterHashObject)),
         new Set(this.getEnabledClusters(du.locations)),
      );

      removed.forEach(cluster => {
         delete clusterHashObject[cluster];
      });
   }

   private patchPerLocationsSettings(spec: TDeployUnitSpec, perLocationSettings: PerLocationSettings) {
      const { strategy, locationOrder, needApproval, isCustom } = perLocationSettings;
      const usePerLocation = strategy === PerLocationStrategy.Sequential;

      patchObject(spec, 'deploy_settings', settings => {
         // спека была кастомная
         const isCustomSpec = DeployUnitConverter.isCustomPerLocationSettings(settings ?? null);
         if (!isCustom) {
            // удаляем, если дефолтные настройки выбраны явно в форме
            if (isCustomSpec) {
               return undefined;
            }
            // дефолтную спеку не трогаем, чтобы оставить максимально zerodiff
            return settings;
         }

         const isDefaultSequence = !usePerLocation && needApproval.size === 0 && !settings.cluster_sequence;

         patchList(settings, 'cluster_sequence', locations => {
            if (isDefaultSequence) {
               // оставляем без изменения дефолтные настройки пустыми
               return undefined;
            }

            const oldLocations = new Map(locations.map(e => [e.yp_cluster, e]));

            return locationOrder.map(location => {
               const oldClusterSettings = oldLocations.get(location);
               const clusterSettings = (oldClusterSettings ?? {}) as Partial<TDeployUnitSpec_TClusterSettings>;

               patchString(clusterSettings, 'yp_cluster', () => location);

               const needLocationApproval = needApproval.has(location);

               if (oldClusterSettings?.hasOwnProperty('need_approval') || needLocationApproval) {
                  // перезаписываем существующее поле или записываем true, false — значение по умолчанию
                  patchBoolean(clusterSettings, 'need_approval', () => needLocationApproval);
               }
               return clusterSettings as TDeployUnitSpec_TClusterSettings;
            });
         });

         // дефолтные настройки для параллельной выкатки
         const isDefaultParallelStrategy = !usePerLocation && Object.keys(settings).length === 0;
         // дефолтные настройки для последовательной выкатки
         const isDefaultSequentalStrategy = usePerLocation && !settings.deploy_strategy && settings.cluster_sequence;

         patchString(settings, 'deploy_strategy', () => {
            if (isDefaultParallelStrategy || isDefaultSequentalStrategy) {
               return undefined;
            }
            return usePerLocation
               ? TDeployUnitSpec_TDeploySettings_EDeployStrategy.SEQUENTIAL
               : TDeployUnitSpec_TDeploySettings_EDeployStrategy.PARALLEL;
         });

         return settings;
      });
   }

   private patchPodTemplateSpec(du: DeployUnit, podTemplateSpec: TPodTemplateSpec, clearIp6: boolean) {
      // FIXME временная мера https://st.yandex-team.ru/DEPLOY-3287
      patchObject(podTemplateSpec, 'labels', labels => {
         const diskIsolationConfirmed = this.visitor.isDuConfirmed(du.id, ConfirmationType.DiskIsolation);

         if (du.tempDiskIsolation || diskIsolationConfirmed) {
            delete labels['disable-disk-isolation-spi-15289'];
         } else {
            labels['disable-disk-isolation-spi-15289'] = true;
         }

         return skipEmpty(labels);
      });

      patchObject(podTemplateSpec, 'spec', podSpec => {
         this.patchResourceRequests(du, podSpec);
         this.patchDisks(du.disks, podSpec, du.id);
         this.patchYasm(du, podSpec);
         this.patchSecrets(du, podSpec);
         this.patchNodeFilters(du, podSpec);

         if (clearIp6) {
            patchList(podSpec, 'ip6_address_requests', () => undefined);
         }

         patchObject(podSpec, 'pod_agent_payload', podAgentPayload => {
            patchObject(podAgentPayload, 'spec', podAgentSpec => {
               this.patchResources(du, podAgentSpec);
               this.patchVolumes(du.disks, podAgentSpec);
               this.patchBoxes(du.boxes, podAgentSpec);
               this.patchWorkloads(du.boxes, podAgentSpec);

               patchObject(podAgentSpec, 'transmit_system_logs_policy', transmitSystemLogsPolicy => {
                  patchString(transmitSystemLogsPolicy, 'transmit_system_logs', () =>
                     du.transmitSystemLogs === ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE
                        ? undefined
                        : du.transmitSystemLogs,
                  );

                  return skipEmpty(transmitSystemLogsPolicy);
               });

               return podAgentSpec;
            });

            return podAgentPayload;
         });

         return podSpec;
      });
   }

   private patchResourceRequests(du: DeployUnit, podSpec: NonNullable<TPodSpec>) {
      patchObject(podSpec, 'resource_requests', resourceRequests => {
         patchNumber(resourceRequests, 'anonymous_memory_limit', () => du.anonymousMemoryLimit);

         const previousVcpuGuarantee = resourceRequests.vcpu_guarantee;
         patchNumber(resourceRequests, 'vcpu_guarantee', () => du.cpu);
         patchNumber(resourceRequests, 'vcpu_limit', createLimitPatcher(previousVcpuGuarantee, du.cpu, du.cpu));

         const previousMemoryGuarantee = resourceRequests.memory_guarantee;
         patchNumber(resourceRequests, 'memory_guarantee', () => du.ram);
         patchNumber(resourceRequests, 'memory_limit', createLimitPatcher(previousMemoryGuarantee, du.ram, du.ram));

         const isNetworkBandwidthUpdateConfirmed = this.visitor.isDuConfirmed(
            du.id,
            ConfirmationType.NetworkBandwidthUpdate,
         );
         patchNumber(resourceRequests, 'network_bandwidth_guarantee', () =>
            isNetworkBandwidthUpdateConfirmed ? DEFAULT_NETWORK_BANDWIDTH_GUARANTEE : du.networkBandwidth.guarantee,
         );
         patchNumber(resourceRequests, 'network_bandwidth_limit', () => du.networkBandwidth.limit);

         return resourceRequests;
      });
   }

   private patchLayer(layer: DiskLayer, specLayer: TLayer, virtualDiskIdRef: string) {
      if (layer.url !== specLayer.url) {
         // удалять meta, если url изменился #DEPLOY-3477
         patchObject(specLayer, 'meta', () => undefined);
      }

      patchString(specLayer, 'id', () => layer.id);
      patchString(specLayer, 'url', () => layer.url);
      patchString(specLayer, 'checksum', () => layer.checksum || EMPTY_CHECKSUM);
      patchString(specLayer, 'virtual_disk_id_ref', () => virtualDiskIdRef);

      patchString(specLayer, 'layer_source_file_storage_policy', () =>
         layer.layerSourceFileStoragePolicy === LayerSourceFileStoragePolicy.None
            ? undefined
            : layerSourceFileStoragePolicies[layer.layerSourceFileStoragePolicy],
      );
   }

   private patchLayers(disks: DeployUnitDisk[], specLayers: TLayer[]) {
      const layerList: string[] = [];

      for (const disk of disks) {
         for (const layer of disk.layers) {
            if (!layer.removed) {
               layerList.push(layer.id);

               const exist =
                  // eslint-disable-next-line no-underscore-dangle
                  layer._order !== undefined && specLayers?.[layer._order] ? specLayers[layer._order] : undefined;

               if (!exist) {
                  const newLayer = {
                     id: layer.id,
                  } as TLayer;

                  specLayers.push(newLayer);

                  this.patchLayer(layer, newLayer, createVirtualDiskIdRef(disks, undefined, disk.id));
               } else {
                  this.patchLayer(layer, exist, createVirtualDiskIdRef(disks, exist.virtual_disk_id_ref, disk.id));
               }
            }
         }
      }

      // handle removed
      const { removed } = getSetDifference(new Set(specLayers.map(v => v.id)), new Set(layerList));

      removed.forEach(id => {
         const i = specLayers.findIndex(v => v.id === id);
         specLayers.splice(i, 1);
      });
   }

   private patchResources(du: DeployUnit, podAgentSpec: TPodAgentSpec) {
      patchObject(podAgentSpec, 'resources', resources => {
         patchString(resources, 'default_layer_source_file_storage_policy', () =>
            du.defaultLayerSourceFileStoragePolicy === LayerSourceFileStoragePolicy.None
               ? undefined
               : layerSourceFileStoragePolicies[du.defaultLayerSourceFileStoragePolicy],
         );

         patchList(resources, 'layers', specLayers => {
            this.patchLayers(du.disks, specLayers);

            return skipEmpty(specLayers);
         });

         patchList(resources, 'static_resources', specStaticResources => {
            this.patchStaticResources(du.disks, specStaticResources);

            return skipEmpty(specStaticResources);
         });

         return resources;
      });
   }

   private patchSecrets(du: DeployUnit, podSpec: TPodSpec) {
      const { initialStageId, secretResolver } = this.visitor;
      if (!initialStageId || !secretResolver) {
         return;
      }

      const isSecretsMigrationConfirmed = this.visitor.isDuConfirmed(du.id, ConfirmationType.SecretsMigration);

      const duId = du.initialId ?? du.id;
      const duSecrets = secretResolver.getDuSecrets(initialStageId, duId);
      if (!duSecrets || duSecrets.length === 0) {
         return;
      }

      for (const duSecret of duSecrets) {
         for (const duVersion of duSecret.versions) {
            // старые секреты
            if (duVersion.legacy) {
               patchObject(podSpec, 'secrets', secrets => {
                  // при миграции удаляем и старые секреты
                  if (duVersion.removed || isSecretsMigrationConfirmed) {
                     delete secrets[duVersion.alias];
                  } else {
                     patchObject(secrets, duVersion.alias, secret => {
                        patchString(secret, 'secret_id', () => duSecret.secretUuid);
                        patchString(secret, 'secret_version', () => duVersion.versionUuid);
                        patchString(secret, 'delegation_token', () =>
                           secretResolver.resolveToken(
                              initialStageId,
                              du.id,
                              duSecret.secretUuid,
                              duVersion.versionUuid,
                           ),
                        );

                        return secret;
                     });

                     // edited alias
                     if (duVersion.newAlias) {
                        secrets[duVersion.newAlias] = secrets[duVersion.alias];
                        delete secrets[duVersion.alias];
                     }
                  }

                  return skipEmpty(secrets);
               });
            }

            // новые секреты и миграция старых
            if (!duVersion.legacy || isSecretsMigrationConfirmed) {
               patchObject(podSpec, 'secret_refs', secretRefs => {
                  if (duVersion.removed) {
                     delete secretRefs[duVersion.alias];
                  } else {
                     patchObject(secretRefs, duVersion.alias, secret => {
                        patchString(secret, 'secret_id', () => duSecret.secretUuid);
                        patchString(secret, 'secret_version', () => duVersion.versionUuid);

                        return secret;
                     });

                     // edited alias
                     if (duVersion.newAlias) {
                        secretRefs[duVersion.newAlias] = secretRefs[duVersion.alias];
                        delete secretRefs[duVersion.alias];
                     }
                  }

                  return skipEmpty(secretRefs);
               });
            }
         }
      }
   }

   private patchStaticResource(
      staticResource: DiskStaticResource,
      specStaticResource: TPodResource,
      virtualDiskIdRef: string,
   ) {
      patchString(specStaticResource, 'id', () => staticResource.id);

      patchString(specStaticResource, 'access_permissions', () =>
         staticResource.accessPermissions !== EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED
            ? staticResource.accessPermissions
            : undefined,
      );

      patchObject(specStaticResource, 'verification', verification => {
         patchBoolean(verification, 'disabled', () => !staticResource.verification.enabled);

         patchString(verification, 'checksum', () => {
            // TODO: EMPTY_CHECKSUM планируем оторвать, когда мигрируем старые стейджи (~3459)
            if (staticResource.verification.enabled) {
               return staticResource.verification.checksum || EMPTY_CHECKSUM;
            }

            return staticResource.verification.checksum;
         });

         return verification;
      });

      patchString(specStaticResource, 'virtual_disk_id_ref', () => virtualDiskIdRef);

      switch (staticResource.type) {
         case StaticResourceType.Url: {
            delete specStaticResource.files;

            // удалять meta, если url изменился #DEPLOY-3477
            if (specStaticResource.url !== staticResource.url) {
               patchObject(specStaticResource, 'meta', () => undefined);
            }

            patchString(specStaticResource, 'url', () => staticResource.url);

            break;
         }

         case StaticResourceType.Files: {
            delete specStaticResource.url;

            // удалять meta, если ресурс изменился??? #DEPLOY-3477
            // patchObject(specStaticResource, 'meta', () => undefined);

            patchObject(specStaticResource, 'files', specStaticResourceFiles => {
               patchList(specStaticResourceFiles, 'files', specFiles => {
                  const files = staticResource.files ?? [];

                  /* eslint-disable no-underscore-dangle */
                  const { removed } = getSetDifference(
                     new Set(specFiles.map((v, i) => i).reverse()),
                     new Set(
                        files
                           ?.filter(v => v._order !== undefined)
                           .map(v => v._order)
                           .reverse(),
                     ),
                  );

                  for (const file of files) {
                     if (file.name && !isEmpty(file.name)) {
                        const exist =
                           file._order !== undefined ? specFiles.find((v, i) => i === file._order) : undefined;

                        if (exist) {
                           this.patchStaticResourceFile(file, exist);
                        } else {
                           const newFile = {} as TFile;

                           this.patchStaticResourceFile(file, newFile);

                           specFiles.push(newFile);
                        }
                     }
                  }
                  /* eslint-enable no-underscore-dangle */

                  // handle removed
                  removed.forEach(i => {
                     if (i !== undefined && specFiles[i]) {
                        specFiles.splice(i, 1);
                     }
                  });

                  return specFiles;
               });

               return specStaticResourceFiles;
            });

            break;
         }
      }
   }

   private patchStaticResourceFile(file: StaticResourceFile, specFile: TFile) {
      patchString(specFile, 'file_name', () => file.name);

      switch (file.type) {
         case StaticResourceFileType.Raw: {
            delete specFile.secret_data;
            delete specFile.multi_secret_data;

            patchString(specFile, 'raw_data', () => file.raw);
            break;
         }

         case StaticResourceFileType.Secret: {
            delete specFile.raw_data;

            if (isMultisecret(file.secret?.key ?? '')) {
               delete specFile.secret_data;

               patchObject(specFile, 'multi_secret_data', specMultiSecret => {
                  patchString(specMultiSecret, 'secret_alias', () =>
                     extractActualAlias(
                        file.secret,
                        this.visitor.secretResolver!,
                        this.parentNodes.stageId,
                        this.du.initialId ?? this.du.id,
                     ),
                  );

                  patchString(specMultiSecret, 'format', () => extractFormat(file.secret?.key ?? ''));

                  return specMultiSecret;
               });
            } else {
               delete specFile.multi_secret_data;

               patchObject(specFile, 'secret_data', specSecret => {
                  patchSecret(
                     specSecret,
                     file.secret,
                     this.visitor.secretResolver!,
                     this.parentNodes.stageId,
                     this.du.initialId ?? this.du.id,
                  );

                  return specSecret;
               });
            }

            break;
         }
      }
   }

   private patchStaticResources(disks: DeployUnitDisk[], specStaticResources: TPodResource[]) {
      /* eslint-disable no-underscore-dangle */
      const staticResourcesList: string[] = [];

      for (const disk of disks) {
         for (const staticResource of disk.staticResources) {
            if (!staticResource.removed) {
               staticResourcesList.push(staticResource.id);

               const exist =
                  staticResource._order !== undefined && specStaticResources?.[staticResource._order]
                     ? specStaticResources[staticResource._order]
                     : undefined;

               if (!exist) {
                  const newStaticResource = {
                     id: staticResource.id,
                     verification: {
                        check_period_ms: staticResource.verification.enabled
                           ? DEFAULT_STATIC_RESOURCE_CHECK_PERIOD
                           : undefined,
                     },
                  } as TPodResource;

                  specStaticResources.push(newStaticResource);

                  this.patchStaticResource(
                     staticResource,
                     newStaticResource,
                     createVirtualDiskIdRef(disks, undefined, disk.id),
                  );
               } else {
                  this.patchStaticResource(
                     staticResource,
                     exist,
                     createVirtualDiskIdRef(disks, exist.virtual_disk_id_ref, disk.id),
                  );
               }
            }
         }
      }

      // handle removed
      const { removed } = getSetDifference(new Set(specStaticResources.map(v => v.id)), new Set(staticResourcesList));

      removed.forEach(id => {
         const i = specStaticResources.findIndex(v => v.id === id);
         specStaticResources.splice(i, 1);
      });
      /* eslint-enable no-underscore-dangle */
   }

   private patchTvm(tvm: DeployUnitTvm, specTvm: TTvmConfig) {
      const defaultTvm = getDefaultTvm();
      if (isEmpty(specTvm) && isEqual(tvm, defaultTvm)) {
         return specTvm;
      }

      patchString(specTvm, 'mode', () => (tvm.enabled ? TTvmConfig_EMode.ENABLED : TTvmConfig_EMode.DISABLED));

      const clients = tvm.clients?.filter(client => !isEmpty(client?.source?.app)) ?? [];

      patchNumber(specTvm, 'client_port', () => tvm.clientPort);

      patchString(specTvm, 'blackbox_environment', blackboxEnvironment =>
         // TODO: @nodejsgirl можно получше продумать логику обнуления конфига в этом месте
         tvm.enabled || !isEmpty(clients) ? tvm.blackbox : blackboxEnvironment,
      );

      patchList(specTvm, 'clients', specClients => {
         /* eslint-disable no-underscore-dangle */
         const { removed: removedClients } = getSetDifference(
            new Set(specClients.map((_, i) => i).reverse()),
            new Set(
               clients
                  ?.filter(v => v._order !== undefined)
                  .map(v => v._order)
                  .reverse(),
            ),
         );

         // для патчинга существующих сложноидентифицируемых объектов в массивах используем _order
         // в клонированных формах _order (индекс в спеке) есть, а исходной спеки нет
         // если спека пустая, то патчить нечего, всегда добавляем новые элементы в массив
         const isEmptySpecClients = isEmpty(specClients);

         for (const client of clients) {
            const existClient =
               client._order !== undefined && !isEmptySpecClients ? specClients[client._order] : undefined;

            if (!existClient) {
               specClients.push({
                  source: {
                     app_id: client.source.app,
                     alias: 'new',
                  },
               } as TTvmClient);
            }

            const clientsLength = isEmpty(specClients) ? 0 : specClients.length - 1;
            const clientIndex = client._order === undefined || !existClient ? clientsLength : client._order;

            // Патчит либо _order клиент, либо только что добавленный (последний)
            patchObject(specClients[clientIndex], 'source', specSource => {
               if (specSource.app_id !== client.source.app) {
                  patchNumber(specSource, 'app_id', () => client.source.app);

                  // TODO: возможно стоит совсем оторвать
                  // эти поля были нужны для саджестов
                  // пока обнуляем, если поменялось приложение
                  patchString(specSource, 'abc_service_id', () => undefined);
               }

               patchString(specSource, 'alias', () => skipEmpty(client.source.alias?.trim()));

               return specSource;
            });

            switch (client.mode) {
               case TvmClientMode.CheckOnly: {
                  patchList(specClients[clientIndex], 'destinations', () => undefined);

                  patchObject(specClients[clientIndex], 'secret_selector', () => undefined);
                  break;
               }

               default: {
                  patchList(specClients[clientIndex], 'destinations', specDestinations => {
                     const destinations = client.destinations?.filter(d => !isEmpty(d.app)) ?? [];
                     const { removed: removedDestinations } = getSetDifference(
                        new Set(specDestinations.map((_, i) => i).reverse()),
                        new Set(
                           destinations
                              ?.filter(v => v._order !== undefined)
                              .map(v => v._order)
                              .reverse(),
                        ),
                     );

                     // для патчинга существующих сложноидентифицируемых объектов в массивах используем _order
                     // в клонированных формах _order (индекс в спеке) есть, а исходной спеки нет
                     // если спека пустая, то патчить нечего, всегда добавляем новые элементы в массив
                     const isEmptySpecDestinations = isEmpty(specDestinations);

                     for (const destination of destinations) {
                        const existDestination =
                           destination._order !== undefined && !isEmptySpecDestinations
                              ? specDestinations[destination._order]
                              : undefined;

                        if (!existDestination) {
                           specDestinations.push({
                              app_id: destination.app,
                           } as TTvmApp);
                        }

                        const destinationsLength = isEmpty(specDestinations) ? 0 : specDestinations.length - 1;
                        const destinationIndex =
                           destination._order === undefined || !existDestination
                              ? destinationsLength
                              : destination._order;

                        if (specDestinations[destinationIndex].app_id !== destination.app) {
                           patchNumber(specDestinations[destinationIndex], 'app_id', () => destination.app);

                           // TODO: возможно стоит совсем оторвать
                           // эти поля были нужны для саджестов
                           // пока обнуляем, если поменялось приложение
                           patchString(specDestinations[destinationIndex], 'abc_service_id', () => undefined);
                        }

                        patchString(specDestinations[destinationIndex], 'alias', () =>
                           skipEmpty(destination.alias?.trim()),
                        );
                     }

                     removedDestinations.forEach(i => {
                        if (i !== undefined && specDestinations[i]) {
                           specDestinations.splice(i, 1);
                        }
                     });

                     return specDestinations.filter(d => !isEmpty(d));
                  });

                  patchObject(specClients[clientIndex], 'secret_selector', secretSelector => {
                     patchSecret(
                        secretSelector,
                        client.secret,
                        this.visitor.secretResolver!,
                        this.parentNodes.stageId,
                        this.du.initialId ?? this.du.id,
                     );

                     return skipEmpty(secretSelector);
                  });
                  break;
               }
            }
         }

         removedClients.forEach(i => {
            if (i !== undefined && specClients[i]) {
               specClients.splice(i, 1);
            }
         });

         return specClients.filter(c => !isEmpty(c.source) || !isEmpty(c.destinations) || !isEmpty(c.secret_selector));
         /* eslint-enable no-underscore-dangle */
      });

      return specTvm;
   }

   private patchVolume(volume: DiskVolume, specVolume: TVolume, disk: DeployUnitDisk, virtualDiskIdRef: string) {
      // слои и статические ресурсы приносим и монтируем только с текущего диска
      const diskLayers = disk.layers;
      const diskStaticResources = disk.staticResources;

      patchString(specVolume, 'persistence_type', () =>
         volume.persistenceType === EVolumePersistenceType.EVolumePersistenceType_NON_PERSISTENT
            ? EVolumePersistenceType.EVolumePersistenceType_NON_PERSISTENT
            : undefined,
      );
      patchString(specVolume, 'id', () => volume.id);
      patchObject(specVolume, 'generic', generic => {
         patchList(generic, 'layer_refs', () => {
            const volumeLayers: string[] = [];

            volume.layers.forEach(volumeLayer => {
               // eslint-disable-next-line no-underscore-dangle
               const layer = diskLayers.find(v => v._ref === volumeLayer._layerRef);

               if (layer?.id) {
                  volumeLayers.push(layer.id);
               }
            });

            return skipEmpty(volumeLayers);
         });

         return generic;
      });

      patchList(specVolume, 'static_resources', specVolumeStaticResources =>
         skipEmpty(
            this.patchVolumeStaticResources(volume.staticResources, specVolumeStaticResources, diskStaticResources),
         ),
      );

      // диск патчить нужно, если их > 1 + он может быть переименован
      patchString(specVolume, 'virtual_disk_id_ref', () => virtualDiskIdRef);
   }

   private patchVolumeStaticResources(
      volumeStaticResources: VolumeStaticResource[],
      specVolumeStaticResources: TVolumeMountedStaticResource[],
      diskStaticResources: DiskStaticResource[],
   ) {
      const newVolumeStaticResources: TVolumeMountedStaticResource[] = [];

      volumeStaticResources.forEach(volumeStaticResource => {
         // eslint-disable-next-line no-underscore-dangle
         const diskStaticResource = diskStaticResources.find(v => v._ref === volumeStaticResource._staticResourceRef);

         if (diskStaticResource?.id) {
            /* eslint-disable no-underscore-dangle */
            const exist =
               volumeStaticResource._order !== undefined && specVolumeStaticResources?.[volumeStaticResource._order]
                  ? specVolumeStaticResources[volumeStaticResource._order]
                  : undefined;
            /* eslint-enable no-underscore-dangle */

            if (exist) {
               patchString(exist, 'resource_ref', () => diskStaticResource.id);

               patchString(exist, 'volume_relative_mount_point', () => volumeStaticResource.volumeRelativeMountPoint);

               newVolumeStaticResources.push(exist);
            } else {
               newVolumeStaticResources.push({
                  resource_ref: diskStaticResource.id,
                  volume_relative_mount_point: volumeStaticResource.volumeRelativeMountPoint || '',
               });
            }
         }
      });

      return newVolumeStaticResources;
   }

   private patchVolumes(disks: DeployUnitDisk[], podAgentSpec: TPodAgentSpec) {
      /* eslint-disable no-underscore-dangle */
      patchList(podAgentSpec, 'volumes', specVolumes => {
         const { removed } = getSetDifference(
            new Set(specVolumes.map((_, i) => i).reverse()),
            new Set(
               disks
                  .flatMap(v => v.volumes ?? [])
                  ?.filter(v => v._order !== undefined && !v.removed)
                  .map(v => v._order)
                  .reverse(),
            ),
         );

         for (const disk of disks) {
            for (const volume of disk.volumes) {
               if (!volume.removed) {
                  const exist =
                     volume._order !== undefined && specVolumes?.[volume._order]
                        ? specVolumes[volume._order]
                        : undefined;

                  if (!exist) {
                     const newVolume = {} as TVolume;

                     specVolumes.push(newVolume);

                     this.patchVolume(volume, newVolume, disk, createVirtualDiskIdRef(disks, undefined, disk.id));
                  } else {
                     this.patchVolume(
                        volume,
                        exist,
                        disk,
                        createVirtualDiskIdRef(disks, exist.virtual_disk_id_ref, disk.id),
                     );
                  }
               }
            }
         }
         /* eslint-enable no-underscore-dangle */

         removed.forEach(i => {
            if (i !== undefined && specVolumes[i]) {
               specVolumes.splice(i, 1);
            }
         });

         return skipEmpty(specVolumes);
      });
   }

   private patchWorkloads(boxes: Box[], podAgentSpec: TPodAgentSpec) {
      const workloadList: string[] = [];

      patchList(podAgentSpec, 'workloads', specWorkloads => {
         for (const box of boxes) {
            for (const workload of box.workloads) {
               const exist = specWorkloads.find(v => v.id === workload.id);

               if (!exist) {
                  specWorkloads.push({
                     id: workload.id,
                  } as TWorkload);
               } else {
                  // TODO: в этом месте надо подумать,
                  // здесь этот способ не подходит из-за пересортировки сущностей в форме (дифф ломается)
                  // const i = specWorkloads.findIndex(v => v.id === workload.id);
                  // specWorkloads.splice(i, 1);
                  // specWorkloads.push(exist);
               }

               workloadList.push(workload.id);

               const i = specWorkloads.findIndex(v => v.id === workload.id);
               specWorkloads[i] = WorkloadPatcher.patch(
                  this.visitor,
                  this.parentNodes.withDuId(this.du).withBoxId(box),
                  specWorkloads[i],
                  workload,
                  box.id,
               );
            }
         }

         // handle removed
         const { removed } = getSetDifference(new Set(specWorkloads.map(v => v.id)), new Set(workloadList));

         removed.forEach(workloadId => {
            const i = specWorkloads.findIndex(v => v.id === workloadId);
            specWorkloads.splice(i, 1);
         });

         return specWorkloads;
      });

      patchList(podAgentSpec, 'mutable_workloads', mutableWorkloads => {
         const { added, removed } = getSetDifference(
            new Set(mutableWorkloads.map(v => v.workload_ref)),
            new Set(workloadList),
         );

         added.forEach(workloadId => {
            mutableWorkloads.push({
               workload_ref: workloadId,
            } as TMutableWorkload);
         });

         removed.forEach(workloadId => {
            const i = mutableWorkloads.findIndex(v => v.workload_ref === workloadId);
            mutableWorkloads.splice(i, 1);
         });

         return isEmpty(mutableWorkloads) ? undefined : mutableWorkloads;
      });
   }

   private patchLogbrokerConfig(du: DeployUnit, logbrokerConfigSpec: TLogbrokerConfig) {
      patchObject(logbrokerConfigSpec, 'custom_topic_request', customTopicRequest => {
         patchString(customTopicRequest, 'topic_name', () => du.logbrokerConfig?.customTopicRequest?.topicName);
         patchNumber(customTopicRequest, 'tvm_client_id', () => du.logbrokerConfig?.customTopicRequest?.tvmClientId);

         patchObject(customTopicRequest, 'secret_selector', secretSelector => {
            patchSecret(
               secretSelector,
               du.logbrokerConfig?.customTopicRequest?.secret,
               this.visitor.secretResolver!,
               this.parentNodes.stageId,
               this.du.initialId ?? this.du.id,
            );

            return skipEmpty(secretSelector);
         });

         return skipEmpty(customTopicRequest);
      });

      patchObject(logbrokerConfigSpec, 'destroy_policy', destroyPolicy => {
         patchNumber(destroyPolicy, 'max_tries', () => du.logbrokerConfig?.destroyPolicy?.maxTries);
         patchNumber(destroyPolicy, 'restart_period_ms', () => du.logbrokerConfig?.destroyPolicy?.restartPeriodMs);

         return skipEmpty(destroyPolicy);
      });

      patchObject(logbrokerConfigSpec, 'pod_additional_resources_request', podAdditionalResourcesRequest => {
         if (du.logbrokerConfig?.podAdditionalResourcesRequest?.setCpuToZero) {
            /*
               we need differentiate empty request and request with 0
               patchNumber considers 0 as empty
               so it won't set it directly to field
             */
            podAdditionalResourcesRequest.vcpu_guarantee = 0;
            podAdditionalResourcesRequest.vcpu_limit = 0;
         } else {
            const oldPodAdditionalResourcesRequest = logbrokerConfigSpec?.pod_additional_resources_request;
            const oldVcpuGuarantee = oldPodAdditionalResourcesRequest?.vcpu_guarantee;
            const oldVcpuLimit = oldPodAdditionalResourcesRequest?.vcpu_limit;

            const isOldCpuAlreadyZero =
               !isEmpty(oldPodAdditionalResourcesRequest) && oldVcpuGuarantee === 0 && oldVcpuLimit === 0;
            if (isOldCpuAlreadyZero) return undefined;
         }

         return skipEmpty(podAdditionalResourcesRequest);
      });
   }

   private patchYasm(du: DeployUnit, podSpec: TPodSpec) {
      const { boxes, yasm } = du;

      patchObject(podSpec, 'host_infra', hostInfra => {
         patchObject(hostInfra, 'monitoring', monitoring => {
            patchObject(monitoring, 'labels', labels => {
               patchYasmLabels({ itype: yasm.yasmTags.itype, tags: du.yasm.yasmTags.tags }, labels);

               return skipEmpty(labels);
            });

            const workloadPortoMap = new Map<string, WorkloadPortoYasm>();
            const workloadUnistatMap = new Map<string, WorkloadUnistatYasm>();

            for (const box of boxes) {
               for (const workload of box.workloads) {
                  const { yasm: workloadYasm, id } = workload;
                  const { porto: yasmPorto, unistats: yasmUnistat } = workloadYasm;

                  for (const unistatRecord of yasmUnistat) {
                     const { url, port } = unistatRecord;

                     // уникальный ключ — тройка id, url, port
                     workloadUnistatMap.set(createKey({ id, url: url ?? '', port: String(port ?? 80) }), unistatRecord);
                  }

                  const { usePortoMetrics } = yasmPorto;

                  if (usePortoMetrics) {
                     workloadPortoMap.set(id, yasmPorto);
                  }
               }
            }

            patchObjectListById(
               monitoring,
               'unistats',
               e => createKey({ id: e.workload_id ?? '', url: e.path ?? '', port: String(e.port ?? 80) }),
               workloadUnistatMap.keys(),
               (unistat, id) => {
                  const { id: workloadId } = restoreObjectFromKey(id);
                  const value = workloadUnistatMap.get(id);
                  if (!value) {
                     return null;
                  }

                  const { yasmTags, port, url, prefix, outputFormat } = value;

                  patchString(unistat, 'workload_id', () => workloadId);
                  patchNumber(unistat, 'port', () => port);
                  patchString(unistat, 'path', () => url);
                  patchString(unistat, 'prefix', () => prefix);
                  patchString(unistat, 'output_format', () => outputFormat);

                  patchBoolean(unistat, 'inherit_missed_labels', () => skipEmpty(value.inheritMissedLabels));

                  patchObject(unistat, 'labels', labels => {
                     patchYasmLabels({ itype: yasmTags.itype, tags: yasmTags.tags }, labels);

                     return skipEmpty(labels);
                  });

                  // unistat всегда будет не пустой, т.к. у него есть workload_id,
                  // нужно проверять, есть ли значения в других полях
                  // const { workload_id, ...params } = unistat;
                  return isEmpty(unistat) ? null : unistat;
               },
            );

            // настройки мониторинга по сбору порто-метрик из Workload - это массив объектов в спеке monitoring.
            patchObjectListById(
               monitoring,
               'workloads',
               e => e.workload_id,
               workloadPortoMap.keys(),
               (workload, id) => {
                  const value = workloadPortoMap.get(id);
                  if (!value) {
                     return null;
                  }
                  const { yasmTags } = value;

                  patchBoolean(workload, 'inherit_missed_labels', () => skipEmpty(value.inheritMissedLabels));

                  patchString(workload, 'workload_id', () => id);

                  patchObject(workload, 'labels', labels => {
                     patchYasmLabels({ itype: yasmTags.itype, tags: yasmTags.tags }, labels);

                     return skipEmpty(labels);
                  });

                  return workload;
               },
            );

            patchObject(monitoring, 'pod_agent', podAgent => {
               const { addPodAgentUserSignals } = yasm.podAgent;

               patchBoolean(podAgent, 'add_pod_agent_user_signals', () => skipEmpty(addPodAgentUserSignals));

               if (addPodAgentUserSignals && !podAgent.labels?.hasOwnProperty('itype')) {
                  patchObject(podAgent, 'labels', labels => ({
                     ...labels,
                  }));
               }

               return skipEmpty(podAgent);
            });

            return monitoring;
         });

         return hostInfra;
      });
   }

   private toValue(): TDeployUnitSpec {
      const { du, raw } = this;
      const spec: TDeployUnitSpec = deepClone(raw ?? {}) as any;

      // Удаляем из спеки ip6_address_requests, если поменялась сеть и настройки не кастомные
      const clearIp6 =
         !du.networkDefaults.customSettings && du.networkDefaults.networkId !== spec.network_defaults?.network_id;

      switch (du.type) {
         case DeployUnitType.PerCluster: {
            patchObject(spec, 'multi_cluster_replica_set', () => undefined);

            patchObject(spec, 'replica_set', replicaSet => {
               patchObject(replicaSet, 'per_cluster_settings', clusterSettings => {
                  this.patchPerClusterLocations(du, clusterSettings);

                  return clusterSettings;
               });

               patchObject(replicaSet, 'replica_set_template', replicaSetTemplate => {
                  const antiaffinity = du.antiaffinity ?? getDefaultAntiaffinity();
                  patchObject(replicaSetTemplate, 'constraints', constraints => {
                     patchList(constraints, 'antiaffinity_constraints', antiaffinityConstraints =>
                        this.patchAntiaffinityConstrains(antiaffinity, antiaffinityConstraints),
                     );

                     return constraints;
                  });

                  patchObject(replicaSetTemplate, 'pod_template_spec', podTemplateSpec => {
                     this.patchPodTemplateSpec(du, podTemplateSpec, clearIp6);

                     return podTemplateSpec;
                  });

                  return replicaSetTemplate;
               });

               return replicaSet;
            });

            this.patchPerLocationsSettings(spec, du.perLocationSettings);

            break;
         }

         case DeployUnitType.MultiCluster: {
            patchObject(spec, 'replica_set', () => undefined);

            patchObject(spec, 'multi_cluster_replica_set', multiClusterReplicaSet => {
               patchObject(multiClusterReplicaSet, 'replica_set', replicaSet => {
                  patchObject(replicaSet, 'pod_template_spec', podTemplateSpec => {
                     this.patchPodTemplateSpec(du, podTemplateSpec, clearIp6);

                     return podTemplateSpec;
                  });

                  patchObject(replicaSet, 'deployment_strategy', deploymentStrategy => {
                     patchNumber(deploymentStrategy, 'max_unavailable', () => du.disruptionBudget);
                     patchNumber(deploymentStrategy, 'max_tolerable_downtime_pods', () => du.maxTolerableDowntimePods);
                     patchNumber(
                        deploymentStrategy,
                        'max_tolerable_downtime_seconds',
                        () => du.maxTolerableDowntimeSeconds,
                     );

                     return deploymentStrategy;
                  });

                  patchList(replicaSet, 'clusters', clusters => {
                     this.patchMultiClusterLocations(du, clusters);

                     return clusters;
                  });

                  return replicaSet;
               });

               return multiClusterReplicaSet;
            });
            break;
         }
      }

      patchObject(spec, 'network_defaults', networkDefaults => {
         patchString(networkDefaults, 'network_id', () => du.networkDefaults.networkId);

         patchList(networkDefaults, 'virtual_service_ids', () => {
            const newSpecIds = du.networkDefaults.virtualServiceIds.filter(v => v && !isEmpty(v));

            return skipEmpty(newSpecIds as string[]);
         });

         patchString(networkDefaults, 'ip4_address_pool_id', () => du.networkDefaults.ipv4AddressPoolId);

         return skipEmpty(networkDefaults);
      });

      // console.log('spec');
      // console.log(spec);
      // console.log('du');
      // console.log(du);

      patchObject(spec, 'box_juggler_configs', boxJugglerConfigs => this.patchJuggler(du.boxes, boxJugglerConfigs));

      patchObject(spec, 'tvm_config', tvm => this.patchTvm(du.tvm, tvm));

      patchObject(spec, 'images_for_boxes', images => this.patchImagesForBoxes(du.boxes, images));

      patchObject(spec, 'logrotate_configs', logrotateConfigs =>
         this.patchLogrotateConfigs(du.boxes, logrotateConfigs),
      );

      patchList(spec, 'endpoint_sets', specEndpointSets => {
         /* eslint-disable no-underscore-dangle */
         const endpointSets = du.endpointSets?.filter(v => !v.removed);

         const { removed } = getSetDifference(
            new Set(specEndpointSets.map((_, i) => i).reverse()),
            new Set(
               endpointSets
                  ?.filter(v => v._order !== undefined)
                  .map(v => v._order)
                  .reverse(),
            ),
         );

         // для патчинга существующих сложноидентифицируемых объектов в массивах используем _order
         // в клонированных формах _order (индекс в спеке) есть, а исходной спеки нет
         // если спека пустая, то патчить нечего, всегда добавляем новые элементы в массив
         const isEmptySpecEndpointSets = isEmpty(specEndpointSets);

         for (const endpointSet of endpointSets) {
            const exist =
               endpointSet._order !== undefined && !isEmptySpecEndpointSets
                  ? specEndpointSets[endpointSet._order]
                  : undefined;

            // существующие Endpoint sets только удаляем:
            // YP не позволит поменять порт в существующем endpointSet
            // Но liveness_limit_ratio менять можно после DEPLOY-4576
            if (!exist) {
               specEndpointSets.push({
                  id: endpointSet.id ?? undefined,
                  port: endpointSet.port ?? undefined,
                  liveness_limit_ratio: endpointSet.liveness_limit_ratio ?? undefined,
               } as TDeployUnitSpec_TEndpointSetTemplate);
            } else {
               patchNumber(exist, 'liveness_limit_ratio', () => endpointSet.liveness_limit_ratio);
            }
         }

         removed.forEach(i => {
            if (i !== undefined && specEndpointSets[i]) {
               specEndpointSets.splice(i, 1);
            }
         });

         return isEmpty(specEndpointSets) || isEqual(specEndpointSets, [{} as any]) ? [] : specEndpointSets;
         /* eslint-enable no-underscore-dangle */
      });

      patchObject(spec, 'coredump_config', coredump => this.patchCoredump(coredump, du.boxes));

      patchNumber(spec, 'patchers_revision', () => {
         const duPatchersAutoupdateRevisionConfirmed = this.visitor.isDuConfirmed(
            du.id,
            ConfirmationType.PatchersRevisionUpdate,
         );

         if (duPatchersAutoupdateRevisionConfirmed) {
            // значение из лейбла
            return du.patchersRevision.label;
         }

         // значение из формы
         return Number(du.patchersRevision.value);
      });

      for (const sidecarType of SidecarsForUpdating) {
         const sidecar = sidecarsUpdateConfig[sidecarType];

         patchObject(spec, sidecar.specPath, value => {
            patchNumber(value, 'revision', () => {
               let revision = du.sidecars[sidecarType].resourceRevision;
               const isConfirmed = this.visitor.isDuConfirmed(du.id, sidecar.confirmationType);

               if (isConfirmed) {
                  revision = du.sidecars[sidecarType].labelRevision;
               }

               return revision;
            });

            if (
               (value.hasOwnProperty('override') && Object.keys(value.override).length > 0) ||
               !isEmpty(du.sidecars[sidecarType].overrideLabels)
            ) {
               patchObject(value, 'override', () =>
                  du.sidecars[sidecarType].overrideLabels
                     .filter(v => !isEmpty(v.key.trim()) && !isEmpty(v.value.trim()))
                     .reduce((acc, item) => {
                        acc[item.key] = item.value;

                        return acc;
                     }, {} as Record<string, string>),
               );
            }

            return value;
         });
      }

      patchObject(spec, 'logbroker_config', logbrokerConfig => {
         this.patchLogbrokerConfig(du, logbrokerConfig);

         return logbrokerConfig;
      });

      patchBoolean(spec, 'enable_dynamic_resource_updater', enableDynamicResourceUpdater =>
         this.visitor.enableDynamicResourceUpdater
            ? // true, если есть динамические ресурсы
              du.boxes.flatMap(v => v.dynamicResources ?? []).some(v => !isEmpty(v))
            : enableDynamicResourceUpdater,
      );

      patchBoolean(spec, 'collect_portometrics_from_sidecars', () => du.collectPortometricsFromSidecars);

      patchObject(spec, 'infra_components', infraComponents => {
         patchBoolean(infraComponents, 'allow_automatic_updates', () => du.infraComponents.allowAutomaticUpdates);
         return infraComponents;
      });

      return spec;
   }
}

/**
 * В идеальном мире гарантии = лимитам, но есть некоторые не очень хорошие сервисы
 * Если человек намеренно указал себе limit больше гарантий, надо оставить как есть
 * Вроде корректно считать следующее - если человек потрогал это поле, то мы ставим лимиты
 */
function createLimitPatcher(oldGuarantee: number | undefined, newGuarantee: number | null, newLimit: number | null) {
   /*
    if (
            !(resourceRequests.vcpu_guarantee && resourceRequests.vcpu_guarantee === du.cpu) ||
            !resourceRequests.vcpu_limit
         ) {
            patchNumber(resourceRequests, 'vcpu_limit', () => du.cpu);
         }
    */
   const isExistAndUnchanged = oldGuarantee && oldGuarantee === newGuarantee;
   const isEmptyOrChanged = !isExistAndUnchanged;

   return (oldLimit: number) => (isEmptyOrChanged || !oldLimit ? newLimit : oldLimit);
}

function createVirtualDiskIdRef(disks: DeployUnitDisk[], oldDiskId: string | undefined, newDiskId: string): string {
   if (disks.length > 1) {
      return newDiskId;
   }

   return oldDiskId && oldDiskId === newDiskId ? newDiskId : '';
}
