/*
Превращает сырой статус в более удобный вид
*/

import {
   TDeployProgress,
   TDeployUnitApproval_TUserApproval_EUserApprovalStatus,
   TDeployUnitOverrides,
   TDeployUnitStatus,
   Timestamp,
   TRuntimeDeployControls,
} from '../../../../proto-typings';
import { getStatusStateInfo, ReplicaSetsInfo, StatusStateInfo } from '../../runtimeDeploy';
import { getDeployUnitReplicaSetsInfo } from '../../runtimeDeploy/ReplicaSet/ReplicaSetsInfo';
import { DeployUnit, PerLocationSettings } from './DeployUnit';
import { DeployUnitConverter } from './DeployUnitConverter';

// TODO: похожие статусы почти для всех объектов, полумать как переиспользовать
export enum DeployUnitState {
   Ready = 'ready',
   InProgress = 'progress',
   Failed = 'failed',
   Unknown = 'unknown',
}

/**
 * null отсутствие информации, 0 — значение поля
 */
interface PodsCount {
   total: number | null;
   ready: number | null;
   inProgress: number | null;
   failed: number | null;
}

export enum LocationApprovalStatus {
   Approved = TDeployUnitApproval_TUserApproval_EUserApprovalStatus.APPROVED,
   Disapproved = TDeployUnitApproval_TUserApproval_EUserApprovalStatus.DISAPPROVED,
}

export interface OverrideDeploySettingsMeta {
   login: string | null;
   time: Timestamp | null;
}

export interface LocationApprove {
   duRevision: number;
   status: LocationApprovalStatus;
   meta: OverrideDeploySettingsMeta;
}

export type LocationApproves = Map<string, LocationApprove>;

export interface DeployUnitOverridePerCluster {
   maxUnavailable: {
      value: number;
      meta: OverrideDeploySettingsMeta;
   };
}

export interface DeployUnitOverridesDeploySettings {
   perLocationSettings: PerLocationSettings;
   meta: OverrideDeploySettingsMeta;
}

export interface DeployUnitOverrides {
   perCluster: Record<string, DeployUnitOverridePerCluster>;
   revision: number;
   deploySettings: DeployUnitOverridesDeploySettings | null;
}

export interface DeployUnitStatus {
   id: string;
   targetRevision: number | null;
   podsCount: PodsCount;
   statusStateInfo: StatusStateInfo;
   statusState: DeployUnitState;
   currentTarget: DeployUnit | null;
   replicaSetsInfo: ReplicaSetsInfo | null;
   locations: Set<string>;

   locationApproves: LocationApproves;
   overrides: DeployUnitOverrides | null;
}

type GetDeployUnitStatusParams = {
   id: string;
   status: TDeployUnitStatus | null;
   controls: TRuntimeDeployControls | null;
};

function getDeployUnitStatus({ id, status, controls }: GetDeployUnitStatusParams): DeployUnitStatus {
   // const { spec, status, id } = rawDeployUnit;

   const replicaSetsInfo = status ? getDeployUnitReplicaSetsInfo(status) : null;

   const statusStateInfo = getStatusStateInfo(status);

   const result: DeployUnitStatus = {
      id,
      targetRevision: status?.target_revision ?? null,
      // TODO #DEPLOY-4961 пустой progress в деплой юните без подов
      podsCount: getPodsCount(status?.progress ?? null),
      statusStateInfo,
      statusState: getDuState(statusStateInfo),
      currentTarget: status?.current_target
         ? DeployUnitConverter.fromApi(
              { id, spec: status.current_target, status: undefined },
              undefined,
              undefined,
              undefined,
           )
         : null,
      replicaSetsInfo,
      locations: getDeployUnitLocations(replicaSetsInfo),
      overrides: controls?.deploy_unit_overrides ? getOverrides(controls.deploy_unit_overrides) : null,
      locationApproves: getLocationApproves(controls),
   };

   return result;
}

function getDeployUnitLocations(replicaSetsInfo: ReplicaSetsInfo | null): Set<string> {
   if (!replicaSetsInfo) {
      return new Set();
   }
   return new Set(Object.values(replicaSetsInfo).flatMap(info => Array.from(info.locations)));
}

function getPodsCount(progress: TDeployProgress | null): PodsCount {
   const count: PodsCount = {
      total: null,
      ready: null,
      inProgress: null,
      failed: null,
   };
   const { pods_total, pods_ready, pods_in_progress, pods_failed } = progress ?? {};
   count.total = pods_total ?? null;
   count.ready = pods_ready ?? null;
   count.inProgress = pods_in_progress ?? null;
   count.failed = pods_failed ?? null;
   return count;
}

// TODO: почти всегда порядок одинаков, возможно переиспользование логики
/**
 * Определяет единственный статус для вывода, порядок проверок полей важен
 */
function getDuState(statusStateInfo: StatusStateInfo): DeployUnitState {
   const { ready, inProgress, failed } = statusStateInfo;
   if (failed.active) {
      return DeployUnitState.Failed;
   }
   if (inProgress.active) {
      return DeployUnitState.InProgress;
   }
   if (ready.active) {
      return DeployUnitState.Ready;
   }
   return DeployUnitState.Unknown;
}

export interface DuStatusLabel {
   state: DeployUnitState;
   text: string;
   message: string | null;
}

const prettyDuReason = (text: string) => text.replaceAll('_', ' ').toLocaleLowerCase();
function getDuStatusLabel(deployUnit: DeployUnit | null, deployUnitStatus: DeployUnitStatus | null): DuStatusLabel {
   if (!deployUnitStatus) {
      return {
         state: DeployUnitState.Unknown,
         text: '',
         message: null,
      };
   }
   // обратный вариант возможно потребует статуса old???
   const { revision } = deployUnit ?? {};
   const { targetRevision, statusState, statusStateInfo, podsCount } = deployUnitStatus;
   const { ready, inProgress, failed } = statusStateInfo;
   const { total = 0, inProgress: inProgressPods = 0 } = podsCount;
   const noProgressPods = inProgressPods === 0;
   const noPods = total === 0;
   const isStatusReady = statusState === DeployUnitState.Ready;
   const isReady = isStatusReady && targetRevision === revision;

   let text: string;
   let message: string | null = null;
   if (isReady) {
      // ready с последней ревизией
      return {
         state: DeployUnitState.Ready,
         text: ready.reason ?? 'ready',
         message: ready.message,
      };
   }

   // прогресс, если ревизии не совпадают
   const state: DeployUnitState = isStatusReady ? DeployUnitState.InProgress : statusState;
   if (state === DeployUnitState.InProgress) {
      const defaultProgressText = noProgressPods && noPods ? 'no pods in progress' : 'in progress';
      if (isStatusReady) {
         // для искуственно выставленного прогресса сообщение по дефолту
         text = defaultProgressText;
      } else {
         // для честного прогресса забираем данные
         text = inProgress.reason ? prettyDuReason(inProgress.reason) : defaultProgressText;
         message = inProgress.message;
      }
   } else {
      // остался ошибочный статус
      text = failed.reason ? prettyDuReason(failed.reason) : 'failed';
      message = failed.message;
   }

   return { state, text, message };
}

function getOverrides(rawOverrides: TDeployUnitOverrides): DeployUnitOverrides {
   // per_cluster_overrides может отсутствовать
   const { revision_to_override, per_cluster_overrides = {}, deploy_settings_override } = rawOverrides;

   let deploySettings: DeployUnitOverridesDeploySettings | null = null;
   if (deploy_settings_override?.deploy_settings) {
      const rawSettings = deploy_settings_override.deploy_settings;
      const clusterList = rawSettings.cluster_sequence ?? [];
      deploySettings = {
         perLocationSettings: {
            isCustom: DeployUnitConverter.isCustomPerLocationSettings(rawSettings),
            strategy: DeployUnitConverter.getPerLocationStrategy(rawSettings.deploy_strategy),
            locationOrder: clusterList.map(e => e.yp_cluster),
            needApproval: new Set(clusterList.filter(e => e.need_approval).map(e => e.yp_cluster)),
         },
         meta: {
            login: deploy_settings_override.author,
            time: (deploy_settings_override.approval_time ?? null) as Timestamp | null,
         },
      };
   }

   return {
      revision: revision_to_override,
      perCluster: Object.keys(per_cluster_overrides).reduce((data, location) => {
         const { max_unavailable } = per_cluster_overrides[location];
         if (max_unavailable) {
            const { value, author } = max_unavailable;

            data[location] = {
               maxUnavailable: { value, meta: { login: author, time: null } },
            };
         }
         return data;
      }, {} as DeployUnitOverrides['perCluster']),
      deploySettings,
   };
}

function getLocationApproves(runtimeDeployControls: TRuntimeDeployControls | null): LocationApproves {
   if (!runtimeDeployControls) {
      return new Map();
   }

   const approves: LocationApproves = new Map();
   const duApproves = runtimeDeployControls.deploy_unit_approvals ?? {};
   const approveClusters = Object.keys(duApproves).filter(e => {
      const { user, payload } = duApproves[e];
      return user && payload;
   });

   for (const duCluster of approveClusters) {
      const { user, payload } = duApproves[duCluster];
      const { cluster, revision } = payload!;
      const { login, status, approval_time } = user!;

      approves.set(cluster, {
         duRevision: revision,
         status:
            status === TDeployUnitApproval_TUserApproval_EUserApprovalStatus.APPROVED
               ? LocationApprovalStatus.Approved
               : LocationApprovalStatus.Disapproved,
         meta: {
            login,
            time: approval_time as Timestamp,
         },
      });
   }

   return approves;
}

export const DeployUnitStatusConverter = {
   getDeployUnitStatus,
   getDuStatusLabel,
};
