import { EAccessControlAction, EAccessControlPermission, TStage } from '../../../../proto-typings';

import { Entity } from '../../../../redux/models';
import { DeepPartial } from '../../../typeHelpers';
import { ProjectItem } from '../../Project';
// noinspection ES6PreferShortImport
import { DeployUnit, getEmptyDeployUnit, ResourceType } from '../DeployUnit/DeployUnit';
import { StageEmptyParams } from '../stage-levels';

export interface SandboxMappingTarget {
   deployUnitId: string;
   id: string;
   type: ResourceType;
}

export interface DockerMappingTarget {
   boxId: string;
   deployUnitId: string;
}

export interface InfraRecord {
   id: number | null;
   name: string | null;
}

export interface Infra {
   service: InfraRecord;
   environment: InfraRecord;
}

export interface Stage extends Entity {
   deployUnits: DeployUnit[];
   dockerTargets: DockerMappingTarget[];
   infra: Infra;
   project: ProjectItem | null;
   soxService: boolean;
   sandboxTargets: SandboxMappingTarget[];
   labels: TStage['labels'];
   revision: number | undefined;
   revisionDescription: string | null;

   /**
    * ID данное стейджу при заведении.
    * В отличие от id - не редактируется и не меняется, вплоть до сохранения.
    */
   initialId?: string;
}

export const getEmptyStage = (params?: StageEmptyParams): Stage => ({
   id: 'stage',
   deployUnits: [getEmptyDeployUnit(params)],
   infra: {
      service: { id: null, name: null },
      environment: { id: null, name: null },
   },
   project: null, // TODO: добавить в EmptyStageParams accountId и заполнять проект
   soxService: false,
   dockerTargets: [],
   sandboxTargets: [],
   labels: {},
   revision: 0,
   revisionDescription: null,
});

export const emptyRawStage: DeepPartial<TStage> = {
   meta: {
      id: '',
      project_id: '',
      acl: [
         {
            action: EAccessControlAction.ACA_ALLOW,
            permissions: [
               EAccessControlPermission.ACP_READ,
               EAccessControlPermission.ACP_READ_SECRETS,
               EAccessControlPermission.ACA_WRITE,
               EAccessControlPermission.ACA_CREATE,
            ],
            subjects: [...(window.CONFIG.deployRobots ?? [])],
         },
      ],
   },
   spec: {
      sox_service: false,
      deploy_units: {},
   },
   labels: {
      deploy_engine: window.CONFIG.deployEngine || 'env_controller',
   },
};

export function getDisabledStageClusters(stage: Stage | null): Set<string> {
   return new Set(stage?.labels?.deploy?.disabled_clusters ?? []);
}
