import { IJsonable } from '@yandex-infracloud-ui/libs';

import { TStage } from '../../../../proto-typings';
import { ProjectItem } from '../../Project';

import { DeployUnit, DeployUnitConverter, DeployUnitRawRecord } from '../DeployUnit';
import { DockerMappingTarget, Infra, SandboxMappingTarget, Stage } from './Stage';

export class StageConverter implements Stage, IJsonable {
   public static fromApi(raw: TStage): Stage {
      return new StageConverter(raw).toJSON();
   }

   public deployUnits: DeployUnit[];

   public dockerTargets: DockerMappingTarget[];

   public id: string;

   public project: ProjectItem | null;

   public infra: Infra;

   public labels: TStage['labels'];

   public soxService: boolean;

   public sandboxTargets: SandboxMappingTarget[];

   public revision: number | undefined;

   public revisionDescription: string | null;

   constructor(private raw: TStage) {
      this.id = this.raw.meta!.id;
      this.project = {
         id: this.raw.meta?.project_id ?? '',
         accountId: this.raw.spec?.account_id ?? '',
      };

      this.revision = this.raw.spec?.revision;
      this.revisionDescription = this.raw.spec?.revision_info?.description ?? null;

      this.infra = {
         service: {
            id: this.raw.labels?.infra_service ?? null,
            name: this.raw.labels?.infra_service_name ?? null,
         },
         environment: {
            id: this.raw.labels?.infra_environment ?? null,
            name: this.raw.labels?.infra_environment_name ?? null,
         },
      };

      this.labels = this.raw.labels ?? {};

      this.deployUnits = this.getDeployUnits(this.labels);

      // исправить после DEPLOY-3110
      this.soxService = Object.values(this.deployUnits).some(e => e.soxService);

      this.sandboxTargets = this.getSandboxTargets();
      this.dockerTargets = this.getDockerTargets();
   }

   public toJSON(): Stage {
      return {
         id: this.id,
         project: this.project,
         sandboxTargets: this.sandboxTargets,
         dockerTargets: this.dockerTargets,
         deployUnits: this.deployUnits,
         soxService: this.soxService,
         infra: this.infra,
         labels: this.labels,
         revision: this.revision,
         revisionDescription: this.revisionDescription,
      };
   }

   private getDeployUnits(labels: TStage['labels']): DeployUnit[] {
      return this.iterateDeployUnits().map(du =>
         DeployUnitConverter.fromApi(du, labels, this.raw.spec?.dynamic_resources, this.raw.spec?.deploy_unit_settings),
      );
   }

   private getDockerTargets(): DockerMappingTarget[] {
      const result: DockerMappingTarget[] = [];

      for (const { id: deployUnitId, spec } of this.iterateDeployUnits()) {
         for (const boxId of Object.keys(spec.images_for_boxes ?? {})) {
            result.push({ boxId, deployUnitId });
         }
      }

      return result;
   }

   private getSandboxTargets(): SandboxMappingTarget[] {
      return this.deployUnits.reduce((acc, du) => {
         for (const resource of du.resources) {
            acc.push({ id: resource.id, type: resource.type, deployUnitId: du.id });
         }

         return acc;
      }, [] as SandboxMappingTarget[]);
   }

   private iterateDeployUnits(): DeployUnitRawRecord[] {
      return Object.keys(this.raw.spec?.deploy_units ?? {}).map(duId => ({
         id: duId,
         spec: this.raw.spec!.deploy_units![duId]!,
         status: this.raw.status?.deploy_units?.[duId],
         runtimeDeployControls: this.raw.status?.runtime_deploy_controls?.[duId],
      }));
   }
}
