import { array, boolean, object, string } from 'yup';
import { getPaths } from '../../../../utils';
import { ProjectItem } from '../../Project';

import { IntegerPositiveNullableSchema, projectIdSchema, stageIdSchema, StringNullableSchema } from '../validation';
import { getEmptyStage, Infra, Stage } from './Stage';

export type StageFormParams = Pick<Stage, 'id' | 'project'> & {
   soxService: boolean;
   tags: string[];
   infra: Infra;
   initialId?: string;
};

export const stagePaths = getPaths<StageFormParams>();

export const infraValidationSchema = object({
   service: object({
      id: IntegerPositiveNullableSchema('Service ID'),
      name: StringNullableSchema('Service Name'),
   }),
   environment: object({
      id: IntegerPositiveNullableSchema('Environment ID'),
      name: StringNullableSchema('Environment Name'),
   }),
});

export const stageValidationSchema = object<StageFormParams>({
   id: stageIdSchema,
   project: object<ProjectItem>({
      id: projectIdSchema.required(),
      accountId: string(),
   })
      .label('Project')
      .nullable()
      .required(),
   soxService: boolean(),
   tags: array().of(string()),
   infra: infraValidationSchema,
});

export function stageToFormParams(item = getEmptyStage()): StageFormParams {
   return {
      id: item.id,
      project: item.project,
      soxService: item.soxService,
      tags: item.labels?.tags ?? [],
      infra: item.infra,
   };
}

export function formParamsToStage(params: StageFormParams, old: Stage): Stage {
   return {
      ...old,
      deployUnits: [], // clear children
      id: params.id,
      project: params.project,
      soxService: params.soxService,
      labels: {
         ...old.labels,
         tags: params.tags,
      },
      infra: params.infra,
      initialId: params.initialId,
   };
}
