import { deepClone } from '@yandex-infracloud-ui/libs';

import {
   EObjectType,
   TDeployUnitSpec,
   TPodTemplateSpec,
   TStage,
   TStageSpec,
   TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment,
   TStageSpec_TDeployUnitSettings_TAlerting_EState,
} from '../../../../proto-typings';

import { DeepPartial } from '../../../typeHelpers';

import { DynamicResourceNotifyPolicyMode } from '../Box';

import { StagePatcherVisitor } from '../StagePatcherVisitor';
import { StageConverter } from './StageConverter';
import { StagePatcher } from './StagePatcher';

const boxId = 'box';

const getInitialPodTemplateSpec = (): DeepPartial<TPodTemplateSpec> => ({
   spec: {
      pod_agent_payload: {
         spec: {
            boxes: [
               {
                  id: boxId,
                  // ...
               },
            ],
            // ...
         },
      },
   },
});

const getPerClusterDeployUnitSpec = (): DeepPartial<TDeployUnitSpec> => ({
   replica_set: {
      replica_set_template: {
         pod_template_spec: getInitialPodTemplateSpec(),
      },
   },
});

const getMultiClusterDeployUnitSpec = (): DeepPartial<TDeployUnitSpec> => ({
   multi_cluster_replica_set: {
      replica_set: {
         pod_template_spec: getInitialPodTemplateSpec(),
      },
   },
});

const DYNAMIC_RESOURCES = {
   customDeployGroups: (revision?: number) => ({
      deploy_unit_ref: 'du1',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: boxId,
                  destination: '/destination/customDeployGroups1',
                  storage_dir: '/storage/customDeployGroups1',
                  cached_revisions_count: 3,
               },
               urls: ['sbr:1234567890'],
            },
            {
               mark: 'all',
               storage_options: {
                  box_ref: 'Box-123',
                  destination: '/destination/customDeployGroups2',
                  storage_dir: '/storage/customDeployGroups2',
                  cached_revisions_count: 3,
                  verification: {
                     check_period_ms: 9,
                     checksum: 'checksum',
                  },
               },
               urls: ['sbr:1234567890'],
            },
         ],
         revision,
         update_window: 2,
      },
   }),

   customRequiredLabels: (revision?: number) => ({
      deploy_unit_ref: 'du2',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: boxId,
                  destination: '/destination/customRequiredLabels',
                  storage_dir: '/storage/customRequiredLabels',
                  cached_revisions_count: 3,
               },
               urls: ['sbr:1234567890'],
               required_labels: {
                  label1: 'label1',
                  label2: 'label2',
               },
            },
         ],
         revision,
         update_window: 2,
      },
   }),

   httpAction: (revision?: number) => ({
      deploy_unit_ref: 'du1',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: boxId,
                  destination: '/destination/http',
                  storage_dir: '/storage/http',
                  cached_revisions_count: 3,
                  http_action: {
                     url: 'http://url...',
                     expected_answer: 'http answer',
                  },
               },
               urls: ['sbr:1234567890'],
            },
         ],
         revision,
         update_window: 2,
      },
   }),

   execAction: (revision?: number) => ({
      deploy_unit_ref: 'du2',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: boxId,
                  destination: '/destination/exec',
                  storage_dir: '/storage/exec',
                  cached_revisions_count: 3,
                  exec_action: {
                     command_line: 'exec command',
                     expected_answer: 'exec answer',
                  },
               },
               urls: ['sbr:1234567890'],
            },
         ],
         revision,
         update_window: 2,
      },
   }),

   disabledAction: (revision?: number) => ({
      deploy_unit_ref: 'du2',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: boxId,
                  destination: '/destination/disabled',
                  storage_dir: '/storage/disabled',
                  cached_revisions_count: 3,
                  allow_deduplication: true,
                  max_download_speed: 100,
               },
               urls: ['sbr:1234567890', 'rbtorrent:abc1234567890...'],
            },
         ],
         revision,
         update_window: 2,
      },
   }),
};

const getCustomDynamicResourcesSpec = (revision?: number): DeepPartial<TStageSpec['dynamic_resources']> => ({
   customDeployGroups: DYNAMIC_RESOURCES.customDeployGroups(revision),
   customRequiredLabels: DYNAMIC_RESOURCES.customRequiredLabels(revision),
   httpAction: DYNAMIC_RESOURCES.httpAction(revision),
   execAction: DYNAMIC_RESOURCES.execAction(revision),
   disabledAction: DYNAMIC_RESOURCES.disabledAction(revision),
});

const getDynamicResourcesSpec = (revision?: number): DeepPartial<TStageSpec['dynamic_resources']> => ({
   httpAction: DYNAMIC_RESOURCES.httpAction(revision),
   execAction: DYNAMIC_RESOURCES.execAction(revision),
   disabledAction: DYNAMIC_RESOURCES.disabledAction(revision),
});

describe('models/ui|StagePatcher', () => {
   const rawStage: TStage = ({
      annotations: {},
      control: undefined,
      labels: {},
      meta: {
         account_id: 'abc:service:3494',
         acl: [],
         creation_time: new Date().getTime(),
         fqid: '',
         id: 'old_stage',
         inherit_acl: false,
         name: 'old_stage',
         project_id: 'deploy-ui',
         type: EObjectType.OT_STAGE,
         uuid: 'test-uuid',
      },
      status: undefined,
      spec: {
         account_id: 'abc:service:3494',
         deploy_units: {
            du1: getPerClusterDeployUnitSpec(),
            du2: getMultiClusterDeployUnitSpec(),
         },
         env: {},
         dynamic_resources: {},
         imp_revision: 0,
         resource_caches: {},
         revision: 1,
         revision_info: { description: 'release notes' },
         sox_service: false,
      },
   } as DeepPartial<TStage>) as TStage;

   const visitor = new StagePatcherVisitor();

   describe('Project', () => {
      it('should keep account_id', () => {
         const stage = StageConverter.fromApi(rawStage);

         expect(stage.project).toEqual({
            id: 'deploy-ui',
            accountId: 'abc:service:3494',
         });

         stage.project!.id = 'deploy-ui-new';

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected = deepClone(rawStage);

         expected!.meta!.project_id = 'deploy-ui-new';
         expected!.meta!.account_id = 'abc:service:3494';
         expected!.spec!.account_id = 'abc:service:3494';

         expect(merged).toEqual(expected);
      });

      it('should remove account_id: abc', () => {
         const stage = StageConverter.fromApi(rawStage);

         stage.project = {
            id: 'deploy-ui-new',
            accountId: 'abc:service:1234567890',
         };

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected: DeepPartial<TStage> = deepClone(rawStage);

         expected!.meta!.project_id = 'deploy-ui-new';
         expected!.meta!.account_id = undefined;
         expected!.spec!.account_id = undefined;

         expect(merged).toEqual(expected);
      });

      it('should remove account_id: tmp', () => {
         const stage = StageConverter.fromApi(rawStage);

         stage.project = {
            id: 'deploy-ui-new',
            accountId: 'tmp',
         };

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected: DeepPartial<TStage> = deepClone(rawStage);

         expected!.meta!.project_id = 'deploy-ui-new';
         expected!.meta!.account_id = undefined;
         expected!.spec!.account_id = undefined;

         expect(merged).toEqual(expected);
      });
   });

   describe('Labels: infra', () => {
      it('should add infra labels', () => {
         const stage = StageConverter.fromApi(rawStage);

         stage.infra = {
            service: { id: 687, name: 'Y.Deploy' },
            environment: { id: 990, name: 'Testing' },
         };

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected = deepClone(rawStage);

         expected!.labels!.infra_service = 687;
         expected!.labels!.infra_service_name = 'Y.Deploy';
         expected!.labels!.infra_environment = 990;
         expected!.labels!.infra_environment_name = 'Testing';

         expect(merged).toEqual(expected);
      });

      it('should edit infra labels', () => {
         const stage = StageConverter.fromApi({
            ...rawStage,
            labels: {
               infra_service: 687,
               infra_service_name: 'Y.Deploy',
               infra_environment: 990,
               infra_environment_name: 'Testing',
            },
         });

         expect(stage.infra.service.id).toEqual(687);
         expect(stage.infra.service.name).toEqual('Y.Deploy');
         expect(stage.infra.environment.id).toEqual(990);
         expect(stage.infra.environment.name).toEqual('Testing');

         stage.infra.service.id = 2178;
         stage.infra.service.name = 'Infracloudui';
         stage.infra.environment.id = 3501;
         stage.infra.environment.name = 'Beta';

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected = deepClone(rawStage);

         expected!.labels!.infra_service = 2178;
         expected!.labels!.infra_service_name = 'Infracloudui';
         expected!.labels!.infra_environment = 3501;
         expected!.labels!.infra_environment_name = 'Beta';

         expect(merged).toEqual(expected);
      });

      it('should clear infra labels', () => {
         const stage = StageConverter.fromApi({
            ...rawStage,
            labels: {
               infra_service: 687,
               infra_service_name: 'Y.Deploy',
               infra_environment: 990,
               infra_environment_name: 'Testing',
            },
         });

         expect(stage.infra.service.id).toEqual(687);
         expect(stage.infra.service.name).toEqual('Y.Deploy');
         expect(stage.infra.environment.id).toEqual(990);
         expect(stage.infra.environment.name).toEqual('Testing');

         stage.infra.service.id = null;
         stage.infra.service.name = null;
         stage.infra.environment.id = null;
         stage.infra.environment.name = null;

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected = deepClone(rawStage);

         expected!.labels!.infra_service = undefined;
         expected!.labels!.infra_service_name = undefined;
         expected!.labels!.infra_environment = undefined;
         expected!.labels!.infra_environment_name = undefined;

         expect(merged).toEqual(expected);
      });
   });

   describe('Deploy unit settings', () => {
      it('should patch alerting', () => {
         const rawStageWithAlerting = deepClone({
            ...rawStage,
            spec: {
               ...rawStage.spec,
               deploy_unit_settings: {
                  'du1': {
                     alerting: {
                        state: TStageSpec_TDeployUnitSettings_TAlerting_EState.ENABLED,
                        notification_channels: { 'ERROR': 'channel1' },
                     },
                     environment: TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.STABLE,
                  },
               },
            },
         } as TStage);

         const stage = StageConverter.fromApi(rawStageWithAlerting);

         const formDu1 = stage.deployUnits.find(du => du.id === 'du1');

         expect(formDu1!.settings.alerting).toEqual({
            state: true,
            notificationChannel: 'channel1',
         });

         const formDu2 = stage.deployUnits.find(du => du.id === 'du2');

         formDu1!.settings.alerting = {
            state: false,
            notificationChannel: '',
         };

         formDu2!.settings.alerting = {
            state: true,
            notificationChannel: 'channel2',
         };

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected = deepClone(rawStage);

         expected!.spec!.deploy_unit_settings = {
            'du1': {
               environment: TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.STABLE,
               alerting: undefined,
            },
            'du2': {
               environment: undefined,
               alerting: {
                  state: TStageSpec_TDeployUnitSettings_TAlerting_EState.ENABLED,
                  notification_channels: { 'ERROR': 'channel2' },
               },
            },
         };

         expect(merged).toEqual(expected);
      });
   });

   describe('Dynamic resources', () => {
      it('should patch dynamic_resources: add', () => {
         const stage = StageConverter.fromApi(rawStage);

         const formDu1 = stage.deployUnits.find(du => du.id === 'du1');
         const formDu2 = stage.deployUnits.find(du => du.id === 'du2');

         formDu1!.boxes[0].dynamicResources = [
            {
               id: 'httpAction',
               urls: ['sbr:1234567890'],
               destination: '/destination/http',
               storageDir: '/storage/http',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Http,
                  httpAction: {
                     url: 'http://url...',
                     expectedAnswer: 'http answer',
                  },
               },

               advancedSettings: {
                  allowDeduplication: false,
                  maxDownloadSpeed: null,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: false,
               },
            },
         ];

         formDu2!.boxes[0].dynamicResources = [
            {
               id: 'execAction',
               urls: ['sbr:1234567890'],
               destination: '/destination/exec',
               storageDir: '/storage/exec',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Exec,
                  execAction: {
                     commandLine: 'exec command',
                     expectedAnswer: 'exec answer',
                  },
               },

               advancedSettings: {
                  allowDeduplication: false,
                  maxDownloadSpeed: null,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: false,
               },
            },
            {
               id: 'disabledAction',
               urls: ['sbr:1234567890', 'rbtorrent:abc1234567890...'],
               destination: '/destination/disabled',
               storageDir: '/storage/disabled',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Disabled,
               },

               advancedSettings: {
                  allowDeduplication: true,
                  maxDownloadSpeed: 100,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: false,
               },
            },
         ];

         const merged = StagePatcher.patch(visitor, rawStage, stage);

         const expected = deepClone(rawStage);

         expected!.spec!.deploy_units.du1.enable_dynamic_resource_updater = true;
         expected!.spec!.deploy_units.du2.enable_dynamic_resource_updater = true;

         expected!.spec!.dynamic_resources = getDynamicResourcesSpec() as TStageSpec['dynamic_resources'];

         expect(merged).toEqual(expected);
      });

      it('should patch dynamic_resources: clear', () => {
         const rawStageWithDynamicResources = deepClone({
            ...rawStage,
            spec: {
               ...rawStage.spec,
               dynamic_resources: getCustomDynamicResourcesSpec(99),
            },
         } as TStage);

         const stage = StageConverter.fromApi(rawStageWithDynamicResources);

         const formDu1 = stage.deployUnits.find(du => du.id === 'du1');
         const formDu2 = stage.deployUnits.find(du => du.id === 'du2');

         formDu1!.boxes[0].dynamicResources = [];
         formDu2!.boxes[0].dynamicResources = [];

         const merged = StagePatcher.patch(visitor, rawStageWithDynamicResources, stage);

         const expected = deepClone(rawStageWithDynamicResources);

         expected!.spec!.dynamic_resources = {} as TStageSpec['dynamic_resources'];

         expect(merged).toEqual(expected);
      });

      it('should patch dynamic_resources: edit', () => {
         const rawStageWithDynamicResources = deepClone({
            ...rawStage,
            spec: {
               ...rawStage.spec,
               dynamic_resources: getCustomDynamicResourcesSpec(99),
            },
         } as TStage);

         const stage = StageConverter.fromApi(rawStageWithDynamicResources);

         const formDu1 = stage.deployUnits.find(du => du.id === 'du1');
         const formDu2 = stage.deployUnits.find(du => du.id === 'du2');

         formDu1!.boxes[0].dynamicResources = [
            {
               // ресурс с кастомными настройками
               // патчер не должен его трогать
               id: 'customDeployGroups',
               initialId: 'customDeployGroups',
               urls: ['sbr:1234567890'],
               destination: '/destination/customDeployGroups/new',
               storageDir: '/storage/customDeployGroups/new',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Disabled,
               },

               advancedSettings: {
                  allowDeduplication: false,
                  maxDownloadSpeed: 100,
                  verification: {
                     checkPeriodMs: 9,
                     checksum: 'checksum',
                  },
               },

               customSettings: {
                  deployGroups: true,
                  requiredLabels: false,
               },
            },
            {
               id: 'httpAction',
               initialId: 'httpAction',
               urls: ['sbr:1234567890'],
               destination: '/destination/http/new',
               storageDir: '/storage/http/new',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Http,
                  httpAction: {
                     url: 'http://url...',
                     expectedAnswer: 'http answer',
                  },
               },

               advancedSettings: {
                  allowDeduplication: false,
                  maxDownloadSpeed: null,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: false,
               },
            },
         ];

         formDu2!.boxes[0].dynamicResources = [
            {
               // ресурс с кастомными настройками
               // патчер не должен его трогать
               id: 'customRequiredLabels',
               initialId: 'customRequiredLabels',
               urls: ['sbr:1234567890'],
               destination: '/destination/customRequiredLabels/new',
               storageDir: '/storage/customRequiredLabels/new',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Disabled,
               },

               advancedSettings: {
                  allowDeduplication: false,
                  maxDownloadSpeed: null,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: true,
               },
            },
            {
               id: 'execAction',
               initialId: 'execAction',
               urls: ['sbr:1234567890'],
               destination: '/destination/exec/new',
               storageDir: '/storage/exec/new',
               updateWindow: 2,
               cachedRevisionsCount: 3,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Exec,
                  execAction: {
                     commandLine: 'exec_new command',
                     expectedAnswer: 'exec_new answer',
                  },
               },

               advancedSettings: {
                  allowDeduplication: false,
                  maxDownloadSpeed: null,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: false,
               },
            },
            {
               id: 'disabledAction',
               initialId: 'disabledAction',
               urls: ['rbtorrent:abc1234567890...'],
               destination: '/destination/disabled/new',
               storageDir: '/storage/disabled/new',
               updateWindow: 20,
               cachedRevisionsCount: 30,

               notifyPolicy: {
                  mode: DynamicResourceNotifyPolicyMode.Disabled,
               },

               advancedSettings: {
                  allowDeduplication: true,
                  maxDownloadSpeed: 15,
                  verification: {
                     checkPeriodMs: null,
                     checksum: null,
                  },
               },

               customSettings: {
                  deployGroups: false,
                  requiredLabels: false,
               },
            },
         ];

         const merged = StagePatcher.patch(visitor, rawStageWithDynamicResources, stage);

         const expected = deepClone(rawStageWithDynamicResources);

         expected!.spec!.deploy_units.du1.enable_dynamic_resource_updater = true;
         expected!.spec!.deploy_units.du2.enable_dynamic_resource_updater = true;

         expected!.spec!.dynamic_resources = ({
            customDeployGroups: DYNAMIC_RESOURCES.customDeployGroups(99),

            customRequiredLabels: DYNAMIC_RESOURCES.customRequiredLabels(99),

            httpAction: {
               deploy_unit_ref: 'du1',
               dynamic_resource: {
                  deploy_groups: [
                     {
                        mark: 'all',
                        storage_options: {
                           box_ref: boxId,
                           cached_revisions_count: 3,
                           destination: '/destination/http/new',
                           storage_dir: '/storage/http/new',
                           http_action: {
                              url: 'http://url...',
                              expected_answer: 'http answer',
                           },
                        },
                        urls: ['sbr:1234567890'],
                     },
                  ],
                  revision: 100,
                  update_window: 2,
               },
            },

            execAction: {
               deploy_unit_ref: 'du2',
               dynamic_resource: {
                  deploy_groups: [
                     {
                        mark: 'all',
                        storage_options: {
                           box_ref: boxId,
                           cached_revisions_count: 3,
                           destination: '/destination/exec/new',
                           storage_dir: '/storage/exec/new',
                           exec_action: {
                              command_line: 'exec_new command',
                              expected_answer: 'exec_new answer',
                           },
                        },
                        urls: ['sbr:1234567890'],
                     },
                  ],
                  revision: 100,
                  update_window: 2,
               },
            },

            disabledAction: {
               deploy_unit_ref: 'du2',
               dynamic_resource: {
                  deploy_groups: [
                     {
                        mark: 'all',
                        storage_options: {
                           box_ref: boxId,
                           cached_revisions_count: 30,
                           destination: '/destination/disabled/new',
                           storage_dir: '/storage/disabled/new',
                           allow_deduplication: true,
                           max_download_speed: 15,
                        },
                        urls: ['rbtorrent:abc1234567890...'],
                     },
                  ],
                  revision: 100,
                  update_window: 20,
               },
            },
         } as unknown) as TStageSpec['dynamic_resources'];

         expect(merged).toEqual(expected);
      });
   });

   it('should update stage.id', () => {
      const stage = StageConverter.fromApi(rawStage);
      stage.id = 'new_stage';

      const merged = StagePatcher.patch(visitor, rawStage, stage);

      const expected = deepClone(rawStage);
      Object.assign(expected.meta, {
         id: 'new_stage',
      });

      expect(merged).toEqual(expected);
   });

   describe('StagePatcher.updateRevision', () => {
      it('should increment revision number', () => {
         const stage: TStage = ({
            spec: {
               revision: 33,
            },
         } as DeepPartial<TStage>) as any;

         StagePatcher.updateRevision(stage, '', true);

         expect(stage.spec?.revision).toBe(34);
      });

      it('should increment empty revision number', () => {
         const stage: TStage = ({
            spec: {},
         } as DeepPartial<TStage>) as any;

         StagePatcher.updateRevision(stage, '', true);

         expect(stage.spec?.revision).toBe(1);
      });

      it('should update description', () => {
         const stage: TStage = ({
            spec: {
               revision_info: {
                  description: 'old',
               },
            },
         } as DeepPartial<TStage>) as any;

         StagePatcher.updateRevision(stage, 'new', true);

         expect(stage.spec?.revision_info?.description).toBe('new');
      });

      it('should add description', () => {
         const stage: TStage = ({
            spec: {
               revision_info: {},
            },
         } as DeepPartial<TStage>) as any;

         StagePatcher.updateRevision(stage, 'new', true);

         expect(stage.spec?.revision_info?.description).toBe('new');
      });

      it('should remove empty description', () => {
         const stage: TStage = ({
            spec: {
               revision_info: {
                  description: 'old',
               },
            },
         } as DeepPartial<TStage>) as any;

         StagePatcher.updateRevision(stage, '', true);

         expect(stage.spec?.revision_info).toEqual({});
      });

      it('should not increment', () => {
         const stage: TStage = ({
            spec: {
               revision: 10,
               revision_info: {},
            },
         } as DeepPartial<TStage>) as any;

         StagePatcher.updateRevision(stage, 'new', false);

         expect(stage.spec?.revision_info?.description).toBe('new');
         expect(stage.spec?.revision).toBe(10);
      });
   });
});
