import { deepClone, getSetDifference, isEmpty, isEqual, omitFields } from '@yandex-infracloud-ui/libs';

import {
   TDynamicResourceSpec_DeployGroup,
   TStage,
   TStageSpec,
   TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment,
   TStageSpec_TDeployUnitSettings_TAlerting_EState,
   TStageSpec_TStageDynamicResourceSpec,
} from '../../../../proto-typings';

import { ConfirmationType } from '../../Confirmations';
import { patchBoolean, patchList, patchNumber, patchObject, patchString, skipEmpty } from '../../utils';
import { DynamicResourceNotifyPolicyMode } from '../Box';

import { DeployUnit, DeployUnitPatcher, EnvironmentSettings } from '../DeployUnit';
import { StageParentNodeIds } from '../StageParentNodeIds';

import { StagePatcherVisitor } from '../StagePatcherVisitor';
import { Stage } from './Stage';

interface PrepareToSaveParams {
   // если true, то патчинг пропускается, spec берется как есть
   asIs: boolean;

   // комментарий к релизу
   description: string;

   // отредактированное значение (модель UI)
   editedValue: Stage;

   // если true, то версия ревизии будет инкрементирована
   increment: boolean;

   // текущая (последняя) версия стейджа (сырой стейдж). Используется только если asIs === true.
   latestRawStage?: TStage;

   // сырой стейдж, которые редактируется (может не быть последним в случае отката)
   rawStage: TStage;

   // настройки патчинга
   visitor: StagePatcherVisitor;
}

export class StagePatcher {
   /**
    * Патчит (мутирует) сырой стейдж свежими данными из модели UI
    *
    * Избегайте прямой вызов метода, предпочтительнее использовать prepareToSave в коде
    * Оставлен доступным для тестирования
    */
   public static patch(visitor: StagePatcherVisitor, raw: TStage, stage: Stage): TStage {
      return new StagePatcher(visitor, raw, stage).toValue();
   }

   /**
    * Обертка над patch, решающая что и как патчить
    *
    * Возвращает полностью готовый к сохранению стейдж. Обновляет ревизию.
    */
   public static prepareToSave({
      asIs,
      description,
      editedValue,
      increment,
      latestRawStage,
      rawStage,
      visitor,
   }: PrepareToSaveParams) {
      let newStage: TStage;
      if (asIs) {
         newStage = deepClone(
            latestRawStage
               ? {
                    ...latestRawStage,
                    spec: rawStage.spec, // При откате "as is" откатываем только spec
                 }
               : rawStage,
         );
      } else {
         newStage = StagePatcher.patch(visitor, rawStage, editedValue);
      }

      StagePatcher.updateRevision(newStage, description, increment);

      return newStage;
   }

   /**
    * Удаляет из сырого стейджа части, которые мы не меняем из UI (чтобы не плодить мусор в diff)
    */
   public static removeUnsavedParts(raw: TStage): Omit<TStage, 'status' | 'annotations'> {
      return omitFields(raw, 'status', 'annotations') as Omit<TStage, 'status' | 'annotations'>;
   }

   public static updateRevision(raw: TStage, description: string, increment: boolean): TStage {
      patchObject(raw, 'spec', spec => {
         if (increment) {
            patchNumber(spec, 'revision', r => r + 1);
         }

         patchObject(spec, 'revision_info', revisionInfo => {
            patchString(revisionInfo, 'description', () => skipEmpty(description));

            return revisionInfo;
         });

         return spec;
      });

      return raw;
   }

   constructor(private visitor: StagePatcherVisitor, private raw: TStage, private stage: Stage) {}

   private patchDynamicResources(deployUnits: DeployUnit[], specDynamicResources: TStageSpec['dynamic_resources']) {
      const dynamicResourceIdList: string[] = [];
      const oldSpecDynamicResources = deepClone(specDynamicResources);

      for (const du of deployUnits) {
         for (const box of du.boxes) {
            for (const dynamicResource of box.dynamicResources) {
               const customDeployGroups = dynamicResource.customSettings?.deployGroups ?? false;
               const customRequiredLabels = dynamicResource.customSettings?.requiredLabels ?? false;

               if (!customDeployGroups && !customRequiredLabels) {
                  patchObject(specDynamicResources, dynamicResource.id, () => {
                     const exist =
                        dynamicResource.initialId !== undefined && oldSpecDynamicResources[dynamicResource.initialId]
                           ? oldSpecDynamicResources[dynamicResource.initialId]
                           : undefined;

                     const specDynamicResource = exist
                        ? deepClone(exist)
                        : ({} as TStageSpec_TStageDynamicResourceSpec);

                     const oldSpecDynamicResource = deepClone(specDynamicResource);

                     patchString(specDynamicResource, 'deploy_unit_ref', () => du.id);

                     patchObject(specDynamicResource, 'dynamic_resource', specDynamicResourceSpec => {
                        patchNumber(specDynamicResourceSpec, 'update_window', () => dynamicResource.updateWindow);

                        patchList(specDynamicResourceSpec, 'deploy_groups', specDeployGroups => {
                           if (!specDeployGroups[0]) {
                              specDeployGroups.push({
                                 // это поле добавляем только для новых ресурсов
                                 // у старых ресурсов его не трогаем
                                 // (оно может быть кастомным и его может совсем не быть)
                                 mark: 'all',
                              } as TDynamicResourceSpec_DeployGroup);
                           }

                           patchList(specDeployGroups[0], 'urls', specUrls => {
                              for (const resourceUrl of dynamicResource.urls) {
                                 const specUrl = specUrls.find(v => v === resourceUrl);

                                 if (!specUrl && resourceUrl) {
                                    specUrls.push(resourceUrl);
                                 }

                                 const { removed } = getSetDifference(new Set(specUrls), new Set(dynamicResource.urls));

                                 removed.forEach(url => {
                                    const i = specUrls.findIndex(v => v === url);
                                    specUrls.splice(i, 1);
                                 });
                              }

                              return skipEmpty(specUrls);
                           });

                           patchObject(specDeployGroups[0], 'storage_options', specStorageOptions => {
                              const { notifyPolicy, advancedSettings } = dynamicResource;

                              patchString(specStorageOptions, 'box_ref', () => box.id);

                              patchString(specStorageOptions, 'storage_dir', () => dynamicResource.storageDir);

                              patchString(specStorageOptions, 'destination', () => dynamicResource.destination);

                              patchNumber(
                                 specStorageOptions,
                                 'cached_revisions_count',
                                 () => dynamicResource.cachedRevisionsCount,
                              );

                              if (notifyPolicy.mode !== DynamicResourceNotifyPolicyMode.Http) {
                                 delete specStorageOptions.http_action;
                              } else {
                                 patchObject(specStorageOptions, 'http_action', specHttpAction => {
                                    patchString(specHttpAction, 'url', () => notifyPolicy.httpAction?.url);
                                    patchString(
                                       specHttpAction,
                                       'expected_answer',
                                       () => notifyPolicy.httpAction?.expectedAnswer,
                                    );

                                    return specHttpAction;
                                 });
                              }

                              if (notifyPolicy.mode !== DynamicResourceNotifyPolicyMode.Exec) {
                                 delete specStorageOptions.exec_action;
                              } else {
                                 patchObject(specStorageOptions, 'exec_action', specExecAction => {
                                    patchString(
                                       specExecAction,
                                       'command_line',
                                       () => notifyPolicy.execAction?.commandLine,
                                    );
                                    patchString(
                                       specExecAction,
                                       'expected_answer',
                                       () => notifyPolicy.execAction?.expectedAnswer,
                                    );
                                    return specExecAction;
                                 });
                              }

                              patchBoolean(
                                 specStorageOptions,
                                 'allow_deduplication',
                                 () => advancedSettings.allowDeduplication,
                              );

                              patchObject(specStorageOptions, 'verification', verification => {
                                 patchString(verification, 'checksum', () => advancedSettings.verification.checksum);

                                 patchNumber(
                                    verification,
                                    'check_period_ms',
                                    () => advancedSettings.verification.checkPeriodMs,
                                 );

                                 return verification;
                              });

                              patchNumber(
                                 specStorageOptions,
                                 'max_download_speed',
                                 () => advancedSettings.maxDownloadSpeed,
                              );

                              return specStorageOptions;
                           });

                           return specDeployGroups;
                        });

                        patchNumber(specDynamicResourceSpec, 'revision', revision => {
                           // ревизия должна автоматически апаться на бекенде

                           const isRenamedDynamicResource = exist && dynamicResource.id !== dynamicResource.initialId;
                           if (isRenamedDynamicResource) {
                              // убираем, если переименовали, бекенд сам проставит нужную
                              return undefined;
                           }

                           if (exist && !isEqual(oldSpecDynamicResource, specDynamicResource)) {
                              // апаем только для красоты (диффа), если что-то поменялось в настройках ресурса
                              return revision ? revision + 1 : revision;
                           }

                           return revision;
                        });

                        return specDynamicResourceSpec;
                     });

                     return specDynamicResource;
                  });
               }

               dynamicResourceIdList.push(dynamicResource.id);
            }
         }
      }

      const { removed } = getSetDifference(
         new Set(Object.keys(specDynamicResources) ?? []),
         new Set(dynamicResourceIdList),
      );

      removed.forEach(id => {
         delete specDynamicResources[id];
      });

      return specDynamicResources;
   }

   private patchDeployUnitSettings(
      deployUnits: DeployUnit[],
      specDeployUnitSettings: TStageSpec['deploy_unit_settings'],
   ) {
      for (const du of deployUnits) {
         patchObject(specDeployUnitSettings, du.id, specDuSettings => {
            patchObject(specDuSettings, 'alerting', specAlerting => {
               patchString(specAlerting, 'state', () =>
                  du.settings.alerting.state ? TStageSpec_TDeployUnitSettings_TAlerting_EState.ENABLED : undefined,
               );

               patchObject(specAlerting, 'notification_channels', notificationChannels => {
                  patchString(notificationChannels, 'ERROR', () =>
                     !isEmpty(du.settings.alerting.notificationChannel)
                        ? du.settings.alerting.notificationChannel
                        : undefined,
                  );

                  return notificationChannels;
               });

               return specAlerting;
            });

            patchString(specDuSettings, 'environment', () => {
               switch (du.settings.environment) {
                  case EnvironmentSettings.TESTING:
                     return TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.TESTING;
                  case EnvironmentSettings.PRESTABLE:
                     return TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.PRESTABLE;
                  case EnvironmentSettings.STABLE:
                     return TStageSpec_TDeployUnitSettings_EDeployUnitEnvironment.STABLE;
                  default:
                     return undefined;
               }
            });

            return specDuSettings;
         });
      }

      const { removed } = getSetDifference(
         new Set(Object.keys(specDeployUnitSettings) ?? []),
         new Set(deployUnits.map(du => du.id)),
      );

      removed.forEach(id => {
         delete specDeployUnitSettings[id];
      });

      return specDeployUnitSettings;
   }

   private toValue(): TStage {
      const patchingStage = deepClone(this.raw);
      const { stage } = this;

      // meta.account_id по смыслу основное поле, spec.account_id - алиас
      // технически можно писать в любое из них
      // если проект поменялся, то удаляем оба поля, чтобы брать квоту из проекта #DEPLOY-5744
      const hasChangedAccountId =
         patchingStage.spec?.account_id && // если такое значение было
         !isEmpty(patchingStage.spec.account_id) && // если оно не было пустым
         patchingStage.spec.account_id !== stage.project?.accountId; // если оно изменилось

      patchObject(patchingStage, 'meta', meta => {
         patchString(meta, 'id', () => stage.id);
         patchString(meta, 'project_id', () => stage.project?.id);
         patchString(meta, 'account_id', () => (hasChangedAccountId ? undefined : stage.project?.accountId)); // DEPLOY-5744

         return meta;
      });

      patchObject(patchingStage, 'labels', labels => {
         patchList(labels, 'tags', () => stage.labels?.tags);

         patchNumber(labels, 'infra_service', () => stage.infra?.service?.id);
         patchString(labels, 'infra_service_name', () => stage.infra?.service?.name);

         patchNumber(labels, 'infra_environment', () => stage.infra?.environment?.id);
         patchString(labels, 'infra_environment_name', () => stage.infra?.environment?.name);

         // https://st.yandex-team.ru/DEPLOY-5195#62068a4d19c28f389b25b928
         patchObject(labels, 'du_patchers_autoupdate_revision', duPatchersAutoupdateRevision => {
            // https://st.yandex-team.ru/DEPLOY-5195#6222258537bf95501da1f2d1
            if (this.visitor.removeDuPatchersAutoupdateRevisionLabel) {
               if (duPatchersAutoupdateRevision) {
                  stage.deployUnits.forEach(du => {
                     const duPatchersAutoupdateRevisionConfirmed = this.visitor.isDuConfirmed(
                        du.id,
                        ConfirmationType.PatchersRevisionUpdate,
                     );

                     if (
                        duPatchersAutoupdateRevision[du.id] &&
                        (du.patchersRevision.value || duPatchersAutoupdateRevisionConfirmed)
                     ) {
                        delete duPatchersAutoupdateRevision[du.id];
                     }
                  });
               }
            }

            return skipEmpty(duPatchersAutoupdateRevision);
         });

         return labels;
      });

      const isSox = stage.soxService;

      patchObject(patchingStage, 'spec', spec => {
         if (hasChangedAccountId && spec.account_id) {
            // только удаляем, если поменялся #DEPLOY-5744
            patchString(spec, 'account_id', () => undefined);
         }

         patchBoolean(spec, 'sox_service', () => isSox);

         patchObject(spec, 'deploy_units', deployUnits => {
            for (const du of stage.deployUnits) {
               patchObject(deployUnits, du.id, deployUnit => {
                  const patchedDeployUnit = DeployUnitPatcher.patch(
                     this.visitor,
                     new StageParentNodeIds(this.stage),
                     deployUnit,
                     du,
                  );

                  // TODO khoden: мне очень не нравится это по двум причинам
                  //  1. Это должно быть внутри DeployUnitPatcher
                  //  2. Нехорошо вносить изменения в спеку, которые не запрашивались
                  patchBoolean(patchedDeployUnit, 'sox_service', () => isSox);

                  return patchedDeployUnit;
               });
            }

            const { removed } = getSetDifference(
               new Set(Object.keys(deployUnits)),
               new Set(stage.deployUnits.map(du => du.id)),
            );

            removed.forEach(du => {
               delete deployUnits[du];

               // Удаляю также из статуса, просто чтобы показать красивый дифф. В API не отсылается.
               patchObject(patchingStage, 'status', status => {
                  patchObject(status, 'deploy_units', statusDeployUnits => {
                     delete statusDeployUnits[du];

                     return statusDeployUnits;
                  });

                  return status;
               });
            });

            return deployUnits;
         });

         patchObject(spec, 'dynamic_resources', specDynamicResources =>
            this.patchDynamicResources(stage.deployUnits, specDynamicResources),
         );

         patchObject(spec, 'deploy_unit_settings', specDeployUnitSettings =>
            this.patchDeployUnitSettings(stage.deployUnits, specDeployUnitSettings),
         );

         return spec;
      });

      return patchingStage;
   }
}
