import { plural } from '@yandex-infracloud-ui/libs';
import { StatusState } from '../../../../components/lib';
import { EConditionStatus, TStage, TStageStatus } from '../../../../proto-typings';
import { parseYpTimestamp } from '../../../../utils';
import {
   DeployUnitState,
   DeployUnitStatus,
   DeployUnitStatusConverter,
   DuStatusLabel,
} from '../DeployUnit/DeployUnitStatusConverter';
import { Stage } from './Stage';

interface ValidationInfo {
   isValid: boolean;
   message: string | null;
   reason: string | null;
}

export enum StageStatusState {
   Ready = 'ready',
   InProgress = 'inProgress',
   Error = 'error',
}

export const stageStatusesMap: Record<StageStatusState, StatusState> = {
   [StageStatusState.Ready]: StatusState.Ok,
   [StageStatusState.InProgress]: StatusState.Progress,
   [StageStatusState.Error]: StatusState.Error,
};

export interface StageStatusLabel {
   state: StageStatusState;
   text: string;
   additional: string;
   deployUnitLabels: Map<string, DuStatusLabel>;
}

interface StageStatusDuCount {
   total: number;
   ready: number;
   inProgress: number;
   failed: number;
   unknown: number;
}

export interface StageStatus {
   revision: number | null;
   validationInfo: ValidationInfo;
   updateDate: Date | null;
   deployUnits: Map<string, DeployUnitStatus>;
   duCount: StageStatusDuCount;
}

function getStageStatus(rawStatus: TStageStatus | null): StageStatus {
   const { validated, spec_timestamp, revision, deploy_units, runtime_deploy_controls } = rawStatus ?? {};

   const deployUnits = new Map(
      deploy_units
         ? Object.keys(deploy_units).map(duName => [
              duName,
              DeployUnitStatusConverter.getDeployUnitStatus({
                 id: duName,
                 status: deploy_units[duName] ?? null,
                 controls: runtime_deploy_controls?.[duName] ?? null,
              }),
           ])
         : [],
   );

   const duCount = getStageDuCount(deployUnits);

   const result: StageStatus = {
      revision: revision ?? null,
      validationInfo: {
         isValid: !validated || validated?.status === EConditionStatus.CS_TRUE,
         message: validated?.message ?? null,
         reason: validated?.reason ?? null,
      },
      updateDate: spec_timestamp ? parseYpTimestamp(spec_timestamp) : null,
      deployUnits,
      duCount,
   };

   return result;
}

const countStatus: Record<DeployUnitState, keyof StageStatusDuCount> = {
   [DeployUnitState.Ready]: 'ready',
   [DeployUnitState.InProgress]: 'inProgress',
   [DeployUnitState.Failed]: 'failed',
   [DeployUnitState.Unknown]: 'unknown',
};

function getStageDuCount(deployUnits: Map<string, DeployUnitStatus>): StageStatusDuCount {
   const count: StageStatusDuCount = {
      total: deployUnits.size,
      ready: 0,
      inProgress: 0,
      failed: 0,
      unknown: 0,
   };

   for (const status of deployUnits.values()) {
      count[countStatus[status.statusState]] += 1;
   }

   return count;
}

const textStatus: Record<DeployUnitState, string> = {
   [DeployUnitState.Ready]: 'ready',
   [DeployUnitState.InProgress]: 'in progress',
   [DeployUnitState.Failed]: 'failed',
   [DeployUnitState.Unknown]: 'unknown',
};

const countText = (total: number) => ([value, state]: readonly [number, DeployUnitState]) =>
   value > 0 ? `${value}/${total} DU${value > 1 ? 's' : ''} ${textStatus[state]}` : '';

function getAdditionalText(duCount: StageStatusDuCount): string {
   const { total, ready, inProgress, failed, unknown } = duCount;
   return ([
      [ready, DeployUnitState.Ready],
      [inProgress, DeployUnitState.InProgress],
      [failed, DeployUnitState.Failed],
      [unknown, DeployUnitState.Unknown],
   ] as const)
      .map(countText(total))
      .filter(Boolean)
      .join(', ');
}

interface Environment {
   deployEngine: string;
}

export const pluralDeployUnits = (count: number) => `${count} deploy ${plural(count, 'unit', 'units')}`;
export const pluralState = (count: number, state: string) => `${state}`;
export const duAllPrefix = (count: number, total: number) => (count === total && count > 1 ? 'All ' : '');
function getStageStatusLabel(
   stage: Stage,
   status: StageStatus,
   labels: TStage['labels'],
   environment: Environment,
): StageStatusLabel {
   const currentEngine: string | undefined = labels?.deploy_engine;
   const { deployEngine } = environment;
   const badEngine = currentEngine !== deployEngine;

   const { revision, deployUnits } = stage;
   const duMap = new Map(deployUnits.map(du => [du.id, du]));
   const { validationInfo, duCount, revision: statusRevision, deployUnits: duStatuses } = status;
   const { isValid, reason, message } = validationInfo;
   const { total, ready, failed, inProgress } = duCount;

   const isValidating = statusRevision !== revision;

   const duIdList = Array.from(new Set([...duMap.keys(), ...duStatuses.keys()]));
   const deployUnitLabels = new Map(
      duIdList.map(name => [
         name,
         DeployUnitStatusConverter.getDuStatusLabel(duMap.get(name) ?? null, duStatuses.get(name) ?? null),
      ]),
   );

   const duCountText = getAdditionalText(duCount);

   let toolTip = '';
   let statusText = '';
   let state: StageStatusState = StageStatusState.Ready;

   // порядок имеет значение
   if (badEngine) {
      // несоотвествие по группе контроллеров
      state = StageStatusState.Error;
      statusText = currentEngine ? `Unknown deploy_engine (${currentEngine})` : 'deploy_engine is not specified';
   } else if (!isValid) {
      // валидация со стороны YP
      state = StageStatusState.Error;
      // Reason can be some ENUM like value, e.g. SANDBOX_RESOLVE_FAILED.
      statusText = reason?.toLocaleLowerCase().replaceAll('_', ' ') ?? '';
      toolTip = message ?? '';
   } else if (isValidating) {
      // если ревизия в спеке и статусе не совпадает
      // борьба с зависаниями???
      // TODO: https://st.yandex-team.ru/DEPLOY-4734#60f57a0360474549b5e2eb78
      state = StageStatusState.InProgress;
      statusText = 'Validating spec';
      if (total === 0) {
         const detectObject = statusRevision ? ' deploy units' : ' a new Stage';
         toolTip = `Waiting for a Stage Controller to detect ${detectObject}`;
      }
   } else if (total === 0) {
      // обратная сторона — старые деплой юниты не сразу исчезают из статуса — обработать
      // пока не починится на бекендах https://st.yandex-team.ru/DEPLOY-3664
      // оставлять ready???
      statusText = 'No deploy units in stage status';
   } else if (failed > 0) {
      // Infer overall stage status text.
      state = StageStatusState.Error;
      statusText = `${duAllPrefix(failed, total)}${pluralDeployUnits(failed)} ${pluralState(failed, 'failed')}`;
      toolTip = duCountText;
   } else if (inProgress > 0) {
      // Infer overall stage status text.
      state = StageStatusState.InProgress;
      statusText = `${duAllPrefix(inProgress, total)}${pluralDeployUnits(inProgress)} ${pluralState(
         inProgress,
         'in progress',
      )}`;
      toolTip = duCountText;
   } else if (ready > 0) {
      // Infer overall stage status text.
      state = StageStatusState.Ready;
      statusText = `${duAllPrefix(ready, total)}${pluralDeployUnits(ready)} ${pluralState(ready, 'ready')}`;
      toolTip = duCountText;
   } else {
      state = StageStatusState.InProgress;
      statusText = 'Unknown';
      toolTip = duCountText;
   }

   return {
      state,
      text: statusText,
      additional: toolTip,
      deployUnitLabels,
   };
}

export const StageStatusConverter = {
   getStageStatus,
   getStageStatusLabel,
};
