// Класс StagePatcherVisitor собирает необходимые данные для спеки, которые не были внесены в форму напрямую.
// - передает набор сгенерированных токенов для секретов
// - передает флаги, обновлять ли версии сайдкаров и других настроек

import { SecretResolver } from '../../../modules/secrets';
import { ConfirmationType, StageConfirmations } from '../Confirmations';

export interface StagePatcherOptions {
   updateLogbrokerTools: boolean;
   enableDynamicResourceUpdater: boolean;
   updatePodAgent: boolean;
   updatePodLayer: boolean;
   updateTvmAgent: boolean;
   updateDruAgent: boolean;

   // https://st.yandex-team.ru/DEPLOY-5195#6222258537bf95501da1f2d1
   removeDuPatchersAutoupdateRevisionLabel: boolean;
}

export const defaultStagePatcherOptions: StagePatcherOptions = {
   updateLogbrokerTools: true,
   enableDynamicResourceUpdater: true,
   updatePodAgent: true,
   updatePodLayer: true,
   updateTvmAgent: true,
   updateDruAgent: true,
   removeDuPatchersAutoupdateRevisionLabel: true,
};

enum PatcherSeverity {
   Info = 'info',
   Warning = 'warning',
   Error = 'error',
}

interface LogMessage {
   path: string;
   severity: PatcherSeverity;
   message: string;
}

interface StagePatcherVisitorProps {
   duConfirmations?: StageConfirmations;
   initialStageId?: string;
   options?: Partial<StagePatcherOptions>;
   secretResolver?: SecretResolver;
}

export class StagePatcherVisitor implements StagePatcherOptions {
   public initialStageId?: string;

   public secretResolver?: SecretResolver;

   public updateDruAgent!: boolean;

   public updateLogbrokerTools!: boolean;

   public enableDynamicResourceUpdater!: boolean;

   public updatePodAgent!: boolean;

   public updatePodLayer!: boolean;

   public updateTvmAgent!: boolean;

   public removeDuPatchersAutoupdateRevisionLabel!: boolean;

   private duConfirmations: StageConfirmations;

   private logMessages: LogMessage[] = [];

   constructor({ initialStageId, options = {}, duConfirmations = {}, secretResolver }: StagePatcherVisitorProps = {}) {
      const def = defaultStagePatcherOptions;

      this.initialStageId = initialStageId;

      this.duConfirmations = duConfirmations;

      for (const o of Object.keys(def)) {
         const option: keyof StagePatcherOptions = o as keyof StagePatcherOptions;
         this[option] = options[option] ?? def[option];
      }

      this.secretResolver = secretResolver;
   }

   public log(severity: PatcherSeverity, path: string, message: string): void {
      this.logMessages.push({ severity, path, message });
   }

   public getLog(): LogMessage[] {
      return this.logMessages;
   }

   public isDuConfirmed(duId: string, confirmationType: ConfirmationType): boolean {
      if (this.duConfirmations?.hasOwnProperty(duId)) {
         return this.duConfirmations[duId].has(confirmationType);
      }

      return false;
   }
}
