import { isEmpty } from '@yandex-infracloud-ui/libs';

import { EnvironmentVariable } from '../../../../modules/environment/models';
import { EUnixSignalType, TCoredumpPolicy, TMonitoringUnistatEndpoint_EOutputFormat } from '../../../../proto-typings';
import { Entity } from '../../../../redux/models';
import { DeepPartial } from '../../../typeHelpers';
import { WorkloadEmptyParams } from '../stage-levels';
import { YasmTags } from '../yasm';

// region Advanced settings
export interface WorkloadCommandLimits {
   cpuLimit: number | null;
   ramLimit: number | null;
}

export interface WorkloadCommandAccess {
   group: string | null;
   user: string | null;
}

// TODO: camelCase
export interface RetryPolicy {
   initialDelayMs: number | null;
   restartPeriodScaleMs: number | null;
   restartPeriodBackOff: number | null;
   maxRestartPeriodMs: number | null;
   minRestartPeriodMs: number | null;
   maxExecutionTimeMs: number | null;
}

export interface WorkloadCommandWithRetryPolicy {
   retryPolicy?: RetryPolicy;
}

export interface WorkloadCommandWithAdvancedSettings extends WorkloadCommandWithRetryPolicy {
   access?: WorkloadCommandAccess;
   limits?: WorkloadCommandLimits;
}

// endregion

export interface WorkloadExec extends WorkloadCommandWithAdvancedSettings {
   _order?: number;
   command: string | null;
}

export interface WorkloadTcp extends WorkloadCommandWithRetryPolicy {
   port: number | null;
   limits?: WorkloadCommandLimits;
}

export interface WorkloadHttp extends WorkloadCommandWithRetryPolicy {
   port: number | null;
   path: string | null;
   expectedAnswer: string | null; // TODO: camelCase
   any: boolean;
   limits?: WorkloadCommandLimits;
}

export interface WorkloadUnixSignal extends WorkloadCommandWithRetryPolicy {
   signal: EUnixSignalType;
}

export enum WorkloadProbeMode {
   HTTP = 'http',
   Exec = 'exec',
   TCP = 'tcp',
}

export enum WorkloadStopMode {
   HTTP = 'http',
   Exec = 'exec',
   UnixSignal = 'unixSignal',
}

export enum WorkloadDestroyMode {
   HTTP = 'http',
   Exec = 'exec',
}

export interface WorkloadProbeCommand {
   exec?: WorkloadExec;
   http?: WorkloadHttp;
   mode: WorkloadProbeMode;
   tcp?: WorkloadTcp;
}

export interface WorkloadStopCommand {
   exec?: WorkloadExec;
   http?: WorkloadHttp;
   maxTries: number | null;
   mode?: WorkloadStopMode;
   unixSignal?: WorkloadUnixSignal;
}

export interface WorkloadDestroyCommand {
   exec?: WorkloadExec;
   http?: WorkloadHttp;
   maxTries: number | null;
   mode?: WorkloadDestroyMode;
}

export interface WorkloadCommands {
   init: WorkloadExec[];
   liveness: WorkloadProbeCommand;
   readiness: WorkloadProbeCommand;
   start: WorkloadExec;
   stop: WorkloadStopCommand;
   destroy: WorkloadDestroyCommand;
}

export interface WorkloadYasm {
   unistats: WorkloadUnistatYasm[];
   porto: WorkloadPortoYasm;
}

export interface WorkloadUnistatYasm {
   port: number | null;
   url: string | null;
   yasmTags: YasmTags;
   inheritMissedLabels: boolean;
   outputFormat: Exclude<
      TMonitoringUnistatEndpoint_EOutputFormat,
      TMonitoringUnistatEndpoint_EOutputFormat.OF_UNKNOWN
   > | null;
   prefix: string | null;
}

export interface WorkloadPortoYasm {
   usePortoMetrics: boolean;
   inheritMissedLabels: boolean;
   yasmTags: YasmTags;
}

export interface Workload extends Entity {
   commands: WorkloadCommands;
   environment: EnvironmentVariable[];
   logs: boolean;
   yasm: WorkloadYasm;
   coredumpPolicy: DeepPartial<TCoredumpPolicy> | null;

   /**
    * ID данное этому Workload при заведении узла в форме.
    * В отличие от id - не редактируется и не меняется, вплоть до сохранения.
    */
   initialId?: string;
}

export const workloadDefaultValues = {
   yasm: {
      unistat: {
         path: '/',
         prefix: 'unistat',
         port: null,
         outputFormat: TMonitoringUnistatEndpoint_EOutputFormat.OF_YASM_JSON,
      },
   },
};

export type WorkloadUnistatEmptyParams = WorkloadEmptyParams & {};

export const getEmptyWorkloadUnistat = (params?: WorkloadUnistatEmptyParams): WorkloadUnistatYasm => ({
   url: null,
   port: null,
   yasmTags: {
      itype: params?.projectId ?? null,
      tags: [],
   },
   inheritMissedLabels: false,
   outputFormat: null,
   prefix: null,
});

export const getEmptyWorkload = (params?: WorkloadEmptyParams): Workload => ({
   id: 'workload',
   commands: {
      start: { command: 'bash -c "echo hello world"' },
      init: [{ command: null }],
      liveness: { mode: WorkloadProbeMode.TCP },
      readiness: { mode: WorkloadProbeMode.TCP, tcp: { port: 80 } },
      stop: { mode: WorkloadStopMode.Exec, maxTries: null },
      destroy: { mode: WorkloadDestroyMode.Exec, maxTries: null },
   },
   yasm: {
      unistats: [],
      porto: {
         usePortoMetrics: false,
         inheritMissedLabels: false,
         yasmTags: {
            itype: params?.projectId ?? null,
            tags: [],
         },
      },
   },
   logs: true,
   environment: [],
   coredumpPolicy: null,
});

export const isExecEmpty = (exec?: WorkloadExec) => {
   let empty = true;

   if (exec?.command) {
      empty = false;
   }

   return empty;
};

export const isUnixSignalEmpty = (unixSignal?: WorkloadUnixSignal) => {
   let empty = true;

   if (unixSignal?.signal) {
      empty = false;
   }

   return empty;
};

export const isProbeEmpty = (probe: WorkloadProbeCommand) => {
   let empty = true;

   if (probe.mode === WorkloadProbeMode.TCP) {
      if (!isEmpty(probe.tcp?.port)) {
         empty = false;
      }
   } else if (probe.mode === WorkloadProbeMode.HTTP) {
      if (!isEmpty(probe.http?.port)) {
         empty = false;
      }
   } else if (probe.mode === WorkloadProbeMode.Exec) {
      if (!isExecEmpty(probe.exec)) {
         empty = false;
      }
   }

   return empty;
};

export const isStopEmpty = (stop: WorkloadStopCommand) => {
   let empty = true;

   if (stop.mode === WorkloadStopMode.HTTP) {
      if (!isEmpty(stop.http?.port)) {
         empty = false;
      }
   } else if (stop.mode === WorkloadStopMode.Exec) {
      if (!isExecEmpty(stop.exec)) {
         empty = false;
      }
   } else if (stop.mode === WorkloadStopMode.UnixSignal) {
      if (!isUnixSignalEmpty(stop.unixSignal)) {
         empty = false;
      }
   }

   return empty;
};

export const isDestroyEmpty = (destroy: WorkloadDestroyCommand) => {
   let empty = true;

   if (destroy.mode === WorkloadDestroyMode.HTTP) {
      if (!isEmpty(destroy.http?.port)) {
         empty = false;
      }
   } else if (destroy.mode === WorkloadDestroyMode.Exec) {
      if (!isExecEmpty(destroy.exec)) {
         empty = false;
      }
   }

   return empty;
};

export const isWorkloadAdvancedSettingsEmpty = (advanced?: WorkloadCommandWithAdvancedSettings) => {
   if (!advanced) {
      return true;
   }

   const checkObjects = [advanced.retryPolicy, advanced.limits, advanced.access];

   for (const element of checkObjects) {
      if (!isEmpty(element)) {
         if (Object.values(element).some(Boolean)) return false;
      }
   }

   return true;
};
