import { EnvironmentVariableType } from '../../../../modules/environment/models';
import {
   EUnixSignalType,
   TDeployUnitSpec,
   TPodAgentSpec,
   TPodTemplateSpec,
   TWorkload,
} from '../../../../proto-typings';
import { noop } from '../../../../utils';
import { DeepPartial } from '../../../typeHelpers';

import { Workload, WorkloadDestroyMode, WorkloadProbeMode, WorkloadStopMode } from './Workload';
import { WorkloadConverter } from './WorkloadConverter';

const getInitialWorkloadSpec = (): DeepPartial<TWorkload> => ({
   id: 'Workload-123',
   // box_ref: 'Box-123',
});

const getInitialPodAgentSpec = (): DeepPartial<TPodAgentSpec> => ({
   // ...
   workloads: [getInitialWorkloadSpec()],
});

const getInitialPodTemplateSpec = (): DeepPartial<TPodTemplateSpec> => ({
   spec: {
      host_infra: {
         monitoring: {
            unistats: [
               {
                  workload_id: 'Workload-Yasm',
                  port: 1,
                  path: 'yasm',
                  labels: {
                     itype: 'itype',
                     key1: 'value1',
                     key2: 'value2',
                     '': '',
                  },
               },
            ],
         },
      },
      pod_agent_payload: {
         spec: getInitialPodAgentSpec(),
      },
   },
});

const getInitialDeployUnitSpec = (): DeepPartial<TDeployUnitSpec> => ({});

const COMMAND_SETTINGS = {
   access: {
      empty: {
         user: null,
         group: null,
      },
      nonempty: {
         user: 'user',
         group: 'user',
      },
   },
   // spec
   compute_resources: {
      vcpu_limit: 100,
      memory_limit: 1234567890,
   },
   // form
   limits: {
      empty: {
         cpuLimit: null,
         ramLimit: null,
      },
      nonempty: {
         cpuLimit: 100,
         ramLimit: 1234567890,
      },
   },
   // spec
   time_limit: {
      initial_delay_ms: 1,
      restart_period_scale_ms: 2,
      restart_period_back_off: 3,
      min_restart_period_ms: 4,
      max_restart_period_ms: 5,
      max_execution_time_ms: 6,
   },
   // form
   retryPolicy: {
      empty: {
         initialDelayMs: null,
         restartPeriodScaleMs: null,
         restartPeriodBackOff: null,
         minRestartPeriodMs: null,
         maxRestartPeriodMs: null,
         maxExecutionTimeMs: null,
      },
      nonempty: {
         initialDelayMs: 1,
         restartPeriodScaleMs: 2,
         restartPeriodBackOff: 3,
         minRestartPeriodMs: 4,
         maxRestartPeriodMs: 5,
         maxExecutionTimeMs: 6,
      },
   },
};

/**
 * @example
 * it('should xxx', check(
 *    spec => {
 *
 *    },
 *    expected => {
 *
 *    },
 *    model => {
 *
 *    }
 *  ))
 */
const check = (
   patchSpecBefore: (s: DeepPartial<TWorkload>) => void,
   patchExpectedModel: (s: DeepPartial<Workload>) => void,
   checkOnlyBranch?: (s: Workload) => any,
) => () => {
   const spec = getInitialWorkloadSpec() as TWorkload;
   const podTemplateSpec = getInitialPodTemplateSpec() as TPodTemplateSpec;
   const deployUnitSpec = getInitialDeployUnitSpec() as TDeployUnitSpec;

   patchSpecBefore(spec);

   const model = WorkloadConverter.fromApi(spec, podTemplateSpec, deployUnitSpec);

   const expectedConvertedModel = getInitialWorkloadSpec();
   patchExpectedModel(expectedConvertedModel);

   if (checkOnlyBranch) {
      expect(checkOnlyBranch(model)).toEqual(checkOnlyBranch(expectedConvertedModel as Workload));
   } else {
      expect(model).toEqual(expectedConvertedModel);
   }
};

describe('models/ui|WorkloadConverter', () => {
   it(
      'should extract id',
      check(
         noop,
         expected => {
            expected.id = 'Workload-123';
         },
         model => model.id,
      ),
   );

   it(
      'should extract logs: empty',
      check(
         noop,
         expected => {
            expected.logs = false;
         },
         model => model.logs,
      ),
   );

   it(
      'should extract logs',
      check(
         spec => {
            spec.transmit_logs = true;
         },
         expected => {
            expected.logs = true;
         },
         model => model.logs,
      ),
   );

   it(
      'should extract env: empty',
      check(
         noop,
         expected => {
            expected.environment = [];
         },
         model => model.environment,
      ),
   );

   it(
      'should extract env',
      check(
         spec => {
            spec.env = [
               {
                  name: 'cba',
                  value: {
                     secret_env: {
                        alias: 'sec-123:ver-123',
                        id: 'secret',
                     },
                  },
               },
               {
                  name: 'abc',
                  value: {
                     literal_env: {
                        value: 'literal',
                     },
                  },
               },
            ];
         },
         expected => {
            expected.environment = [
               {
                  name: 'cba',
                  type: EnvironmentVariableType.Secret,
                  value: {
                     alias: 'sec-123:ver-123',
                     key: 'secret',
                  },
               },
               {
                  name: 'abc',
                  type: EnvironmentVariableType.Literal,
                  value: 'literal',
               },
            ];
         },
         model => model.environment,
      ),
   );

   it(
      'should extract yasm: empty',
      check(
         noop,
         expected => {
            expected.yasm = {
               unistats: [],
               porto: {
                  inheritMissedLabels: false,
                  usePortoMetrics: false,
                  yasmTags: {
                     itype: null,
                     tags: [],
                  },
               },
            };
         },
         model => model.yasm,
      ),
   );

   it(
      'should extract yasm',
      check(
         spec => {
            spec.id = 'Workload-Yasm';
         },
         expected => {
            expected.yasm = {
               unistats: [
                  {
                     port: 1,
                     url: 'yasm',
                     yasmTags: {
                        itype: 'itype',
                        tags: [
                           {
                              key: 'key1',
                              value: 'value1',
                           },
                           {
                              key: 'key2',
                              value: 'value2',
                           },
                        ],
                     },
                     inheritMissedLabels: false,
                     prefix: null,
                     outputFormat: null,
                  },
               ],
               porto: {
                  usePortoMetrics: false,
                  inheritMissedLabels: false,
                  yasmTags: {
                     itype: null,
                     tags: [],
                  },
               },
            };
         },
         model => model.yasm,
      ),
   );

   describe('should extract commands', () => {
      it(
         'empty commands',
         check(
            noop,
            expected => {
               const emptyExec = {
                  command: null,
                  access: COMMAND_SETTINGS.access.empty,
                  limits: COMMAND_SETTINGS.limits.empty,
                  retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
               };

               expected.commands = {
                  start: emptyExec,
                  init: [emptyExec],
                  liveness: {
                     mode: WorkloadProbeMode.TCP,
                     tcp: {
                        port: null,
                        retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                     },
                  },
                  readiness: {
                     mode: WorkloadProbeMode.TCP,
                     tcp: {
                        port: null,
                        retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                     },
                  },
                  stop: {
                     mode: WorkloadStopMode.Exec,
                     exec: emptyExec,
                     maxTries: null,
                  },
                  destroy: {
                     mode: WorkloadDestroyMode.Exec,
                     exec: emptyExec,
                     maxTries: null,
                  },
               };
            },
            model => model.commands,
         ),
      );

      describe('should extract start command', () => {
         it(
            'start command: empty',
            check(
               spec => {
                  spec.start = {
                     command_line: 'start command',
                  };
               },
               expected => {
                  expected.commands = {
                     start: {
                        command: 'start command',
                        access: COMMAND_SETTINGS.access.empty,
                        retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        limits: COMMAND_SETTINGS.limits.empty,
                     },
                  };
               },
               model => model.commands.start,
            ),
         );

         it(
            'start command: advanced settings',
            check(
               spec => {
                  spec.start = {
                     command_line: 'start command',
                     compute_resources: COMMAND_SETTINGS.compute_resources,
                     time_limit: COMMAND_SETTINGS.time_limit,
                     user: 'username',
                     group: 'groupname',
                  };
               },
               expected => {
                  expected.commands = {
                     start: {
                        command: 'start command',
                        access: {
                           user: 'username',
                           group: 'groupname',
                        },
                        retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        limits: COMMAND_SETTINGS.limits.nonempty,
                     },
                  };
               },
               model => model.commands.start,
            ),
         );
      });

      describe('should extract init commands', () => {
         it(
            'init commands: empty array',
            check(
               spec => {
                  spec.init = [];
               },
               expected => {
                  expected.commands = {
                     init: [
                        {
                           command: null,
                           access: COMMAND_SETTINGS.access.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     ],
                  };
               },
               model => model.commands.init,
            ),
         );

         it(
            'init commands',
            check(
               spec => {
                  spec.init = [
                     {
                        // command_line: 'init command 0',
                     },
                     {
                        command_line: 'init command 1',
                     },
                     {
                        command_line: 'init command 2',
                        user: 'username2',
                        group: 'groupname2',
                     },
                     {
                        command_line: 'init command 3',
                        user: 'username3',
                        group: 'groupname3',
                        compute_resources: COMMAND_SETTINGS.compute_resources,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  ];
               },
               expected => {
                  expected.commands = {
                     init: [
                        {
                           _order: 0,
                           command: null,
                           access: COMMAND_SETTINGS.access.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                        },
                        {
                           _order: 1,
                           command: 'init command 1',
                           access: COMMAND_SETTINGS.access.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                        },
                        {
                           _order: 2,
                           command: 'init command 2',
                           access: {
                              user: 'username2',
                              group: 'groupname2',
                           },
                           limits: COMMAND_SETTINGS.limits.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                        {
                           _order: 3,
                           command: 'init command 3',
                           access: {
                              user: 'username3',
                              group: 'groupname3',
                           },
                           limits: COMMAND_SETTINGS.limits.nonempty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     ],
                  };
               },
               model => model.commands.init,
            ),
         );
      });

      describe('should extract liveness probe', () => {
         it(
            'liveness probe TCP: empty',
            check(
               spec => {
                  spec.liveness_check = {
                     tcp_check: {
                        // port: 80,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     liveness: {
                        mode: WorkloadProbeMode.TCP,
                        tcp: {
                           port: null,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     },
                  };
               },
               model => model.commands.liveness,
            ),
         );

         it(
            'liveness probe TCP: advanced settings',
            check(
               spec => {
                  spec.liveness_check = {
                     tcp_check: {
                        port: 80,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     liveness: {
                        mode: WorkloadProbeMode.TCP,
                        tcp: {
                           port: 80,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     },
                  };
               },
               model => model.commands.liveness,
            ),
         );

         it(
            'liveness probe HTTP: empty',
            check(
               spec => {
                  spec.liveness_check = {
                     http_get: {
                        // port: 80,
                        // path: 'liveness path',
                        // expected_answer: 'liveness answer',
                        // any: false,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     liveness: {
                        mode: WorkloadProbeMode.HTTP,
                        http: {
                           port: null,
                           path: null,
                           expectedAnswer: null,
                           any: false,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     },
                  };
               },
               model => model.commands.liveness,
            ),
         );

         it(
            'liveness probe HTTP: advanced settings',
            check(
               spec => {
                  spec.liveness_check = {
                     http_get: {
                        port: 80,
                        path: 'liveness path',
                        expected_answer: 'liveness answer',
                        any: true,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     liveness: {
                        mode: WorkloadProbeMode.HTTP,
                        http: {
                           port: 80,
                           path: 'liveness path',
                           expectedAnswer: 'liveness answer',
                           any: true,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     },
                  };
               },
               model => model.commands.liveness,
            ),
         );

         it(
            'liveness probe Exec: empty',
            check(
               spec => {
                  spec.liveness_check = {
                     container: {
                        // command_line: 'Exec command',
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     liveness: {
                        mode: WorkloadProbeMode.Exec,
                        exec: {
                           command: null,
                           access: COMMAND_SETTINGS.access.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     },
                  };
               },
               model => model.commands.liveness,
            ),
         );

         it(
            'liveness probe Exec: advanced settings',
            check(
               spec => {
                  spec.liveness_check = {
                     container: {
                        command_line: 'Exec command',
                        user: COMMAND_SETTINGS.access.nonempty.user,
                        group: COMMAND_SETTINGS.access.nonempty.group,
                        compute_resources: COMMAND_SETTINGS.compute_resources,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     liveness: {
                        mode: WorkloadProbeMode.Exec,
                        exec: {
                           command: 'Exec command',
                           access: COMMAND_SETTINGS.access.nonempty,
                           limits: COMMAND_SETTINGS.limits.nonempty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     },
                  };
               },
               model => model.commands.liveness,
            ),
         );
      });

      describe('should extract readiness probe', () => {
         it(
            'readiness probe TCP: empty',
            check(
               spec => {
                  spec.readiness_check = {
                     tcp_check: {
                        // port: 80,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     readiness: {
                        mode: WorkloadProbeMode.TCP,
                        tcp: {
                           port: null,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     },
                  };
               },
               model => model.commands.readiness,
            ),
         );

         it(
            'readiness probe TCP: advanced settings',
            check(
               spec => {
                  spec.readiness_check = {
                     tcp_check: {
                        port: 80,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     readiness: {
                        mode: WorkloadProbeMode.TCP,
                        tcp: {
                           port: 80,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     },
                  };
               },
               model => model.commands.readiness,
            ),
         );

         it(
            'readiness probe HTTP: empty',
            check(
               spec => {
                  spec.readiness_check = {
                     http_get: {
                        // port: 80,
                        // path: 'readiness path',
                        // expected_answer: 'readiness answer',
                        // any: false,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     readiness: {
                        mode: WorkloadProbeMode.HTTP,
                        http: {
                           port: null,
                           path: null,
                           expectedAnswer: null,
                           any: false,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     },
                  };
               },
               model => model.commands.readiness,
            ),
         );

         it(
            'readiness probe HTTP: advanced settings',
            check(
               spec => {
                  spec.readiness_check = {
                     http_get: {
                        port: 80,
                        path: 'readiness path',
                        expected_answer: 'readiness answer',
                        any: true,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     readiness: {
                        mode: WorkloadProbeMode.HTTP,
                        http: {
                           port: 80,
                           path: 'readiness path',
                           expectedAnswer: 'readiness answer',
                           any: true,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     },
                  };
               },
               model => model.commands.readiness,
            ),
         );

         it(
            'readiness probe Exec: empty',
            check(
               spec => {
                  spec.readiness_check = {
                     container: {
                        // command_line: 'Exec command',
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     readiness: {
                        mode: WorkloadProbeMode.Exec,
                        exec: {
                           command: null,
                           access: COMMAND_SETTINGS.access.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                     },
                  };
               },
               model => model.commands.readiness,
            ),
         );

         it(
            'readiness probe Exec: advanced settings',
            check(
               spec => {
                  spec.readiness_check = {
                     container: {
                        command_line: 'Exec command',
                        user: COMMAND_SETTINGS.access.nonempty.user,
                        group: COMMAND_SETTINGS.access.nonempty.group,
                        compute_resources: COMMAND_SETTINGS.compute_resources,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     readiness: {
                        mode: WorkloadProbeMode.Exec,
                        exec: {
                           command: 'Exec command',
                           access: COMMAND_SETTINGS.access.nonempty,
                           limits: COMMAND_SETTINGS.limits.nonempty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                     },
                  };
               },
               model => model.commands.readiness,
            ),
         );
      });

      describe('should extract stop policy probe', () => {
         it(
            'stop policy: max_tries',
            check(
               spec => {
                  spec.stop_policy = {
                     max_tries: 2,
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.UnixSignal,
                        maxTries: 2,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );

         it(
            'stop policy Exec: empty',
            check(
               spec => {
                  spec.stop_policy = {
                     container: {
                        // command_line: 'Exec command',
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.Exec,
                        exec: {
                           command: null,
                           access: COMMAND_SETTINGS.access.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                        maxTries: null,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );

         it(
            'stop policy Exec: advanced settings',
            check(
               spec => {
                  spec.stop_policy = {
                     container: {
                        command_line: 'Exec command',
                        user: COMMAND_SETTINGS.access.nonempty.user,
                        group: COMMAND_SETTINGS.access.nonempty.group,
                        compute_resources: COMMAND_SETTINGS.compute_resources,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                     max_tries: 3,
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.Exec,
                        exec: {
                           command: 'Exec command',
                           access: COMMAND_SETTINGS.access.nonempty,
                           limits: COMMAND_SETTINGS.limits.nonempty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                        maxTries: 3,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );

         it(
            'stop policy HTTP: empty',
            check(
               spec => {
                  spec.stop_policy = {
                     http_get: {
                        // port: 80,
                        // path: 'stop path',
                        // expected_answer: 'stop answer',
                        // any: false,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.HTTP,
                        http: {
                           port: null,
                           path: null,
                           expectedAnswer: null,
                           any: false,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                        maxTries: null,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );

         it(
            'stop policy HTTP: advanced settings',
            check(
               spec => {
                  spec.stop_policy = {
                     http_get: {
                        port: 80,
                        path: 'stop path',
                        expected_answer: 'stop answer',
                        any: true,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                     max_tries: 3,
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.HTTP,
                        http: {
                           port: 80,
                           path: 'stop path',
                           expectedAnswer: 'stop answer',
                           any: true,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                        maxTries: 3,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );

         it(
            'stop policy unix_signal: empty',
            check(
               spec => {
                  spec.stop_policy = {
                     unix_signal: {},
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.UnixSignal,
                        unixSignal: {
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                        maxTries: null,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );

         it(
            'stop policy unix_signal: advanced settings',
            check(
               spec => {
                  spec.stop_policy = {
                     unix_signal: {
                        signal: EUnixSignalType.EUnixSignalType_SIGUSR2,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                     max_tries: 3,
                  };
               },
               expected => {
                  expected.commands = {
                     stop: {
                        mode: WorkloadStopMode.UnixSignal,
                        unixSignal: {
                           signal: EUnixSignalType.EUnixSignalType_SIGUSR2,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                        maxTries: 3,
                     },
                  };
               },
               model => model.commands.stop,
            ),
         );
      });

      describe('should extract destroy policy probe', () => {
         it(
            'destroy policy Exec: empty',
            check(
               spec => {
                  spec.destroy_policy = {
                     container: {
                        // command_line: 'Exec command',
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     destroy: {
                        mode: WorkloadDestroyMode.Exec,
                        exec: {
                           command: null,
                           access: COMMAND_SETTINGS.access.empty,
                           limits: COMMAND_SETTINGS.limits.empty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                        maxTries: null,
                     },
                  };
               },
               model => model.commands.destroy,
            ),
         );

         it(
            'destroy policy Exec: advanced settings',
            check(
               spec => {
                  spec.destroy_policy = {
                     container: {
                        command_line: 'Exec command',
                        user: COMMAND_SETTINGS.access.nonempty.user,
                        group: COMMAND_SETTINGS.access.nonempty.group,
                        compute_resources: COMMAND_SETTINGS.compute_resources,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                     max_tries: 3,
                  };
               },
               expected => {
                  expected.commands = {
                     destroy: {
                        mode: WorkloadDestroyMode.Exec,
                        exec: {
                           command: 'Exec command',
                           access: COMMAND_SETTINGS.access.nonempty,
                           limits: COMMAND_SETTINGS.limits.nonempty,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                        maxTries: 3,
                     },
                  };
               },
               model => model.commands.destroy,
            ),
         );

         it(
            'destroy policy HTTP: empty',
            check(
               spec => {
                  spec.destroy_policy = {
                     http_get: {
                        // port: 80,
                        // path: 'destroy path',
                        // expected_answer: 'destroy answer',
                        // any: false,
                     },
                  };
               },
               expected => {
                  expected.commands = {
                     destroy: {
                        mode: WorkloadDestroyMode.HTTP,
                        http: {
                           port: null,
                           path: null,
                           expectedAnswer: null,
                           any: false,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.empty,
                        },
                        maxTries: null,
                     },
                  };
               },
               model => model.commands.destroy,
            ),
         );

         it(
            'destroy policy HTTP: advanced settings',
            check(
               spec => {
                  spec.destroy_policy = {
                     http_get: {
                        port: 80,
                        path: 'destroy path',
                        expected_answer: 'destroy answer',
                        any: true,
                        time_limit: COMMAND_SETTINGS.time_limit,
                     },
                     max_tries: 3,
                  };
               },
               expected => {
                  expected.commands = {
                     destroy: {
                        mode: WorkloadDestroyMode.HTTP,
                        http: {
                           port: 80,
                           path: 'destroy path',
                           expectedAnswer: 'destroy answer',
                           any: true,
                           retryPolicy: COMMAND_SETTINGS.retryPolicy.nonempty,
                        },
                        maxTries: 3,
                     },
                  };
               },
               model => model.commands.destroy,
            ),
         );
      });
   });
});
