import { IJsonable, isEmpty } from '@yandex-infracloud-ui/libs';

import {
   convertEnvironment,
   EnvironmentVariable,
   getSecretUsagesFromEnvironment,
} from '../../../../modules/environment';
import {
   TCoredumpPolicy,
   TDeployUnitSpec,
   TDestroyPolicy,
   TLivenessCheck,
   TMonitoringUnistatEndpoint,
   TMonitoringUnistatEndpoint_EOutputFormat,
   TPodTemplateSpec,
   TReadinessCheck,
   TStopPolicy,
   TTimeLimit,
   TUtilityContainer,
   TWorkload,
} from '../../../../proto-typings';
import { getSecretUsagePath, SecretUsage } from '../../secrets';
import { convertYasmLabels } from '../yasm';
import {
   RetryPolicy,
   Workload,
   WorkloadCommands,
   WorkloadDestroyCommand,
   WorkloadDestroyMode,
   WorkloadExec,
   WorkloadHttp,
   WorkloadPortoYasm,
   WorkloadProbeCommand,
   WorkloadProbeMode,
   WorkloadStopCommand,
   WorkloadStopMode,
   WorkloadTcp,
   WorkloadUnistatYasm,
   WorkloadUnixSignal,
   WorkloadYasm,
} from './Workload';

export class WorkloadConverter implements Workload, IJsonable {
   public static fromApi(raw: TWorkload, podTemplateSpec: TPodTemplateSpec, deployUnitSpec: TDeployUnitSpec): Workload {
      return new WorkloadConverter(raw, podTemplateSpec, deployUnitSpec).toJSON();
   }

   public static getSecretUsages(boxId: string, workload: Workload, alias: string | null): SecretUsage[] {
      const environmentPathPrefix = getSecretUsagePath({
         field: 'environment',
         boxId,
         workloadId: workload.id,
      });

      return getSecretUsagesFromEnvironment(environmentPathPrefix, workload.environment, alias);
   }

   public commands: WorkloadCommands;

   public environment: EnvironmentVariable[];

   public id: string;

   public logs: boolean;

   public yasm: WorkloadYasm;

   public coredumpPolicy: TCoredumpPolicy | null;

   constructor(
      private raw: TWorkload,
      private podTemplateSpec: TPodTemplateSpec,
      private deployUnitSpec: TDeployUnitSpec,
   ) {
      this.id = this.raw.id;
      this.commands = this.getCommands();

      this.environment = convertEnvironment(this.raw.env);

      this.logs = this.raw.transmit_logs ?? false;
      this.yasm = this.parseYasm(this.podTemplateSpec);

      this.coredumpPolicy = this.getCoredumpPolicy();
   }

   public toJSON(): Workload {
      return {
         id: this.id,
         commands: this.commands,
         environment: this.environment,
         logs: this.logs,
         yasm: this.yasm,
         coredumpPolicy: this.coredumpPolicy,
      };
   }

   private getCoredumpPolicy(): TCoredumpPolicy | null {
      const coredumpConfig = this.deployUnitSpec?.coredump_config || {};
      if (Object.prototype.hasOwnProperty.call(coredumpConfig, this.id)) {
         return coredumpConfig[this.id] || null;
      }
      return null;
   }

   private getCommands(): WorkloadCommands {
      return {
         init: this.parseUtilityContainerList(this.raw.init),
         start: this.parseUtilityContainer(this.raw.start),
         liveness: this.parseProbe(this.raw.liveness_check),
         readiness: this.parseProbe(this.raw.readiness_check),
         stop: this.parseStop(this.raw.stop_policy),
         destroy: this.parseDestroy(this.raw.destroy_policy),
      };
   }

   private static parseTimeLimit(timeLimit?: TTimeLimit | undefined): RetryPolicy {
      return {
         initialDelayMs: timeLimit?.initial_delay_ms ?? null,
         restartPeriodScaleMs: timeLimit?.restart_period_scale_ms ?? null,
         restartPeriodBackOff: timeLimit?.restart_period_back_off ?? null,
         minRestartPeriodMs: timeLimit?.min_restart_period_ms ?? null,
         maxRestartPeriodMs: timeLimit?.max_restart_period_ms ?? null,
         maxExecutionTimeMs: timeLimit?.max_execution_time_ms ?? null,
      };
   }

   private parseProbe(rawCheck: TReadinessCheck | TLivenessCheck | undefined): WorkloadProbeCommand {
      let mode = WorkloadProbeMode.TCP;

      let tcp: WorkloadTcp | undefined = undefined;
      if (!rawCheck || rawCheck.tcp_check) {
         // mode = WorkloadProbeMode.TCP;
         tcp = {
            port: rawCheck?.tcp_check?.port ?? null,
            retryPolicy: WorkloadConverter.parseTimeLimit(rawCheck?.tcp_check?.time_limit),
         };
      }

      let http: WorkloadHttp | undefined = undefined;
      if (rawCheck?.http_get) {
         mode = WorkloadProbeMode.HTTP;
         http = {
            port: rawCheck.http_get.port ?? null,
            path: rawCheck.http_get.path ?? null,
            expectedAnswer: rawCheck.http_get.expected_answer ?? null,
            any: rawCheck.http_get.any ?? false,
            retryPolicy: WorkloadConverter.parseTimeLimit(rawCheck.http_get.time_limit),
         };
      }

      let exec: WorkloadExec | undefined = undefined;
      if (rawCheck?.container) {
         mode = WorkloadProbeMode.Exec;
         exec = this.parseUtilityContainer(rawCheck.container);
      }

      return {
         http,
         mode,
         tcp,
         exec,
      };
   }

   private parseStop(policy: TStopPolicy | undefined): WorkloadStopCommand {
      // если не задан ни один из параметров backend (http_get, unix_signal, container),
      // то проставляется именно unix_signal по дефолту (со значением signal = sigterm)
      let mode = WorkloadStopMode.UnixSignal;

      let exec: WorkloadExec | undefined = undefined;
      if (!policy || policy?.container) {
         mode = WorkloadStopMode.Exec;
         exec = this.parseUtilityContainer(policy?.container);
      }

      let http: WorkloadHttp | undefined = undefined;
      if (policy?.http_get) {
         mode = WorkloadStopMode.HTTP;
         http = {
            port: policy.http_get?.port ?? null,
            path: policy.http_get?.path ?? null,
            expectedAnswer: policy.http_get?.expected_answer ?? null,
            any: policy.http_get?.any ?? false,
            retryPolicy: WorkloadConverter.parseTimeLimit(policy.http_get?.time_limit),
         };
      }

      let unixSignal: WorkloadUnixSignal | undefined = undefined;
      if (policy?.unix_signal) {
         // mode = WorkloadStopMode.UnixSignal;
         unixSignal = {
            signal: policy.unix_signal?.signal,
            retryPolicy: WorkloadConverter.parseTimeLimit(policy.unix_signal?.time_limit),
         };
      }

      return {
         mode,
         exec,
         http,
         unixSignal,
         maxTries: policy?.max_tries ?? null,
      };
   }

   private parseDestroy(policy: TDestroyPolicy | undefined): WorkloadDestroyCommand {
      let mode = WorkloadDestroyMode.Exec;

      let exec: WorkloadExec | undefined = undefined;
      if (!policy || policy?.container) {
         // mode = WorkloadStopMode.Exec;
         exec = this.parseUtilityContainer(policy?.container);
      }

      let http: WorkloadHttp | undefined = undefined;
      if (policy?.http_get) {
         mode = WorkloadDestroyMode.HTTP;
         http = {
            port: policy.http_get?.port ?? null,
            path: policy.http_get?.path ?? null,
            expectedAnswer: policy.http_get?.expected_answer ?? null,
            any: policy.http_get?.any ?? false,
            retryPolicy: WorkloadConverter.parseTimeLimit(policy.http_get?.time_limit),
         };
      }

      return {
         mode,
         exec,
         http,
         maxTries: policy?.max_tries ?? null,
      };
   }

   private parseUtilityContainer(container?: TUtilityContainer | undefined, _order?: number | undefined): WorkloadExec {
      return {
         _order,
         command: container?.command_line ?? null,
         access: {
            group: container?.group ?? null,
            user: container?.user ?? null,
         },
         retryPolicy: WorkloadConverter.parseTimeLimit(container?.time_limit),
         limits: {
            cpuLimit: container?.compute_resources?.vcpu_limit ?? null,
            ramLimit: container?.compute_resources?.memory_limit ?? null,
         },
      };
   }

   private parseUtilityContainerList(containerList: TUtilityContainer[] | undefined): WorkloadExec[] {
      if (containerList && !isEmpty(containerList)) {
         return containerList.map((item, i) => this.parseUtilityContainer(item, i));
      }

      return [this.parseUtilityContainer()];
   }

   private parseYasm(podTemplateSpec: TPodTemplateSpec): WorkloadYasm {
      const yasmUnistatRecords: Partial<TMonitoringUnistatEndpoint>[] =
         podTemplateSpec.spec?.host_infra?.monitoring?.unistats?.filter(u => u.workload_id === this.id) ?? [];

      const unistatSettings: WorkloadUnistatYasm[] = yasmUnistatRecords.map(record => {
         const { path, port, labels, inherit_missed_labels, prefix, output_format } = record;
         const { itype, tags } = convertYasmLabels(labels);

         return {
            port: port ?? null,
            url: path ?? null,
            yasmTags: { itype, tags },
            inheritMissedLabels: Boolean(inherit_missed_labels),
            prefix: prefix ?? null,
            outputFormat:
               output_format && output_format !== TMonitoringUnistatEndpoint_EOutputFormat.OF_UNKNOWN
                  ? output_format
                  : null,
         };
      });

      const yasmPortoRecord = podTemplateSpec.spec?.host_infra?.monitoring?.workloads?.find(
         u => u.workload_id === this.id,
      );

      let portoSettings: WorkloadPortoYasm;

      if (!yasmPortoRecord) {
         portoSettings = {
            usePortoMetrics: false,
            inheritMissedLabels: false,
            yasmTags: { itype: null, tags: [] },
         };
      } else {
         portoSettings = {
            usePortoMetrics: true,
            inheritMissedLabels: yasmPortoRecord.inherit_missed_labels ?? false,
            yasmTags: convertYasmLabels(yasmPortoRecord.labels),
         };
      }

      return {
         unistats: unistatSettings,
         porto: portoSettings,
      };
   }
}
