/* eslint-disable no-template-curly-in-string */
import { isEmpty } from '@yandex-infracloud-ui/libs';
import { array, boolean, number, object, Schema, string } from 'yup';
import { ValidationContext } from '../../../../components/huge-form';

import { environmentValidationSchema } from '../../../../modules/environment/models';
import {
   EUnixSignalType,
   TCoredumpAggregator,
   TMonitoringUnistatEndpoint_EOutputFormat,
} from '../../../../proto-typings';
import { createKey, getFilledValue, getPaths, isClone } from '../../../../utils';
import { portSchema, StringNullableSchema, workloadIdSchema } from '../validation';
import { yasmTagsValidationSchema } from '../yasm';

import {
   getEmptyWorkload,
   isStopEmpty,
   RetryPolicy,
   Workload,
   WorkloadCommandAccess,
   WorkloadCommandLimits,
   WorkloadCommands,
   workloadDefaultValues,
   WorkloadDestroyCommand,
   WorkloadDestroyMode,
   WorkloadExec,
   WorkloadHttp,
   WorkloadPortoYasm,
   WorkloadProbeCommand,
   WorkloadProbeMode,
   WorkloadStopCommand,
   WorkloadStopMode,
   WorkloadTcp,
   WorkloadUnistatYasm,
   WorkloadUnixSignal,
   WorkloadYasm,
} from './Workload';

export interface CoredumpAggregatorFormParams {
   enabled: boolean;
   url?: string;
   serviceName?: string;
   ctype?: string;
}

export interface CoredumpPolicyFormParams {
   enabled: boolean;
   countLimit?: number;
   totalSizeLimitMegabytes?: number;
   probability?: number;
   cleanupTtlSeconds?: number;
   volumeId: string | null;

   aggregator?: CoredumpAggregatorFormParams;

   output?: string;
}

export type WorkloadUnistatYasmParams = WorkloadUnistatYasm & {
   form: {
      old: boolean;
      removed: boolean;
      index: number;
   };
};

export type WorkloadYasmParams = Omit<WorkloadYasm, 'unistats'> & {
   unistats: WorkloadUnistatYasmParams[];
};

export type WorkloadFormParams = Omit<Workload, 'coredumpPolicy' | 'yasm'> & {
   coredumpPolicy: CoredumpPolicyFormParams;
   yasm: WorkloadYasmParams;
};

export const workloadPaths = getPaths<WorkloadFormParams>();

export const workloadYasmUnistatValidationSchema = object<WorkloadUnistatYasmParams>({
   url: StringNullableSchema('Unistat path').matches(/^\/.*$/, '${path} must start with a character `/`'),
   port: portSchema,
   yasmTags: yasmTagsValidationSchema,
   inheritMissedLabels: boolean(),
   outputFormat: string()
      .nullable()
      .oneOf([
         TMonitoringUnistatEndpoint_EOutputFormat.OF_YASM_JSON,
         TMonitoringUnistatEndpoint_EOutputFormat.OF_YASM_PROTO,
         TMonitoringUnistatEndpoint_EOutputFormat.OF_SOLOMON,
         null,
      ]) as Schema<WorkloadUnistatYasm['outputFormat'] | null>,
   prefix: string().nullable(),
   form: object({
      old: boolean(),
      removed: boolean(),
      index: number().test('uniq', 'Unique path and port', function check(index: number) {
         const context = this.options.context as ValidationContext | undefined;
         if (!context) {
            return true;
         }
         const { unistats } = ((context.form as unknown) as WorkloadFormParams).yasm;
         const unistat = unistats[index];
         if (!unistat || unistat.form.removed) {
            return true;
         }
         const { path, port } = workloadDefaultValues.yasm.unistat;

         const clone = isClone(
            unistats,
            e =>
               createKey({
                  path: getFilledValue(e.url, '', path),
                  port: String(getFilledValue(e.port, null, port)),
               }),
            index,
         );

         return !clone;
      }),
   }),
});

export const workloadYasmPortoValidationSchema = object<WorkloadPortoYasm>({
   usePortoMetrics: boolean(),
   inheritMissedLabels: boolean(),
   yasmTags: yasmTagsValidationSchema,
});

export const workloadYasmValidationSchema = object<WorkloadYasmParams>({
   unistats: array(workloadYasmUnistatValidationSchema),
   porto: workloadYasmPortoValidationSchema,
});

const commandAccessValidationSchema = object<WorkloadCommandAccess>({
   user: string().label('User').nullable(),
   group: string().label('Group').nullable(),
});

const commandLimitsValidationSchema = object<WorkloadCommandLimits>({
   cpuLimit: number().label('CPU limit').nullable(),
   ramLimit: number().label('RAM limit').nullable(),
});

const retryPolicyValidationSchema = object<RetryPolicy>({
   initialDelayMs: number().label('Initial delay').positive().nullable(),
   maxExecutionTimeMs: number().label('Max execution').positive().nullable(),
   maxRestartPeriodMs: number().label('Max restart period').positive().nullable(),
   minRestartPeriodMs: number().label('Min restart  Period').positive().nullable(),
   restartPeriodBackOff: number().label('Restart period backoff').positive().nullable(),
   restartPeriodScaleMs: number().label('Restart period scale').positive().nullable(),
});

const commandLineValidationSchema = string()
   .test('command-space', 'Command line begins/ends with space character', value => {
      if (!isEmpty(value) && value !== value.trim()) {
         return false;
      }

      return true;
   })
   .label('Command line')
   .nullable();

const execValidationSchema = object<WorkloadExec>({
   _order: number().min(0),
   command: commandLineValidationSchema,
   access: commandAccessValidationSchema,
   limits: commandLimitsValidationSchema,
   retryPolicy: retryPolicyValidationSchema,
});

const httpValidationSchema = object<WorkloadHttp>({
   port: portSchema,
   path: string().label('Path').nullable(),
   expectedAnswer: string().label('Expected answer').nullable(),
   any: boolean(),
   retryPolicy: retryPolicyValidationSchema,
});

export const signalSchema = string()
   .oneOf([
      EUnixSignalType.EUnixSignalType_DEFAULT,
      EUnixSignalType.EUnixSignalType_SIGALRM,
      EUnixSignalType.EUnixSignalType_SIGHUP,
      EUnixSignalType.EUnixSignalType_SIGINT,
      EUnixSignalType.EUnixSignalType_SIGKILL,
      EUnixSignalType.EUnixSignalType_SIGQUIT,
      EUnixSignalType.EUnixSignalType_SIGTERM,
      EUnixSignalType.EUnixSignalType_SIGURG,
      EUnixSignalType.EUnixSignalType_SIGUSR1,
      EUnixSignalType.EUnixSignalType_SIGUSR2,
      null,
   ])
   .label('Signal')
   .nullable() as Schema<EUnixSignalType>;

const unixSignalValidationSchema = object<WorkloadUnixSignal>({
   signal: signalSchema,
   retryPolicy: retryPolicyValidationSchema,
});

const tcpValidationSchema = object<WorkloadTcp>({
   port: portSchema,
   retryPolicy: retryPolicyValidationSchema,
});

const probeValidationSchema = object<WorkloadProbeCommand>({
   mode: string()
      .label('Mode')
      .oneOf([WorkloadProbeMode.Exec, WorkloadProbeMode.HTTP, WorkloadProbeMode.TCP] as string[])
      .required() as Schema<WorkloadProbeMode>,
   exec: object<WorkloadExec>().when('mode', {
      is: WorkloadProbeMode.Exec,
      then: execValidationSchema,
      otherwise: object(),
   }),
   http: object<WorkloadHttp>().when('mode', {
      is: WorkloadProbeMode.HTTP,
      then: httpValidationSchema,
      otherwise: object(),
   }),
   tcp: object<WorkloadTcp>().when('mode', {
      is: WorkloadProbeMode.TCP,
      then: tcpValidationSchema,
      otherwise: object(),
   }),
});

const stopValidationSchema = object<WorkloadStopCommand>({
   mode: string()
      .label('Mode')
      .oneOf([WorkloadStopMode.Exec, WorkloadStopMode.HTTP, WorkloadStopMode.UnixSignal] as string[])
      .required() as Schema<WorkloadStopMode>,
   exec: object<WorkloadExec>().when('mode', {
      is: WorkloadStopMode.Exec,
      then: execValidationSchema,
      otherwise: object(),
   }),
   http: object<WorkloadHttp>().when('mode', {
      is: WorkloadStopMode.HTTP,
      then: httpValidationSchema,
      otherwise: object(),
   }),
   unixSignal: object<WorkloadUnixSignal>().when('mode', {
      is: WorkloadStopMode.UnixSignal,
      then: unixSignalValidationSchema,
      otherwise: object(),
   }),
   maxTries: number().label('Max tries').nullable(),
}).test('max-tries', 'Max tries must be greater than 0', value => {
   const empty = isStopEmpty(value);

   if (!empty) {
      if (!(value.maxTries && value.maxTries > 0)) {
         return false;
      }
   }

   return true;
});

const destroyValidationSchema = object<WorkloadDestroyCommand>({
   mode: string()
      .label('Mode')
      .oneOf([WorkloadDestroyMode.Exec, WorkloadDestroyMode.HTTP] as string[])
      .required() as Schema<WorkloadDestroyMode>,
   exec: object<WorkloadExec>().when('mode', {
      is: WorkloadDestroyMode.Exec,
      then: execValidationSchema,
      otherwise: object(),
   }),
   http: object<WorkloadHttp>().when('mode', {
      is: WorkloadDestroyMode.HTTP,
      then: httpValidationSchema,
      otherwise: object(),
   }),
   maxTries: number().label('Max tries').nullable(),
}).test('max-tries', 'Max tries must be greater than 0', value => {
   const empty = isStopEmpty(value);

   return empty || value.maxTries > 0;
});

const workloadCommandsValidationSchema = object<WorkloadCommands>({
   start: execValidationSchema,
   init: array().of(execValidationSchema),
   liveness: probeValidationSchema,
   readiness: probeValidationSchema,
   stop: stopValidationSchema,
   destroy: destroyValidationSchema,
});

export const coredumpPolicyValidationSchema = object<CoredumpPolicyFormParams>({
   enabled: boolean().required(),
   countLimit: number().label('Limit').required().integer().positive(),
   totalSizeLimitMegabytes: number().label('Size').required().integer().positive(),
   probability: number().label('Probability').min(0).max(100),
   cleanupTtlSeconds: number().label('Seconds').integer().min(0),
   output: string(), // TODO: nullable()
   volumeId: string().nullable(),
   aggregator: object<CoredumpAggregatorFormParams>({
      enabled: boolean().required(),
      url: string().label('URL').url(), // TODO: nullable()
      serviceName: string(), // TODO: nullable()
      ctype: string(), // TODO: nullable()
   }),
});

export const workloadValidationSchema = object<WorkloadFormParams>({
   id: workloadIdSchema,
   commands: workloadCommandsValidationSchema,
   environment: environmentValidationSchema,
   logs: boolean(),
   yasm: workloadYasmValidationSchema,
   coredumpPolicy: coredumpPolicyValidationSchema,
});

export function workloadToFormParams(item = getEmptyWorkload()): WorkloadFormParams {
   const { coredumpPolicy, yasm, ...params } = item;
   const formParams = {
      ...params,
   } as WorkloadFormParams;

   formParams.yasm = {
      ...yasm,
      unistats: yasm.unistats.map((e, i) => ({ ...e, form: { old: true, removed: false, index: i } })),
   };

   const {
      coredump_processor: {
         count_limit: countLimit = 2,
         total_size_limit_megabytes: totalSizeLimitMegabytes = 500,
         probability = 100,
         cleanup_ttl_seconds: cleanupTtlSeconds = 0,
         output = '',
         aggregator = {} as TCoredumpAggregator,
         volume_id: volumeId = null,
      } = {},
   } = coredumpPolicy || {};

   formParams.coredumpPolicy = {
      enabled: Boolean(coredumpPolicy),
      countLimit,
      totalSizeLimitMegabytes,
      probability,
      cleanupTtlSeconds,
      output,
      volumeId,
   };

   const { enabled, url = '', service_name: serviceName = '', ctype = '' } = aggregator;

   formParams.coredumpPolicy.aggregator = {
      enabled: Boolean(enabled),
      url,
      serviceName,
      ctype,
   };

   return formParams;
}

export function formParamsToWorkload(params: WorkloadFormParams, old: Workload): Workload {
   const { coredumpPolicy, yasm, ...fields } = params;

   const workload = {
      ...old,
      ...fields,
   };

   workload.yasm = {
      ...yasm,
      unistats: yasm.unistats
         .filter(e => !e.form.removed)
         .map(e => {
            const { form, ...unistat } = e;
            return unistat;
         }),
   };

   // TODO khoden: Почему это здесь, а не в патчере?
   if (coredumpPolicy.enabled) {
      const {
         countLimit,
         totalSizeLimitMegabytes,
         probability,
         cleanupTtlSeconds,
         output,
         aggregator,
      } = coredumpPolicy;

      workload.coredumpPolicy = {
         coredump_processor: {
            count_limit: countLimit,
            total_size_limit_megabytes: totalSizeLimitMegabytes,
            probability,
            cleanup_ttl_seconds: cleanupTtlSeconds,
            output: output || '',
            aggregator: aggregator
               ? {
                    enabled: aggregator.enabled || false,
                    url: aggregator.url || '',
                    service_name: aggregator.serviceName || '',
                    ctype: aggregator.ctype || '',
                 }
               : undefined,
         },
      };
   } else {
      workload.coredumpPolicy = null;
   }

   return workload;
}
