import { EnvironmentVariableType } from '../../../../modules/environment/models';
import { EUnixSignalType, TDeployUnitSpec, TPodTemplateSpec, TWorkload } from '../../../../proto-typings';
import { noop } from '../../../../utils';
import { DeepPartial } from '../../../typeHelpers';
import { StageParentNodeIds } from '../StageParentNodeIds';
import { StagePatcherVisitor } from '../StagePatcherVisitor';

import { RetryPolicy, Workload, WorkloadDestroyMode, WorkloadProbeMode, WorkloadStopMode } from './Workload';
import { WorkloadConverter } from './WorkloadConverter';
import { WorkloadPatcher } from './WorkloadPatcher';

const getInitialWorkloadSpec = (): DeepPartial<TWorkload> => ({
   id: 'workload',
});

const getInitialPodTemplateSpec = (): DeepPartial<TPodTemplateSpec> => ({
   spec: {
      host_infra: {
         monitoring: {},
      },
      pod_agent_payload: {
         spec: {
            // ...
            workloads: [getInitialWorkloadSpec()],
         },
      },
   },
});

const getInitialDeployUnitSpec = (): DeepPartial<TDeployUnitSpec> => ({});

/**
 * @example
 * it('should xxx', check(
 *    spec => {
 *
 *    },
 *    w => {
 *
 *    },
 *    expected => {
 *
 *    },
 *    spec => {
 *
 *    }
 *  ))
 */
const check = (
   patchSpecBefore: (s: DeepPartial<TWorkload>) => void,
   patchModel: (m: Workload) => void,
   patchExpectedSpec: (s: DeepPartial<TWorkload>) => void,
   checkOnlyBranch?: (s: TWorkload) => any,
) => () => {
   const visitor = new StagePatcherVisitor();
   const spec = getInitialWorkloadSpec() as TWorkload;
   const podTemplateSpec = getInitialPodTemplateSpec() as TPodTemplateSpec;
   const deployUnitSpec = getInitialDeployUnitSpec() as TDeployUnitSpec;

   patchSpecBefore(spec);

   const model = WorkloadConverter.fromApi(spec, podTemplateSpec, deployUnitSpec);
   patchModel(model);

   const expectedPatchedSpec = getInitialWorkloadSpec();
   patchExpectedSpec(expectedPatchedSpec);

   const patchedSpec = WorkloadPatcher.patch(
      visitor,
      new StageParentNodeIds({ id: 'test-stage' }, { id: 'test-du' }, { id: 'test-box' }),
      spec,
      model,
      'Box',
   );

   if (checkOnlyBranch) {
      expect(checkOnlyBranch(patchedSpec)).toEqual(checkOnlyBranch(expectedPatchedSpec as TWorkload));
   } else {
      expect(patchedSpec).toEqual(expectedPatchedSpec);
   }
};

describe('models/ui|WorkloadPatcher', () => {
   it(
      'should patch logs: enable',
      check(
         noop,
         w => {
            w.logs = true;
         },
         expected => {
            expected.transmit_logs = true;
         },
         spec => spec.transmit_logs,
      ),
   );

   it(
      'should patch logs: disable',
      check(
         spec => {
            spec.transmit_logs = true;
         },
         w => {
            w.logs = false;
         },
         expected => {
            expected.transmit_logs = false;
         },
         spec => spec.transmit_logs,
      ),
   );

   it(
      'should patch env: add',
      check(
         noop,
         w => {
            w.environment = [
               {
                  name: 'cba',
                  type: EnvironmentVariableType.Secret,
                  value: {
                     alias: 'sec-123:ver-123',
                     key: 'secret',
                  },
               },
               {
                  name: 'abc',
                  type: EnvironmentVariableType.Literal,
                  value: 'literal',
               },
            ];
         },
         expected => {
            expected.env = [
               {
                  name: 'cba',
                  value: {
                     secret_env: {
                        alias: 'sec-123:ver-123',
                        id: 'secret',
                     },
                  },
               },
               {
                  name: 'abc',
                  value: {
                     literal_env: {
                        value: 'literal',
                     },
                  },
               },
            ];
         },
         spec => spec.env,
      ),
   );

   it(
      'should patch env: edit',
      check(
         spec => {
            // noinspection SpellCheckingInspection
            spec.env = [
               {
                  name: 'OLD',
                  value: {
                     literal_env: {
                        value: 'renamed',
                     },
                  },
               },
               {
                  name: 'cba',
                  value: {
                     secret_env: {
                        alias: 'sec-123:ver-123',
                        id: 'secret',
                     },
                  },
               },
               {
                  name: 'abc',
                  value: {
                     literal_env: {
                        value: 'literal',
                     },
                  },
               },
               {
                  name: 'abcd',
                  value: {
                     literal_env: {
                        value: 'literal2',
                     },
                  },
               },
            ];
         },
         w => {
            // noinspection SpellCheckingInspection
            w.environment = [
               {
                  name: 'NEW',
                  type: EnvironmentVariableType.Literal,
                  value: 'renamed',
               },
               {
                  name: 'cba',
                  type: EnvironmentVariableType.Literal,
                  value: 'literal1',
               },
               {
                  name: 'abcd',
                  type: EnvironmentVariableType.Literal,
                  value: 'literal2',
               },
               {
                  name: 'abc',
                  type: EnvironmentVariableType.Secret,
                  value: {
                     alias: 'sec-1234:ver-123',
                     key: 'secret_new',
                  },
               },
               {
                  name: 'abcde',
                  type: EnvironmentVariableType.Literal,
                  value: 'new env',
               },
            ];
         },
         expected => {
            // noinspection SpellCheckingInspection
            expected.env = [
               {
                  name: 'cba',
                  value: {
                     literal_env: {
                        value: 'literal1',
                     },
                  },
               },
               {
                  name: 'abc',
                  value: {
                     secret_env: {
                        alias: 'sec-1234:ver-123',
                        id: 'secret_new',
                     },
                  },
               },
               {
                  name: 'abcd',
                  value: {
                     literal_env: {
                        value: 'literal2',
                     },
                  },
               },
               {
                  name: 'NEW',
                  value: {
                     literal_env: {
                        value: 'renamed',
                     },
                  },
               },
               {
                  name: 'abcde',
                  value: {
                     literal_env: {
                        value: 'new env',
                     },
                  },
               },
            ];
         },
         spec => spec.env,
      ),
   );

   it(
      'should patch env: clear',
      check(
         spec => {
            spec.env = [
               {
                  name: 'cba',
                  value: {
                     secret_env: {
                        alias: 'sec-123:ver-123',
                        id: 'secret',
                     },
                  },
               },
               {
                  name: 'abc',
                  value: {
                     literal_env: {
                        value: 'literal',
                     },
                  },
               },
            ];
         },
         w => {
            w.environment = [];
         },
         expected => {
            expected.env = undefined;
         },
         spec => spec.env,
      ),
   );

   it(
      'should patch commands: add',
      check(
         noop,
         w => {
            w.commands.start = {
               command: 'start',
            };
            w.commands.init = [
               {
                  command: 'init1',
               },
               {
                  command: 'init2',
               },
            ];
            w.commands.liveness = {
               mode: WorkloadProbeMode.Exec,
               exec: {
                  command: 'liveness',
               },
            };
            w.commands.readiness = {
               mode: WorkloadProbeMode.TCP,
               tcp: {
                  port: 88,
               },
            };
            w.commands.destroy = {
               mode: WorkloadDestroyMode.HTTP,
               maxTries: null,
               http: {
                  port: 1234,
                  path: 'destroy',
                  expectedAnswer: 'answer',
                  any: false,
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.start = {
               command_line: 'start',
            };

            expected.init = [
               {
                  command_line: 'init1',
               },
               {
                  command_line: 'init2',
               },
            ];

            expected.liveness_check = {
               container: {
                  command_line: 'liveness',
               },
            };

            expected.readiness_check = {
               tcp_check: {
                  port: 88,
               },
            };

            expected.destroy_policy = {
               http_get: {
                  port: 1234,
                  path: 'destroy',
                  expected_answer: 'answer',
               },
            };
         },
      ),
   );

   it(
      'should patch commands: edit',
      check(
         spec => {
            spec.start = {
               command_line: 'start',
            };

            spec.init = [
               {
                  command_line: 'init1',
               },
               {
                  command_line: 'init2',
                  user: 'init2',
                  group: 'init2',
               },
            ];

            spec.liveness_check = {
               container: {
                  command_line: 'liveness',
                  user: 'liveness',
                  group: 'liveness',
               },
            };

            spec.readiness_check = {
               tcp_check: {
                  port: 88,
               },
            };

            spec.destroy_policy = {
               http_get: {
                  port: 1234,
                  path: 'destroy',
                  expected_answer: 'answer',
               },
            };
         },
         w => {
            w.commands.start = {
               command: 'start new',
               access: {
                  user: 'start',
                  group: 'start',
               },
               limits: {
                  cpuLimit: 100,
                  ramLimit: 1073741824,
               },
               retryPolicy: {
                  initialDelayMs: 5000,
                  maxRestartPeriodMs: 99999999999999,
                  minRestartPeriodMs: 30000,
                  maxExecutionTimeMs: 1800000,
                  restartPeriodScaleMs: 60000,
                  restartPeriodBackOff: 120000,
               },
            };
            w.commands.init = [
               {
                  command: 'init1 new',
                  access: {
                     user: 'init1_new',
                     group: 'init1_new',
                  },
               },
               {
                  command: 'init2',
                  access: {
                     user: 'init2_new',
                     group: 'init2_new',
                  },
                  retryPolicy: {
                     initialDelayMs: 5000,
                     restartPeriodScaleMs: 60000,
                     restartPeriodBackOff: 120000,
                  } as RetryPolicy,
               },
               {
                  command: 'init4',
                  retryPolicy: {
                     initialDelayMs: 5000,
                  } as RetryPolicy,
               },
            ];
            w.commands.liveness = {
               mode: WorkloadProbeMode.TCP,
               tcp: {
                  port: 88,
               },
            };
            w.commands.readiness = {
               mode: WorkloadProbeMode.Exec,
               exec: {
                  command: 'readiness',
               },
            };
            w.commands.destroy = {
               mode: WorkloadDestroyMode.Exec,
               maxTries: null,
               exec: {
                  command: 'destroy',
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.start = {
               command_line: 'start new',
               user: 'start',
               group: 'start',
               compute_resources: {
                  memory_limit: 1073741824,
                  vcpu_limit: 100,
               },
               time_limit: {
                  initial_delay_ms: 5000,
                  max_restart_period_ms: 99999999999999,
                  min_restart_period_ms: 30000,
                  max_execution_time_ms: 1800000,
                  restart_period_scale_ms: 60000,
                  restart_period_back_off: 120000,
               },
            };

            expected.init = [
               {
                  command_line: 'init1 new',
                  user: 'init1_new',
                  group: 'init1_new',
               },
               {
                  command_line: 'init2',
                  user: 'init2_new',
                  group: 'init2_new',
                  time_limit: {
                     initial_delay_ms: 5000,
                     restart_period_scale_ms: 60000,
                     restart_period_back_off: 120000,
                  },
               },
               {
                  command_line: 'init4',
                  time_limit: {
                     initial_delay_ms: 5000,
                  },
               },
            ];

            expected.liveness_check = {
               tcp_check: {
                  port: 88,
               },
            };

            expected.readiness_check = {
               container: {
                  command_line: 'readiness',
               },
            };

            expected.destroy_policy = {
               container: {
                  command_line: 'destroy',
               },
            };
         },
      ),
   );

   it(
      'should patch commands: clear',
      check(
         spec => {
            spec.start = {
               command_line: 'start',
               user: 'start',
               group: 'start',
               compute_resources: {
                  memory_limit: 1073741824,
                  vcpu_limit: 100,
               },
               time_limit: {
                  initial_delay_ms: 5000,
                  max_restart_period_ms: 99999999999999,
                  min_restart_period_ms: 30000,
                  max_execution_time_ms: 1800000,
                  restart_period_scale_ms: 60000,
                  restart_period_back_off: 120000,
               },
            };

            spec.init = [
               {
                  command_line: 'init1',
               },
               {
                  command_line: 'init2',
                  user: 'init2',
                  group: 'init2',
                  compute_resources: {
                     memory_limit: 1073741824,
                     vcpu_limit: 100,
                  },
                  time_limit: {
                     initial_delay_ms: 5000,
                     max_restart_period_ms: 99999999999999,
                     min_restart_period_ms: 30000,
                     max_execution_time_ms: 1800000,
                     restart_period_scale_ms: 60000,
                     restart_period_back_off: 120000,
                  },
               },
            ];

            spec.liveness_check = {
               container: {
                  command_line: 'liveness',
                  user: 'liveness',
                  group: 'liveness',
                  compute_resources: {
                     memory_limit: 1073741824,
                     vcpu_limit: 100,
                  },
                  time_limit: {
                     initial_delay_ms: 5000,
                     max_restart_period_ms: 99999999999999,
                     min_restart_period_ms: 30000,
                     max_execution_time_ms: 1800000,
                     restart_period_scale_ms: 60000,
                     restart_period_back_off: 120000,
                  },
               },
            };

            spec.readiness_check = {
               tcp_check: {
                  port: 88,
                  time_limit: {
                     initial_delay_ms: 5000,
                     max_restart_period_ms: 99999999999999,
                     min_restart_period_ms: 30000,
                     max_execution_time_ms: 1800000,
                     restart_period_scale_ms: 60000,
                     restart_period_back_off: 120000,
                  },
               },
            };

            spec.destroy_policy = {
               http_get: {
                  port: 1234,
                  path: 'destroy',
                  expected_answer: 'answer',
                  time_limit: {
                     initial_delay_ms: 5000,
                     max_restart_period_ms: 99999999999999,
                     min_restart_period_ms: 30000,
                     max_execution_time_ms: 1800000,
                     restart_period_scale_ms: 60000,
                     restart_period_back_off: 120000,
                  },
               },
            };
         },
         w => {
            w.commands.start = {
               command: '',
               access: {
                  user: null,
                  group: null,
               },
               limits: {
                  cpuLimit: null,
                  ramLimit: null,
               },
               retryPolicy: {
                  initialDelayMs: null,
                  restartPeriodScaleMs: null,
                  restartPeriodBackOff: null,
                  maxRestartPeriodMs: null,
                  minRestartPeriodMs: null,
                  maxExecutionTimeMs: null,
               },
            };
            w.commands.init = [
               {
                  command: '',
               },
            ];
            w.commands.liveness = {
               mode: WorkloadProbeMode.TCP,
               tcp: {
                  port: null,
               },
            };
            w.commands.readiness = {
               mode: WorkloadProbeMode.Exec,
               exec: {
                  command: '',
               },
            };
            w.commands.destroy = {
               mode: WorkloadDestroyMode.Exec,
               maxTries: null,
               exec: {
                  command: '',
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';
            expected.start = {
               compute_resources: {},
               time_limit: {},
            };
            expected.init = [];
            expected.liveness_check = {};
            expected.readiness_check = {};
            expected.destroy_policy = undefined;
         },
      ),
   );

   it(
      'should patch stop command http: add',
      check(
         noop,
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.HTTP,
               maxTries: null,
               http: {
                  port: 123,
                  path: 'stop',
                  expectedAnswer: 'any',
                  any: true,
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               http_get: {
                  port: 123,
                  path: 'stop',
                  any: true,
               },
            };
         },
      ),
   );

   it(
      'should patch stop command http: edit',
      check(
         spec => {
            spec.stop_policy = {
               http_get: {
                  port: 1234,
                  path: 'destroy',
                  expected_answer: 'answer',
               },
            };
         },
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.Exec,
               maxTries: null,
               exec: {
                  command: 'stop',
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               container: {
                  command_line: 'stop',
               },
            };
         },
      ),
   );

   it(
      'should patch stop command http: clear',
      check(
         spec => {
            spec.stop_policy = {
               http_get: {
                  port: 1234,
                  path: 'destroy',
                  expected_answer: 'answer',
               },
            };
         },
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.UnixSignal,
               maxTries: null,
               http: {
                  port: null,
                  path: '',
                  expectedAnswer: '',
                  any: false,
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = undefined;
         },
      ),
   );

   it(
      'should patch stop command exec: add',
      check(
         noop,
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.Exec,
               maxTries: null,
               exec: {
                  command: 'stop',
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               container: {
                  command_line: 'stop',
               },
            };
         },
      ),
   );

   it(
      'should patch stop command exec: edit',
      check(
         spec => {
            spec.stop_policy = {
               container: {
                  command_line: 'stop',
               },
            };
         },
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.UnixSignal,
               maxTries: null,
               unixSignal: {
                  signal: EUnixSignalType.EUnixSignalType_DEFAULT,
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               unix_signal: {
                  signal: EUnixSignalType.EUnixSignalType_DEFAULT,
               },
            };
         },
      ),
   );

   it(
      'should patch stop command exec: clear',
      check(
         spec => {
            spec.stop_policy = {
               container: {
                  command_line: 'stop',
               },
            };
         },
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.Exec,
               maxTries: null,
               exec: {
                  command: '',
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = undefined;
         },
      ),
   );

   it(
      'should patch stop command unixSignal: max_tries',
      check(
         noop,
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.UnixSignal,
               maxTries: 3,
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               max_tries: 3,
            };
         },
      ),
   );

   it(
      'should patch stop command unixSignal: add',
      check(
         noop,
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.UnixSignal,
               maxTries: null,
               unixSignal: {
                  signal: EUnixSignalType.EUnixSignalType_DEFAULT,
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               unix_signal: {
                  signal: EUnixSignalType.EUnixSignalType_DEFAULT,
               },
            };
         },
      ),
   );

   it(
      'should patch stop command unixSignal: edit',
      check(
         spec => {
            spec.stop_policy = {
               unix_signal: {
                  signal: EUnixSignalType.EUnixSignalType_DEFAULT,
               },
            };
         },
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.HTTP,
               maxTries: null,
               http: {
                  port: 123,
                  path: 'stop',
                  expectedAnswer: 'any',
                  any: true,
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = {
               http_get: {
                  port: 123,
                  path: 'stop',
                  any: true,
               },
            };
         },
      ),
   );

   it(
      'should patch stop command unixSignal: clear',
      check(
         spec => {
            spec.stop_policy = {
               unix_signal: {
                  signal: EUnixSignalType.EUnixSignalType_DEFAULT,
               },
            };
         },
         w => {
            w.commands.stop = {
               mode: WorkloadStopMode.UnixSignal,
               maxTries: null,

               exec: {
                  command: '',
               },
            };
         },
         expected => {
            expected.box_ref = 'Box';

            expected.stop_policy = undefined;
         },
      ),
   );
});
