import { deepClone, getSetDifference, isEmpty } from '@yandex-infracloud-ui/libs';

import { patchEnvironment } from '../../../../modules/environment';
import {
   TDestroyPolicy,
   THttpGet,
   TLivenessCheck,
   TReadinessCheck,
   TStopPolicy,
   TTcpCheck,
   TTimeLimit,
   TUnixSignal,
   TUtilityContainer,
   TWorkload,
} from '../../../../proto-typings';
import { patchBoolean, patchList, patchNumber, patchObject, patchString, skipEmpty } from '../../utils';

import { StageParentNodeIds } from '../StageParentNodeIds';
import { StagePatcherVisitor } from '../StagePatcherVisitor';

import {
   RetryPolicy,
   Workload,
   WorkloadDestroyCommand,
   WorkloadDestroyMode,
   WorkloadExec,
   WorkloadHttp,
   WorkloadProbeCommand,
   WorkloadProbeMode,
   WorkloadStopCommand,
   WorkloadStopMode,
   WorkloadTcp,
   WorkloadUnixSignal,
} from './Workload';

export class WorkloadPatcher {
   public static patch(
      visitor: StagePatcherVisitor,
      parentNodeIds: StageParentNodeIds,
      raw: TWorkload | undefined,
      workload: Workload,
      boxRef: string,
   ): TWorkload {
      return new WorkloadPatcher(visitor, parentNodeIds, raw, workload, boxRef).toValue();
   }

   constructor(
      private visitor: StagePatcherVisitor,
      private parentNodeIds: StageParentNodeIds,
      private raw: TWorkload | undefined,
      private workload: Workload,
      private boxRef: string,
   ) {}

   private patchExecCommand(exec: WorkloadExec, spec: TUtilityContainer) {
      patchString(spec, 'command_line', () => exec.command?.trim());

      patchString(spec, 'user', () => exec.access?.user);
      patchString(spec, 'group', () => exec.access?.group);

      patchObject(spec, 'compute_resources', computeResources => {
         patchNumber(computeResources, 'vcpu_limit', () => exec.limits?.cpuLimit);
         patchNumber(computeResources, 'memory_limit', () => exec.limits?.ramLimit);

         return computeResources;
      });

      patchObject(spec, 'time_limit', timeLimit => this.patchTimeLimit(exec.retryPolicy, timeLimit));

      return spec;
   }

   private patchExecCommandList(execList: WorkloadExec[], spec: TUtilityContainer[]): TUtilityContainer[] {
      /* eslint-disable no-underscore-dangle */
      const { removed } = getSetDifference(
         new Set(spec.map((v, i) => i).reverse()),
         new Set(
            execList
               ?.filter(v => v._order !== undefined)
               .map(v => v._order)
               .reverse(),
         ),
      );

      // для патчинга существующих сложноидентифицируемых объектов в массивах используем _order
      // в клонированных формах _order (индекс в спеке) есть, а исходной спеки нет
      // если спека пустая, то патчить нечего, всегда добавляем новые элементы в массив
      const isEmptySpec = isEmpty(spec);

      if (!isEmpty(execList)) {
         for (const command of execList) {
            const exist = command._order !== undefined && !isEmptySpec ? spec[command._order] : undefined;

            if (!exist) {
               spec.push({ command_line: command.command } as TUtilityContainer);
            }

            const l = isEmpty(spec) ? 0 : spec.length - 1;
            const i = command._order === undefined || !exist ? l : command._order;

            spec[i] = this.patchExecCommand(command, spec[i]);
         }
      }

      removed.forEach(i => {
         if (i !== undefined && spec[i]) {
            spec.splice(i, 1);
         }
      });

      return spec;
      /* eslint-enable no-underscore-dangle */
   }

   private patchHttpGet(http: WorkloadHttp, httpGet: THttpGet) {
      patchNumber(httpGet, 'port', () => http.port);
      patchString(httpGet, 'path', () => http.path);

      if (http.any) {
         if (httpGet.expected_answer) {
            delete httpGet.expected_answer;
         }

         patchBoolean(httpGet, 'any', () => http.any);
      } else {
         if (httpGet.any) {
            delete httpGet.any;
         }

         patchString(httpGet, 'expected_answer', () => http.expectedAnswer);
      }

      patchObject(httpGet, 'time_limit', timeLimit => this.patchTimeLimit(http.retryPolicy, timeLimit));

      return httpGet;
   }

   private patchUnixSignal(item: WorkloadUnixSignal, spec: TUnixSignal) {
      patchString(spec, 'signal', () => item.signal);
      patchObject(spec, 'time_limit', timeLimit => this.patchTimeLimit(item.retryPolicy, timeLimit));

      return spec;
   }

   private patchProbe(item: WorkloadProbeCommand, spec: TReadinessCheck | TLivenessCheck) {
      if (item.mode !== WorkloadProbeMode.TCP) {
         delete spec.tcp_check;
      }
      if (item.mode !== WorkloadProbeMode.HTTP) {
         delete spec.http_get;
      }
      if (item.mode !== WorkloadProbeMode.Exec) {
         delete spec.container;
      }

      switch (item.mode) {
         case WorkloadProbeMode.TCP: {
            // считаем команду незаполненной, если порт не задан (связано с валидацией на бекенде)
            patchObject(spec, 'tcp_check', tcpCheck =>
               item.tcp?.port ? this.patchTcpCheck(item.tcp, tcpCheck) : undefined,
            );
            break;
         }

         case WorkloadProbeMode.HTTP: {
            // считаем команду незаполненной, если порт не задан (связано с валидацией на бекенде)
            patchObject(spec, 'http_get', httpGet =>
               item.http?.port ? this.patchHttpGet(item.http, httpGet) : undefined,
            );
            break;
         }

         case WorkloadProbeMode.Exec: {
            // считаем команду незаполненной, если команда не задана (связано с валидацией на бекенде)
            patchObject(spec, 'container', container =>
               item.exec?.command?.trim() ? this.patchExecCommand(item.exec, container) : undefined,
            );
            break;
         }
      }

      return spec;
   }

   private patchStop(item: WorkloadStopCommand, spec: TStopPolicy) {
      if (item.mode !== WorkloadStopMode.HTTP) {
         delete spec.http_get;
      }
      if (item.mode !== WorkloadStopMode.Exec) {
         delete spec.container;
      }
      if (item.mode !== WorkloadStopMode.UnixSignal) {
         delete spec.unix_signal;
      }

      switch (item.mode) {
         case WorkloadStopMode.HTTP: {
            // считаем команду незаполненной, если порт не задан (связано с валидацией на бекенде)
            patchNumber(spec, 'max_tries', () => (item.http?.port ? item.maxTries : undefined));
            patchObject(spec, 'http_get', httpGet =>
               item.http?.port ? this.patchHttpGet(item.http, httpGet) : undefined,
            );

            break;
         }

         case WorkloadStopMode.Exec: {
            // считаем команду незаполненной, если команда не задана (связано с валидацией на бекенде)
            patchNumber(spec, 'max_tries', () => (item.exec?.command?.trim() ? item.maxTries : undefined));
            patchObject(spec, 'container', container =>
               item.exec?.command?.trim() ? this.patchExecCommand(item.exec, container) : undefined,
            );

            break;
         }

         case WorkloadStopMode.UnixSignal: {
            patchNumber(spec, 'max_tries', () => (item.maxTries ? item.maxTries : undefined));
            patchObject(spec, 'unix_signal', container =>
               item.unixSignal ? this.patchUnixSignal(item.unixSignal, container) : undefined,
            );
         }
      }

      return spec;
   }

   private patchDestroy(item: WorkloadDestroyCommand, spec: TDestroyPolicy) {
      if (item.mode !== WorkloadDestroyMode.HTTP) {
         delete spec.http_get;
      }
      if (item.mode !== WorkloadDestroyMode.Exec) {
         delete spec.container;
      }

      switch (item.mode) {
         case WorkloadDestroyMode.HTTP: {
            // считаем команду незаполненной, если порт не задан (связано с валидацией на бекенде)
            patchNumber(spec, 'max_tries', () => (item.http?.port ? item.maxTries : undefined));
            patchObject(spec, 'http_get', httpGet =>
               item.http?.port ? this.patchHttpGet(item.http, httpGet) : undefined,
            );
            break;
         }

         case WorkloadDestroyMode.Exec: {
            // считаем команду незаполненной, если команда не задана (связано с валидацией на бекенде)
            patchNumber(spec, 'max_tries', () => (item.exec?.command?.trim() ? item.maxTries : undefined));
            patchObject(spec, 'container', container =>
               item.exec?.command?.trim() ? this.patchExecCommand(item.exec, container) : undefined,
            );
            break;
         }
      }

      return spec;
   }

   private patchTcpCheck(tcp: WorkloadTcp, tcpCheck: TTcpCheck) {
      patchNumber(tcpCheck, 'port', () => tcp.port);

      patchObject(tcpCheck, 'time_limit', timeLimit => this.patchTimeLimit(tcp.retryPolicy, timeLimit));

      return tcpCheck;
   }

   private patchTimeLimit(retryPolicy: RetryPolicy | undefined, timeLimit: TTimeLimit) {
      patchNumber(timeLimit, 'initial_delay_ms', () => retryPolicy?.initialDelayMs);
      patchNumber(timeLimit, 'restart_period_scale_ms', () => retryPolicy?.restartPeriodScaleMs);
      patchNumber(timeLimit, 'restart_period_back_off', () => retryPolicy?.restartPeriodBackOff);
      patchNumber(timeLimit, 'max_restart_period_ms', () => retryPolicy?.maxRestartPeriodMs);
      patchNumber(timeLimit, 'min_restart_period_ms', () => retryPolicy?.minRestartPeriodMs);
      patchNumber(timeLimit, 'max_execution_time_ms', () => retryPolicy?.maxExecutionTimeMs);

      return timeLimit;
   }

   private toValue(): TWorkload {
      const { raw, workload, boxRef } = this;
      const spec: TWorkload = deepClone(raw ?? {}) as any;

      // console.log('spec');
      // console.log(spec);
      // console.log('workload');
      // console.log(workload);

      patchString(spec, 'box_ref', () => boxRef);

      patchObject(spec, 'start', startCommand => {
         this.patchExecCommand(workload.commands.start, startCommand);
         if (isEmpty(startCommand.command_line)) {
            delete (startCommand as Partial<TUtilityContainer>).command_line;
         }

         return startCommand;
      });

      patchList(spec, 'init', initCommands => {
         this.patchExecCommandList(workload.commands.init, initCommands);

         return initCommands.filter(v => !isEmpty(v) && !isEmpty(v.command_line));
      });

      patchObject(spec, 'liveness_check', livenessCheck => {
         this.patchProbe(workload.commands.liveness, livenessCheck);

         return livenessCheck;
      });

      patchObject(spec, 'readiness_check', readinessCheck => {
         this.patchProbe(workload.commands.readiness, readinessCheck);

         return readinessCheck;
      });

      patchObject(spec, 'stop_policy', stopPolicy => {
         this.patchStop(workload.commands.stop, stopPolicy);

         return skipEmpty(stopPolicy);
      });

      patchObject(spec, 'destroy_policy', destroyPolicy => {
         this.patchDestroy(workload.commands.destroy, destroyPolicy);

         return skipEmpty(destroyPolicy);
      });

      patchList(spec, 'env', specEnv =>
         skipEmpty(patchEnvironment(specEnv, workload.environment, this.visitor.secretResolver!, this.parentNodeIds)),
      );

      patchBoolean(spec, 'transmit_logs', () => workload.logs);

      return spec;
   }
}
