import { SummaryLevel } from '../../../../modules/infra-doctor';

export interface XRayIssuesSummary {
   high: number;
   medium: number;
   low: number;
}

export type XRayAnalysisStatus = 'unknown' | 'started' | 'done' | 'failed' | 'aborted';

export type XRaySummaryType = 'ok' | 'warning' | 'info' | 'critical' | 'failed' | 'loading' | 'unknown';

export interface XRayStatusSummary {
   issues: XRayIssuesSummary;
   stageHealth: 'n/a' | 'ok' | 'info' | 'warning' | 'critical';
   analysisStatus: XRayAnalysisStatus;
   summaryType: XRaySummaryType;
}

export const getDefaultXrayStatusSummary: () => XRayStatusSummary = () => ({
   issues: { low: 0, high: 0, medium: 0 },
   stageHealth: 'n/a',
   analysisStatus: 'unknown',
   summaryType: 'unknown',
});

export const getXRaySummaryType = ({
   analysisStatus,
   issues,
   stageHealth,
}: Omit<XRayStatusSummary, 'summaryType'>): XRaySummaryType => {
   if (analysisStatus !== 'done') {
      if (isRetryingXRayStatus(analysisStatus)) {
         return 'loading';
      }
      if (isRejectingXRayStatus(analysisStatus)) {
         return 'failed';
      }
      return 'unknown';
   }
   if (stageHealth === 'ok') {
      return 'ok';
   }
   if (issues.high > 0) {
      return 'critical';
   }
   if (issues.medium > 0) {
      return 'warning';
   }
   return 'info';
};

export const StageProblemTypeByXray: Record<XRaySummaryType, SummaryLevel> = {
   ok: 'ok',
   info: 'info',
   warning: 'warn',
   critical: 'crit',
   loading: 'unknown',
   failed: 'unknown',
   unknown: 'unknown',
};

export const retryingXRayStatuses: Set<XRayAnalysisStatus> = new Set(['unknown', 'started']);
export const isRetryingXRayStatus = (status: XRayAnalysisStatus): boolean => retryingXRayStatuses.has(status);

export const rejectingXRayStatuses: Set<XRayAnalysisStatus> = new Set(['failed', 'aborted']);
export const isRejectingXRayStatus = (status: XRayAnalysisStatus): boolean => rejectingXRayStatuses.has(status);
