import React from 'react';
import { Box, defaultPatcherRevision, DeployUnitDisk, DiskType } from './index';
import { ResourceGroup } from '../../../modules/resources/config';
import { buildTree, createKey, mergeStores, StoreKeysMeta } from '../../../utils';
import { QuotaResourcesStore } from '../quota/model';
import { Stage } from './Stage';
import { emptySidecarResources, SidecarName, SidecarRecources, SidecarsQuota } from '../Sidecars';
import { ByteMeasureUnit, BytePerSecondMeasureUnit, CoresMeasureUnit } from '../../../modules/resources/helpers';

// группы ресурсов, которые мы отображаем
// по мере поддержки в UI остальных они будут добавлены
//
// По умолчанию имеются в виду ГАРАНТИИ на ресурсы, а не лимиты
// (даже если они поддерживаются, как, например, NetworkBandwidth)
export const stageQuotaResourceGroups = [
   ResourceGroup.Cpu,
   ResourceGroup.Mem,
   ResourceGroup.Disk,
   ResourceGroup.DiskBandwidth,
   ResourceGroup.NetworkBandwidth,
] as const;

type stageQuotaResourceGroupsType = typeof stageQuotaResourceGroups;
export type StageQuotaResourceGroups = stageQuotaResourceGroupsType[Exclude<
   keyof stageQuotaResourceGroupsType,
   keyof []
>];

export const stageQuotaUnit: Record<StageQuotaResourceGroups, [string, string]> = {
   [ResourceGroup.Cpu]: [CoresMeasureUnit.cores, CoresMeasureUnit.ms],
   [ResourceGroup.Mem]: [ByteMeasureUnit.GB, ByteMeasureUnit.B],
   [ResourceGroup.Disk]: [ByteMeasureUnit.GB, ByteMeasureUnit.B],
   [ResourceGroup.DiskBandwidth]: [BytePerSecondMeasureUnit['MB/s'], BytePerSecondMeasureUnit['B/s']],
   [ResourceGroup.NetworkBandwidth]: [BytePerSecondMeasureUnit['MB/s'], BytePerSecondMeasureUnit['B/s']],
};

/* #region Resource Stores */

// плоская структура ресурсов, в которую легко добавлять новые значения для пересчёта
/**
 * resourceKey === createKey({deployUnit, resorceGroup, resource})
 */
export type StageResourcesStore = {
   [resourceKey: string]: number;
} & StoreKeysMeta<'deployUnit' | 'resourceGroup' | 'resource'>;

// плоская структура подов, в которую легко добавлять новые значения для пересчёта
/**
 * podCountKey === createKey({deployUnit, location})
 */
export type StagePodCountStore = {
   [podCountKey: string]: number;
} & StoreKeysMeta<'deployUnit' | 'location'>;

export interface StageSidecarQuotaParams {
   /** кастомные значения ресурсов */
   resources?: Partial<SidecarRecources>;
   /** кастомный тип диска, если сайдкару, будет нужна квота на диск
    * если не указан, значение будет взято из бокса / деплой юнита, такие значения хранятся в */
   diskType?: DiskType;
   /** использование сайдкара, для тех сайдкаров, которые используются несколько раз, но квота общая (например логи) */
   usageKeys?: Set<string>;
}

// плоская структура сайдкаров, в которую легко добавлять новые значения для пересчёта
// box='' эквивалентно тому, что сайдкар не зависит от бокса
// workload='' эквивалентно тому, что сайдкар не зависит от ворклоада
/**
 * sidecarsKey === createKey({sidecar, deployUnit, box, workload})
 */
export type StageSidecarsStore = {
   [sidecarsKey: string]: StageSidecarQuotaParams;
} & StoreKeysMeta<'sidecar' | 'deployUnit' | 'box' | 'workload'>;

// плоская структура типов дисков для сайдкаров по умолчанию, в которую легко добавлять новые значения для пересчёта
// box='' эквивалентно тому, что сайдкар не зависит от бокса
// дефолтные значения не зависят от типа сайдкара, только от параметров деплой юнита (и бокса для сайдкаров на уровне боксов)
/**
 * sidecarsKey === createKey({deployUnit, box})
 */
export type StageDiskTypeStore = {
   [sidecarsDiskType: string]: DiskType;
} & StoreKeysMeta<'deployUnit' | 'box'>;

// ревизии патчеров для деплой юнитов
export type StagePatcherRevisions = {
   [deployUnit: string]: number;
};

/* #endregion */

/* #region  get resources from stage */

export type StageDiskResources = Partial<
   Record<DiskType, { [ResourceGroup.Disk]?: number; [ResourceGroup.DiskBandwidth]?: number }>
>;

export function getStageDiskResources(disks: DeployUnitDisk[]): StageDiskResources {
   const diskResources: StageDiskResources = {};
   const diskTypes = [...new Set(disks.map(e => e.type))];

   for (const type of diskTypes) {
      diskResources[type] = {};

      const disksWithType = disks.filter(e => e.type === type);
      diskResources[type]![ResourceGroup.Disk] = disksWithType.reduce((sum, e) => sum + (e.size ?? 0), 0);

      diskResources[type]![ResourceGroup.DiskBandwidth] = disksWithType.reduce(
         (sum, e) => sum + (e.bandwidth.guarantee ?? 0),
         0,
      );
   }

   return diskResources;
}

/**
 * Получает плоскую модель ресурсов из стейджа
 *
 * По умолчанию имеются в виду ГАРАНТИИ на ресурсы, а не лимиты
 * (даже если они поддерживаются, как, например, NetworkBandwidth)
 */
export function getStageResourcesStore(stage: Stage): StageResourcesStore {
   const store: StageResourcesStore = {};
   for (const du of stage.deployUnits) {
      const { cpu, ram, networkBandwidth, disks, id } = du;

      store[createKey({ deployUnit: id, resourceGroup: ResourceGroup.Cpu, resource: '' })] = cpu ?? 0;
      store[createKey({ deployUnit: id, resourceGroup: ResourceGroup.Mem, resource: '' })] = ram ?? 0;
      store[createKey({ deployUnit: id, resourceGroup: ResourceGroup.NetworkBandwidth, resource: '' })] =
         networkBandwidth.guarantee ?? 0;

      const diskResources = getStageDiskResources(disks);

      for (const type of Object.keys(diskResources)) {
         const { disk = 0, diskBandwidth = 0 } = diskResources[type as DiskType]!;

         store[createKey({ deployUnit: id, resourceGroup: ResourceGroup.Disk, resource: type })] = disk;

         store[
            createKey({ deployUnit: id, resourceGroup: ResourceGroup.DiskBandwidth, resource: type })
         ] = diskBandwidth;
      }
   }
   return store;
}

/**
 * Получает плоскую модель подов из стейджа
 */
export function getStagePodCountStore(stage: Stage): StagePodCountStore {
   const store: StagePodCountStore = {};
   for (const du of stage.deployUnits) {
      const { locations, id } = du;
      const locationNames = Object.keys(locations).filter(name => locations[name].enabled);

      for (const name of locationNames) {
         store[createKey({ deployUnit: id, location: name })] = locations[name].podCount;
      }
   }
   return store;
}

/**
 * Получает плоскую модель сайдкаров из стейджа
 *
 * Каждый сайдкар учитывает свой уровень вложенности
 */
export function getStageSidecarsStore(stage: Stage): StageSidecarsStore {
   const store: StageSidecarsStore = {};
   for (const du of stage.deployUnits) {
      const { boxes, tvm, id: duId, disks, podAgentSidecarDiskType, logbrokerSidecarDiskType } = du;

      const createDuKey = (data: Record<string, string>) => createKey({ deployUnit: duId, ...data });

      const createExistOnlyDuKey = (data: Record<string, string>) => createDuKey({ box: '', workload: '', ...data });

      store[createExistOnlyDuKey({ sidecar: SidecarName.PodAgent })] = {
         diskType: podAgentSidecarDiskType ?? undefined,
      };

      if (tvm.enabled) {
         const { cpuLimit, memoryLimit, diskType } = tvm;

         store[createExistOnlyDuKey({ sidecar: SidecarName.TVM })] = {
            diskType: diskType ?? undefined,
            resources: {
               [ResourceGroup.Cpu]: cpuLimit ?? undefined,
               [ResourceGroup.Mem]: memoryLimit ?? undefined,
            },
         };
      }

      // использование логов по деплой юнитам, заполняется ниже
      const logbrokerUsageKeys: Set<string> = new Set();

      for (const box of boxes) {
         const { juggler, logrotateConfig, workloads, dynamicResources, id: boxId } = box;

         const createBoxKey = (data: Record<string, string>) => createDuKey({ box: boxId, ...data });
         const createExistOnlyBoxKey = (data: Record<string, string>) => createBoxKey({ workload: '', ...data });

         if (juggler.enabled) {
            store[createExistOnlyBoxKey({ sidecar: SidecarName.Juggler })] = {};
         }

         if (logrotateConfig.rawConfig) {
            store[createExistOnlyBoxKey({ sidecar: SidecarName.LogRotate })] = {};
         }

         if (dynamicResources.length > 0) {
            store[createExistOnlyBoxKey({ sidecar: SidecarName.DynamicResource })] = {};
         }

         for (const workload of workloads) {
            const { coredumpPolicy, id: workloadId, logs } = workload;
            const createExistOnlyWorkloadKey = (data: Record<string, string>) =>
               createBoxKey({ workload: workloadId, ...data });

            if (logs) {
               logbrokerUsageKeys.add(createExistOnlyWorkloadKey({ sidecar: SidecarName.Logbroker }));
            }

            if (coredumpPolicy) {
               const coredumpDiskType = getVolumeDiskType(disks, coredumpPolicy.coredump_processor?.volume_id ?? null);

               store[createExistOnlyWorkloadKey({ sidecar: SidecarName.Coredump })] = {
                  diskType: coredumpDiskType ?? undefined,
               };
            }
         }
      }

      if (logbrokerUsageKeys.size > 0) {
         // квота выделяется на весь деплой юнит
         // см. https://deploy.yandex-team.ru/docs/concepts/pod/sidecars/logs/logs#resources
         store[createExistOnlyDuKey({ sidecar: SidecarName.Logbroker })] = {
            diskType: logbrokerSidecarDiskType ?? undefined,
            usageKeys: logbrokerUsageKeys,
         };
      }
   }
   return store;
}

export function getStageDiskType(stage: Stage): StageDiskTypeStore {
   const store: StageDiskTypeStore = {};
   for (const deployUnit of stage.deployUnits) {
      const { boxes, disks, id: duId } = deployUnit;
      const duDiskType = getDuDiskType(disks);

      if (duDiskType) {
         store[createKey({ deployUnit: duId, box: '' })] = duDiskType;
      }

      for (const box of boxes) {
         const { id: boxId, virtualDiskIdRef } = box;
         const boxDiskType = getBoxDiskType(disks, virtualDiskIdRef);

         if (boxDiskType) {
            store[createKey({ deployUnit: duId, box: boxId })] = boxDiskType;
         }
      }
   }
   return store;
}

export function getBoxDiskType(disks: DeployUnitDisk[], virtualDiskIdRef: Box['virtualDiskIdRef']): DiskType | null {
   return disks.find(disk => disk.id === virtualDiskIdRef)?.type ?? null;
}

export function getDuDiskType(disks: DeployUnitDisk[]): DiskType | null {
   return disks[0]?.type ?? null;
}

export function getVolumeDiskType(disks: DeployUnitDisk[], volumeId: string | null): DiskType | null {
   if (!volumeId) {
      return null;
   }

   const disk = disks.find(d => d.volumes.some(v => v.id === volumeId));

   return disk?.type ?? null;
}

export function getStagePatcherRevisions(stage: Stage): StagePatcherRevisions {
   const { deployUnits } = stage;

   return deployUnits.reduce((revisions, du) => {
      const { id, patchersRevision } = du;

      if (patchersRevision.value) {
         revisions[id] = patchersRevision.value;
      }

      return revisions;
   }, {} as StagePatcherRevisions);
}

/* #endregion */

/* #region calc quota resources */

// подсчёт общих ресурсов квоты для всех подов в стейдже
// для каждого деплой юнита значения ресурсов пода указываются пользователем
// ресурсы каждого пода умножаются на их количество и суммируются по всем деплой юнитам
export function getUserQuotaResources(
   stageResourcesStore: StageResourcesStore,
   stagePodCountStore: StagePodCountStore,
): QuotaResourcesStore {
   const store: QuotaResourcesStore = {};

   // количество подов (чтобы умножать ресурсы для каждого пода на общее число)
   const podCountTree = buildTree(stagePodCountStore, ['deployUnit', 'location']);

   // значения ресурсов
   const resourcesTree = buildTree(stageResourcesStore, ['deployUnit', 'resourceGroup', 'resource']);

   for (const duId of Object.keys(resourcesTree)) {
      const duPodCount = podCountTree[duId] ?? {};
      const duLocations = Object.keys(podCountTree[duId] ?? {});

      // группы ресурсов для текущего деплой юнита
      const resourceGroupTree = resourcesTree[duId] ?? {};

      for (const locationName of duLocations) {
         const podCount = duPodCount[locationName] ?? 0;
         // только если поды есть
         if (podCount > 0) {
            for (const resourceGroup of Object.keys(resourceGroupTree)) {
               const resourceTree = resourceGroupTree[resourceGroup] ?? {};

               for (const resourceName of Object.keys(resourceTree)) {
                  const key = createKey({ location: locationName, resourceGroup, resource: resourceName });
                  store[key] = (store[key] ?? 0) + (resourceTree[resourceName] ?? 0) * podCount;
               }
            }
         }
      }
   }
   return store;
}

/**
 * Поиск дефолтных значений для сайдкара с учетом версий патчеров
 */
export function getDefaultSidecarResources(
   defaultSidecarQuota: SidecarsQuota,
   sidecarName: string,
   patcherRevision: number,
): SidecarRecources {
   const sidecarQuotaData = defaultSidecarQuota[sidecarName as SidecarName] ?? {};

   let targetRevision: number | null = null;
   if (String(patcherRevision) in sidecarQuotaData) {
      targetRevision = patcherRevision;
   } else {
      const revisions = Object.keys(sidecarQuotaData).map(Number);

      for (const revision of revisions) {
         if (revision <= patcherRevision) {
            if (targetRevision === null || revision > targetRevision) {
               targetRevision = revision;
            }
         }
      }
   }

   if (targetRevision === null) {
      return emptySidecarResources;
   }

   return sidecarQuotaData[String(targetRevision)];
}

export interface GetSidecarResourcesParams {
   sidecar: string;
   deployUnit: string;
   box: string;
   workload: string;
   stageSidecarsStore: StageSidecarsStore;
   stagePatcherRevisions: StagePatcherRevisions;
   defaultSidecarQuota: SidecarsQuota;
   stageDiskTypeStore: StageDiskTypeStore;
}

export function getSidecarResources({
   sidecar,
   deployUnit,
   box,
   workload,
   stageSidecarsStore,
   stagePatcherRevisions,
   defaultSidecarQuota,
   stageDiskTypeStore,
}: GetSidecarResourcesParams): StageSidecarQuotaParams {
   const params: StageSidecarQuotaParams = { resources: {} };
   const { resources: customResources = {}, diskType: customDiskType } =
      stageSidecarsStore[createKey({ sidecar, deployUnit, box, workload })] ?? {};
   const patcherRevision = stagePatcherRevisions[deployUnit] ?? defaultPatcherRevision;

   const duSidecarQuota = getDefaultSidecarResources(defaultSidecarQuota, sidecar, patcherRevision);
   const defaultDuDiskType = stageDiskTypeStore[createKey({ deployUnit, box: '' })];
   const defaultBoxDiskType = stageDiskTypeStore[createKey({ deployUnit, box })];

   for (const resourceGroup of [ResourceGroup.Cpu, ResourceGroup.Mem] as const) {
      const customValue = customResources[resourceGroup];
      const defaultValue = duSidecarQuota[resourceGroup];

      params.resources![resourceGroup] = customValue ?? defaultValue;
   }

   const diskType = customDiskType ?? defaultBoxDiskType ?? defaultDuDiskType;

   if (diskType) {
      const customDiskValue = customResources[ResourceGroup.Disk];
      const defaultDiskValue = duSidecarQuota[ResourceGroup.Disk];

      params.diskType = diskType;
      params.resources![ResourceGroup.Disk] = customDiskValue ?? defaultDiskValue;
   }

   return params;
}

// подсчёт общей квоты для сайдкаров для всех подов в стейдже
// в зависимости от сайдкара суммируются значения вплоть до ворклоада
export function getSidecarsQuotaResources(
   stagePodCountStore: StagePodCountStore,
   stageSidecarsStore: StageSidecarsStore,
   defaultSidecarQuota: SidecarsQuota,
   stagePatcherRevisions: StagePatcherRevisions,
   stageDiskTypeStore: StageDiskTypeStore,
): QuotaResourcesStore {
   const store: QuotaResourcesStore = {};

   // количество подов (чтобы умножать ресурсы для каждого пода на общее число)
   const podCountTree = buildTree(stagePodCountStore, ['deployUnit', 'location']);

   // значения сайдкаров
   const sidecarsTree = buildTree(stageSidecarsStore, ['deployUnit', 'sidecar', 'box', 'workload']);

   // дефолтные типы дисков для сайдкаров

   for (const duId of Object.keys(sidecarsTree)) {
      const duPodCount = podCountTree[duId] ?? {};
      // сайдкары для текущего деплой юнита
      const duSudecarsTree = sidecarsTree[duId] ?? {};
      for (const locationName of Object.keys(duPodCount)) {
         const podCount = duPodCount[locationName] ?? 0;
         // только если поды есть
         if (podCount > 0) {
            // ресурсы для сайдкаров
            for (const sidecarName of Object.keys(duSudecarsTree)) {
               const boxSidecarTree = duSudecarsTree[sidecarName]!;
               for (const box of Object.keys(boxSidecarTree)) {
                  const workloadSidecarsTree = boxSidecarTree[box]!;
                  for (const workload of Object.keys(workloadSidecarsTree)) {
                     const { resources = {}, diskType } = getSidecarResources({
                        sidecar: sidecarName,
                        deployUnit: duId,
                        box,
                        workload,
                        stageSidecarsStore,
                        stagePatcherRevisions,
                        defaultSidecarQuota,
                        stageDiskTypeStore,
                     });
                     const sidecarResourceNames = {
                        [ResourceGroup.Cpu]: '',
                        [ResourceGroup.Mem]: '',
                        [ResourceGroup.Disk]: diskType ?? null,
                     };
                     for (const group of Object.keys(sidecarResourceNames)) {
                        const resourceGroup = group as keyof typeof sidecarResourceNames;
                        const resourceName = sidecarResourceNames[resourceGroup];
                        if (resourceName !== null) {
                           const key = createKey({
                              location: locationName,
                              resourceGroup,
                              resource: resourceName,
                           });
                           store[key] = (store[key] ?? 0) + (resources[resourceGroup] ?? 0) * podCount;
                        }
                     }
                  }
               }
            }
         }
      }
   }
   return store;
}

export function getAllQuotaResources(
   userResources: QuotaResourcesStore,
   sidecarResources: QuotaResourcesStore,
): QuotaResourcesStore {
   return mergeStores(userResources, sidecarResources, (a, b) => (a ?? 0) + (b ?? 0));
}

/* #endregion */

export interface ResourceContextValue {
   stageResourcesStore: StageResourcesStore;
   stagePodCountStore: StagePodCountStore;
   stageSidecarsStore: StageSidecarsStore;
   stageDiskTypeStore: StageDiskTypeStore;
   stagePatcherRevisions: StagePatcherRevisions;
}

export const emptyResourceContextValue: ResourceContextValue = {
   stageResourcesStore: {},
   stagePodCountStore: {},
   stageSidecarsStore: {},
   stageDiskTypeStore: {},
   stagePatcherRevisions: {},
};

export const ResourcesContext = React.createContext(emptyResourceContextValue);
