/* eslint-disable no-template-curly-in-string */
import { BYTES } from '@yandex-infracloud-ui/libs';

import { MixedSchema, number, string } from 'yup';

import { ValidationContext } from '../../../components/huge-form';

export const orderSchema = number().integer().min(0).notRequired();

export const IntegerSchema = (label: string) => number().label(label).integer();
export const IntegerNullableSchema = (label: string) => IntegerSchema(label).nullable();
export const IntegerPositiveNullableSchema = (label: string) => IntegerNullableSchema(label).min(1);

const StringSchema = (label: string) => string().label(label);
export const StringRequiredSchema = (label: string) => StringSchema(label).required();
export const StringNullableSchema = (label: string) => StringSchema(label).nullable();
export const StringNullableRequiredSchema = (label: string) => StringNullableSchema(label).min(1).required();

export const StringMatchesSchema = (label: string, re: RegExp, message?: string) =>
   StringSchema(label).matches(re, message);

export const StringMatchesRequiredSchema = (label: string, re: RegExp, message?: string) =>
   StringMatchesSchema(label, re, message).required();

export const StringNotMatchesRequiredSchema = (label: string, re: RegExp, message?: string) =>
   StringSchema(label)
      .test('is-correct', message ?? '${path} is incorrect', value => !re.test(value))
      .required();

export const hasOnlyUniqueValues = (items: any[]) => new Set(items).size === items.length;

// region PORT validation

const RESERVED_PORTS = [1, 2, 3, 22, 1234];

const RESERVED_TVM_CLIENT_PORTS = [1, 3, 22, 1234];

const createReservedPortValidator = (reserved: number[]) => (value: number | null) => {
   if (value === null) {
      return true;
   }

   const reservedPorts = new Set(reserved);

   return !(reservedPorts.has(value) || (value >= 12500 && value <= 13000));
};

export const portSchema = IntegerPositiveNullableSchema('Port')
   .max(65535) // TODO .max(32767) см. DEPLOY-3340
   .test(
      'is-not-reserved',
      `Port numbers ${RESERVED_PORTS.join(', ')}, 12500-13000 are reserved`,
      createReservedPortValidator(RESERVED_PORTS),
   );

export const tvmPortSchema = IntegerPositiveNullableSchema('Port')
   .max(65535) // TODO .max(32767) см. DEPLOY-3340
   .test(
      'is-not-reserved',
      `Port numbers ${RESERVED_TVM_CLIENT_PORTS.join(', ')}, 12500-13000 are reserved`,
      createReservedPortValidator(RESERVED_TVM_CLIENT_PORTS),
   );

// endregion

// изначально делали для статических ресурсов
// см. https://st.yandex-team.ru/DEPLOY-1125#5e020c0d34b2182e6e1b8dc8
export const unixPathSchema = (label: string) =>
   StringNotMatchesRequiredSchema(
      label,
      /^(((\/)?((\.)?\.\/))(.+)|(.*)((\/\/)|(\/\.\.\/)|(\/\.\/)|([а-яА-ЯёЁ]{1,}))(.*)|(.*)(\/\/)(.*)|(.+)((\/\.(\.)?)(\/)?))$/,
   )
      .nullable()
      .test('noWhitespace', 'Mount point must not contain whitespace', (v: string) => (v ? !/\s/.test(v) : true));

export const checksumSchema = StringNullableSchema('Checksum').matches(
   /^(EMPTY:|MD5:[a-fA-F0-9]{32}|SHA256:[a-fA-F0-9]{64})$/,
   '${path} format is invalid (MD5:, EMPTY:, SHA256:)',
);

// region URL validation

export const resourceUrlSchema = StringSchema('URL')
   .matches(
      /^(https?:\/\/.+|rbtorrent:[a-f0-9]{40}|raw:(.|\s)*|sbr:[\d]+)$/,
      '${path} format is invalid (http(s)://, rbtorrent:, sbr:, raw:)',
   )
   .max(5 * BYTES.KB);

// @nodejsgirl привести все поля к одной схеме в рамках DEPLOY-3580 и DEPLOY-3357
export const layerUrlSchema = resourceUrlSchema.required();
export const staticResourceUrlSchema = resourceUrlSchema.required();
export const dynamicResourceUrlSchema = resourceUrlSchema.nullable().required();

// endregion

// region ID validation

const createSiblingIdSchema = (re: RegExp) =>
   StringSchema('ID')
      .test('UniqueId', 'ID must be unique', function UniqueId(value: string) {
         const context: ValidationContext | undefined = this.options.context as any;

         if (context) {
            return !context.siblingIds.has(value);
         }

         return true;
      })
      .matches(re)
      .required();

const createSiblingAndCousinIdSchema = (re: RegExp) =>
   StringSchema('ID')
      .test('UniqueId', 'ID must be unique', function UniqueId(value: string) {
         const context: ValidationContext | undefined = this.options.context as any;

         if (context) {
            return !context.siblingAndCousinIds.has(value);
         }

         return true;
      })
      .matches(re)
      .required();

const PATTERN_ID_1 = /^[\w-]{1,70}$/;
const PATTERN_ID_2 = /^[\w-@:.]{1,64}$/;

export const stageIdSchema = createSiblingIdSchema(PATTERN_ID_1).label('Stage ID');
export const projectIdSchema = createSiblingIdSchema(PATTERN_ID_1).label('Project ID');
export const deployUnitIdSchema = createSiblingIdSchema(PATTERN_ID_1).label('Deploy Unit ID');
export const boxIdSchema = createSiblingIdSchema(PATTERN_ID_2).label('Box ID');
export const workloadIdSchema = createSiblingAndCousinIdSchema(PATTERN_ID_2).label('Workload ID');

// layer, static resource, dynamic resource, volume
export const resourceIdValidationSchema = StringMatchesRequiredSchema('ID', PATTERN_ID_2);

// endregion

export const requiredIfEnabled = <T extends MixedSchema>(enabled: boolean, schema: T) =>
   enabled ? schema.nullable().required() : schema.nullable().notRequired();

// Либо значение, либо генератор значения
type ValueOrThunk<I, O = I> = O | ((i: I) => O);

export const schemeIfEnabled = <S extends MixedSchema, T extends MixedSchema = S>(
   childSchema: ValueOrThunk<T, S>,
   fallback?: ValueOrThunk<T, S>,
) => (enabled: boolean, schema: T) => {
   const actualChildSchema = typeof childSchema === 'function' ? childSchema(schema) : childSchema;
   const actualFallback = typeof fallback === 'function' ? fallback(schema) : fallback;

   return enabled ? actualChildSchema : actualFallback ?? schema;
};
