// Общие части yasm для всех частей стейджа

import { getSetDifference, isEmpty } from '@yandex-infracloud-ui/libs';
import { array, object, string } from 'yup';
import { patchString, skipEmpty } from '../utils';

export interface YasmTag {
   key: string | null;
   value: string | null;
}

export interface YasmTags {
   itype: string | null;
   tags: YasmTag[];
}

export function convertYasmLabels(labels: { [key: string]: string } | undefined) {
   const { itype, ...tags } = labels ?? {};

   return {
      itype: itype ?? null,
      tags: tags
         ? Object.keys(tags)
              .map(key => ({ key, value: tags[key] }))
              .filter(tag => !isEmpty(tag.key) && !isEmpty(tag.value))
         : [],
   };
}

export function patchYasmLabels(yasm: YasmTags, specLabels: { [key: string]: string }) {
   patchString(specLabels, 'itype', () => skipEmpty(yasm.itype));

   yasm.tags.forEach(({ key, value }) => {
      if (key) {
         patchString(specLabels, key, () => skipEmpty(value));
      }
   });

   const labelKeys = yasm.tags.map(v => v.key).filter(v => !isEmpty(v));

   if (!isEmpty(yasm.itype)) {
      labelKeys.push('itype');
   }

   const { removed } = getSetDifference(new Set(Object.keys(specLabels)), new Set(labelKeys));

   removed.forEach(key => {
      delete specLabels[key ?? ''];
   });

   return specLabels;
}

export const yasmTagsValidationSchema = object<YasmTags>({
   itype: string()
      .nullable()
      .label('YASM itype')
      .matches(/^[\w][\w-]{0,40}$/),
   tags: array()
      .of(
         object<YasmTag>({
            key: string().nullable().label('Tag name').min(1).required(),
            value: string().nullable().label('Tag value').min(1).required(),
         }),
      )
      .test('duplicate-tag', 'Duplicate tag name', value => {
         let result = true;

         if (value) {
            const keys: string[] = [];

            value.forEach((v: YasmTag) => {
               if (v.key) {
                  if (keys.includes(v.key)) {
                     result = false;
                  } else {
                     keys.push(v.key);
                  }
               }
            });
         }

         return result;
      }),
});
