import { TIMES_IN_MS } from '@yandex-infracloud-ui/libs';
import { EPodCurrentState } from '../../../../proto-typings';

import { customMap } from '../../../../utils';

import {
   Pod,
   PodWorkload,
   PodWarningLevel,
   getLevelWarningCount,
   getMapWarningCount,
   getDefaultStatusInfo,
   StatusStateInfo,
   PodCustomCurrentState,
} from '../../runtimeDeploy';
import { DeployUnit, Workload } from '../../stage';
import { PodStatusView } from './model';
import { getPodWorkloadStatusView } from './PodWorkloadStatusView';

function getDuWorkloadsMap(deployUnit: DeployUnit | null): Map<string, Workload> {
   return new Map((deployUnit?.boxes ?? []).flatMap(box => box.workloads.map(workload => [workload.id, workload])));
}

type GetPodStatusViewParams = {
   pod: Pod;
   deployUnit: DeployUnit | null;
   location: string;
};

export function getPodStatusView({ pod, deployUnit, location }: GetPodStatusViewParams): PodStatusView {
   const podStatusView: PodStatusView = {
      ...pod,
      __type: 'PodStatusView' as const,
      location,
      lastUpdateTimestamp: 0, // заполняется ниже
   };

   const duWorkloadsMap = getDuWorkloadsMap(deployUnit);

   podStatusView.workloads = customMap(podStatusView.workloads as Map<string, PodWorkload>, podWorkload => {
      const podWorkloadStatusView = getPodWorkloadStatusView({
         podWorkload,
         workloadStageSpec: duWorkloadsMap.get(podWorkload.id) ?? null,
      });
      return podWorkloadStatusView;
   });

   podStatusView.currentState = getPodCurrentState({ podStatusView });
   podStatusView.statusStateInfo = getPodStatusStateInfo({ podStatusView });

   podStatusView.warningCount = getLevelWarningCount(podStatusView.warning);

   podStatusView.totalWarningCount =
      podStatusView.warningCount +
      getMapWarningCount(podStatusView.workloads) +
      getMapWarningCount(podStatusView.boxes) +
      getMapWarningCount(podStatusView.layers) +
      getMapWarningCount(podStatusView.volumes) +
      getMapWarningCount(podStatusView.staticResources);

   podStatusView.lastUpdateTimestamp = Date.now();

   return podStatusView;
}

const issMessages = new Set([
   'Downloading resources...',
   'NETWORK_NOT_READY',
   'DAEMON_STARTING',
   // https://bb.yandex-team.ru/projects/SEARCH_INFRA/repos/iss/browse/iss-agent-parent/agent/src/main/scala/ru/yandex/iss/agent/drivers/FilesListDriver.scala#175
   'FILE_TASK_STARTED',
   'WAITING_FOR_POD_AGENT_SOCKET',
]);

function getPodStatusStateInfo({ podStatusView }: { podStatusView: PodStatusView }): StatusStateInfo {
   const { statusStateInfo, totalWarningCount } = podStatusView;
   const { failed, inProgress } = statusStateInfo;

   const isUnknown = podStatusView.currentState === EPodCurrentState.PCS_UNKNOWN;

   // scheduling error
   if (isUnknown && !failed.active && totalWarningCount === 1) {
      const scheduling = podStatusView.warning[PodWarningLevel.Scheduling];
      if (scheduling && scheduling.length === 1 && scheduling[0] && scheduling[0].error?.message) {
         statusStateInfo.failed = getDefaultStatusInfo();
         statusStateInfo.failed.active = true;
         statusStateInfo.failed.reason = 'sсheduling error';
      }
   }

   const isStartPending = podStatusView.currentState === EPodCurrentState.PCS_START_PENDING;
   const issForUnknown = isUnknown && !failed.active && !inProgress.active;
   const issForStartPending = isStartPending && !failed.active && inProgress.active;
   // iss progress messages
   if ((issForUnknown || issForStartPending) && totalWarningCount === 1) {
      const nodeAgentError = podStatusView.warning[PodWarningLevel.NodeAgent];
      if (nodeAgentError && nodeAgentError.length === 1 && nodeAgentError[0] && nodeAgentError[0].error?.message) {
         const { error } = nodeAgentError[0];
         const { message, code, inner_errors } = error;
         if (
            (code === 1 && message.endsWith('_IN_PROGRESS')) ||
            issMessages.has(message) ||
            inner_errors?.length === 0
         ) {
            statusStateInfo.inProgress = getDefaultStatusInfo();
            statusStateInfo.inProgress.active = true;
            statusStateInfo.inProgress.reason = prettyYtReason(message);
            delete podStatusView.warning[PodWarningLevel.NodeAgent];
         }
      }
   }

   return statusStateInfo;
}

const inactiveTimeToSuspected = TIMES_IN_MS.Minute * 5 * 1000;
function getPodCurrentState({ podStatusView }: { podStatusView: PodStatusView }) {
   const { agentLastHeartbeatTime, currentState } = podStatusView;
   if (!agentLastHeartbeatTime) {
      return currentState;
   }

   const isSuspected = Date.now() * 1000 - agentLastHeartbeatTime >= inactiveTimeToSuspected;
   return isSuspected ? PodCustomCurrentState.PCS_SUSPECTED : currentState;
}

function prettyYtReason(message: string): string {
   const fullText = message.replace(/_/g, ' ').toLowerCase();
   if (fullText.length > 30) {
      return `${fullText.slice(0, 30)}...`;
   }
   return fullText;
}
