import { EWorkloadState } from '../../../../proto-typings';
import {
   getWorkloadWarning,
   PodWorkload,
   WorkloadWarningLevel,
   getLevelWarningCount,
   StatusStateInfo,
} from '../../runtimeDeploy';
import { Workload } from '../../stage';
import { PodWorkloadStatusView } from './model';

interface WorkloadOptions {
   getWorkloadState(state: EWorkloadState): EWorkloadState;
   getWorkloadWarning(workload: Omit<PodWorkload, 'warning'>): PodWorkload['warning'];
   getStatusStateInfo(status: StatusStateInfo): StatusStateInfo;
   isDefault?: true;
}

const Id = <T>(e: T) => e;
const defaultWorkloadOptions: WorkloadOptions = {
   getWorkloadState: Id,
   getWorkloadWarning,
   getStatusStateInfo: Id,
   isDefault: true,
};

const deadDruWorkloadOptions: WorkloadOptions = {
   getWorkloadState: () => EWorkloadState.EWorkloadState_ACTIVE,
   getWorkloadWarning: workload => {
      const warning = defaultWorkloadOptions.getWorkloadWarning(workload);
      delete warning[WorkloadWarningLevel.Workload];
      delete warning[WorkloadWarningLevel.Start];
      return warning;
   },
   getStatusStateInfo: status => {
      status.failed.active = false;
      return status;
   },
};

/**
 * почему 70 секунд? подобрано опытным путём
 */
const FAKE_DEAD_DRU_TIMEOUT = 70 * 1000;

function isFakeDeadDru(podWorkload: PodWorkload, workloadStageSpec: Workload | null): boolean {
   const { id, state, statusStateInfo } = podWorkload;
   const { failed } = statusStateInfo;

   const isAutoDru = workloadStageSpec === null && id.endsWith('__dru');
   const nowTime = Date.now();
   const timeDiff = nowTime - Number(failed.lastTransitionTime ?? 0);
   const isFewTime = timeDiff < FAKE_DEAD_DRU_TIMEOUT;
   const isDead = state === EWorkloadState.EWorkloadState_DEAD && failed.active;

   return isAutoDru && isDead && isFewTime;
}

function getWorkloadOptions(podWorkload: PodWorkload, workloadStageSpec: Workload | null): WorkloadOptions {
   // DRU https://st.yandex-team.ru/DEPLOY-5277#61ba0d3936eaaa6bd66a9fdc
   if (isFakeDeadDru(podWorkload, workloadStageSpec)) {
      return deadDruWorkloadOptions;
   }

   return defaultWorkloadOptions;
}

type GetPodWorkloadStatusViewParams = {
   podWorkload: PodWorkload;
   workloadStageSpec: Workload | null;
};

export function getPodWorkloadStatusView({
   podWorkload,
   workloadStageSpec,
}: GetPodWorkloadStatusViewParams): PodWorkloadStatusView {
   const options = getWorkloadOptions(podWorkload, workloadStageSpec);

   if (options.isDefault) {
      return podWorkload;
   }

   // переписывание состояния
   podWorkload.state = options.getWorkloadState(podWorkload.state);

   // переписывание ошибок
   podWorkload.warning = options.getWorkloadWarning(podWorkload);
   podWorkload.warningCount = getLevelWarningCount(podWorkload.warning);

   // переписывание статусов
   podWorkload.statusStateInfo = options.getStatusStateInfo(podWorkload.statusStateInfo);

   return podWorkload;
}
