import { deepClone } from '@yandex-infracloud-ui/libs';

import {
   ReplicaSet,
   ReplicaSetConverter,
   ReplicaSetPodsSummary,
   ReplicaSetType,
   getDefaultStatusInfo,
   StatusStateInfo,
} from '../../runtimeDeploy';
import { ReplicaSetLocationInfo, ReplicaSetStatusView } from './model';
import { PodStatusView } from '../pod';
import { getPodSummaryFromPodList, recalculatePodCounts } from './recalculatePodCounts';
import { YpLocation } from '../../../api';
import { MergeStatusStateInfo } from '../mergeStatus';
import { DeployUnit, DeployUnitStatus, getDisabledStageClusters, Stage } from '../../stage';
import { createKey } from '../../../../utils';

type LoadPodsByType = Record<keyof ReplicaSetPodsSummary, boolean>;

type GetReplicaSetStatusViewParams = {
   /** модель из yp */
   replicaSet: ReplicaSet;

   /** видимые поды rs на странице, нужны для устранения расхождения их количества */
   visiblePods: PodStatusView[];

   loadAllPods?: LoadPodsByType;

   // связанный "родительский" деплой-юнит
   deployUnit: DeployUnit | null;
   deployUnitStatus: DeployUnitStatus | null;

   // связанный "родительский" стейдж
   stage: Stage | null;

   // поля из деплой-юнита
   // replicaSetInfo: ReplicaSetInfo;
   // replicaSetDisruptionBudgets: ReplicaSetDisruptionBudgets | null;
};

export function getReplicaSetStatusView({
   replicaSet,
   visiblePods,
   loadAllPods,
   deployUnit,
   deployUnitStatus,
   stage,
}: GetReplicaSetStatusViewParams): ReplicaSetStatusView {
   const { id, location } = replicaSet;
   const rsKey = createKey({ id, location });
   const replicaSetInfo = deployUnitStatus?.replicaSetsInfo?.[rsKey] ?? null;

   const replicaSetDisruptionBudgets =
      deployUnit && deployUnitStatus && replicaSetInfo
         ? ReplicaSetConverter.getDisruptionBudgets(replicaSetInfo, deployUnit, deployUnitStatus)
         : null;

   const replicaSetStatusView: ReplicaSetStatusView = {
      ...replicaSet,
      __type: 'ReplicaSetStatusView',
      replicaSetInfo,
      replicaSetDisruptionBudgets,
      replicaSetLocationInfo: getReplicaSetLocationInfo({ replicaSetData: replicaSet }).replicaSetLocationInfo,
      actualPodsSummary: {} as any, // заполняется ниже
      mergePodsSummary: {} as any, // заполняется ниже
      mergeStatusStateInfo: {} as any, // заполняется ниже
      lastUpdateTimestamp: 0, // заполняется ниже
   };

   const { currentRevision } = replicaSetStatusView;

   // разбивка количества видимых подов по типам
   const { summary } = getPodSummaryFromPodList({
      pods: visiblePods,
      currentRevision,
      disabledClusters: getDisabledStageClusters(stage),
   });

   // сохраняем актуальное количество
   replicaSetStatusView.actualPodsSummary = summary;

   // пересчитываем количество подов
   replicaSetStatusView.mergePodsSummary = recalculatePodCounts({
      replicaSetCounts: replicaSetStatusView.podsSummary,
      loadAll: loadAllPods ?? null,
      loadPodCounts: summary,
   }).result;

   // пересчитываем статус
   replicaSetStatusView.mergeStatusStateInfo = getReplicaSetMergeStatusStateInfo({
      statusStateInfo: replicaSetStatusView.statusStateInfo,
      mergePodsSummary: replicaSetStatusView.mergePodsSummary,
      disruptionBudget: replicaSetDisruptionBudgets?.override?.value ?? replicaSetDisruptionBudgets?.spec.value ?? null,
   });

   // выставляем время обновления
   replicaSetStatusView.lastUpdateTimestamp = Date.now();

   return replicaSetStatusView;
}

function getReplicaSetMergeStatusStateInfo({
   statusStateInfo,
   mergePodsSummary,
   disruptionBudget,
}: {
   statusStateInfo: StatusStateInfo;
   mergePodsSummary: ReplicaSetPodsSummary;
   disruptionBudget: number | null;
}): MergeStatusStateInfo {
   const info: MergeStatusStateInfo = { ...deepClone(statusStateInfo), calculate: new Set() };

   const { ready, total, inProgress, oldInProgress } = mergePodsSummary;

   const needed = total - (disruptionBudget ?? 0);
   const existReady = ready > 0;
   const isReady = ready >= needed && existReady;

   const existInProgress = inProgress + oldInProgress > 0;

   if (info.failed.active) {
      // ошибки не меняем
      return info;
   }

   if (info.inProgress.active && isReady) {
      // прогресс кончился, все поды готовы, хотя rs ещё inProgress
      info.inProgress.active = false;
      info.ready = getDefaultStatusInfo();
      info.ready.active = true;
      info.ready.reason = 'pods are ready';

      info.calculate.add('ready');

      return info;
   }

   if (!info.inProgress.active && !isReady && existInProgress) {
      // прогресс начался, хотя rs ещё ready
      info.inProgress = getDefaultStatusInfo();
      info.inProgress.active = true;
      info.inProgress.reason = 'pods deploy started';

      info.calculate.add('inProgress');

      return info;
   }

   return info;
}

export function getReplicaSetLocationInfo<T extends { type: ReplicaSetType; locations: Set<string> }>({
   replicaSetData,
}: {
   replicaSetData: T;
}): { replicaSetLocationInfo: ReplicaSetLocationInfo } {
   const { locations, type } = replicaSetData;

   const oneLocation = type === ReplicaSetType.PerCluster ? (Array.from(locations.values())[0] as YpLocation) : null;
   const ypLocation = type === ReplicaSetType.PerCluster ? oneLocation! : YpLocation.XDC;
   const replicaSetLocationInfo: ReplicaSetLocationInfo = {
      oneLocation,
      ypLocation,
   };

   return { replicaSetLocationInfo };
}
