import { getMapFromRecord, recalculateCounts } from '../../../../utils';
import { getEmptyReplicaSetPodsSummary, ReplicaSetPodsSummary } from '../../runtimeDeploy';
import { getRecordFromMap } from '../../../../utils/helpers/mapTransform';
import { PodStatusView } from '../pod';

export type RecalculatePodCounts = {
   replicaSetCounts: ReplicaSetPodsSummary | null;
   loadPodCounts: ReplicaSetPodsSummary | null;
   loadAll: Record<keyof ReplicaSetPodsSummary, boolean> | null;
};

export type RecalculatePodCountsResult = {
   result: ReplicaSetPodsSummary;
};

export function recalculatePodCounts(params: RecalculatePodCounts): RecalculatePodCountsResult {
   const { replicaSetCounts, loadPodCounts, loadAll } = params;
   const visibleCounts = getMapFromRecord(replicaSetCounts ?? getEmptyReplicaSetPodsSummary());
   visibleCounts.delete('total');
   const actualPodCounts = getMapFromRecord(loadPodCounts ?? getEmptyReplicaSetPodsSummary());
   actualPodCounts.delete('total');
   const loadAllMap: Map<keyof ReplicaSetPodsSummary, boolean> = loadAll ? getMapFromRecord(loadAll) : new Map();

   const { result } = recalculateCounts({
      visible: visibleCounts,
      actual: actualPodCounts,
      loadAll: loadAllMap,
   });

   const counts = getRecordFromMap(result);
   counts.total = Object.values(counts).reduce((sum, count) => sum + count, 0);

   return { result: counts };
}

type GetPodSummaryFromPodListParams = {
   pods: PodStatusView[];
   currentRevision: number | null;
   disabledClusters: Set<string>;
};

export function getPodSummaryFromPodList({
   pods,
   currentRevision,
   disabledClusters,
}: GetPodSummaryFromPodListParams): { summary: ReplicaSetPodsSummary } {
   const summary: ReplicaSetPodsSummary = getEmptyReplicaSetPodsSummary();

   for (const pod of pods) {
      const { revision, statusStateInfo, targetRevision, location } = pod;

      if (disabledClusters.has(location)) {
         // пропускаем все поды в отключенных локациях, считаем, что нет информации о них
         // eslint-disable-next-line no-continue
         continue;
      }
      const isRevisionProgress = targetRevision ? targetRevision > (revision ?? 0) : false;

      const { ready, failed, inProgress } = statusStateInfo;
      const old = revision !== currentRevision;

      summary.total += 1;

      if (failed.active) {
         old ? (summary.oldFailed += 1) : (summary.failed += 1);
      } else if (inProgress.active) {
         old ? (summary.oldInProgress += 1) : (summary.inProgress += 1);
      } else if (isRevisionProgress) {
         // ВАЖНО: только при отсутствии ошибок, ветка условия перед самым ready

         // считаем, что если под стремится к целевой ревизии,
         // значит он уже обрабатывается rs контроллером и помечен как под в прогрессе
         summary.inProgress += 1;
      } else if (ready.active) {
         old ? (summary.oldReady += 1) : (summary.ready += 1);
      } else {
         summary.unknown += 1;
      }
   }

   return { summary };
}

const summaryKeyList = Object.keys(getEmptyReplicaSetPodsSummary()).sort() as (keyof ReplicaSetPodsSummary)[];
export function createPodSummaryKey(summary: Partial<ReplicaSetPodsSummary>): string {
   return summaryKeyList.map(type => `${type}:${summary[type]}`).join(',');
}
