import { deepClone } from '@yandex-infracloud-ui/libs';

import { DeployUnit, PerLocationSettings } from '../../stage';
import {
   DeployUnitState,
   DeployUnitStatus,
   DeployUnitStatusConverter,
   DuStatusLabel,
   LocationApprovalStatus,
   LocationApproves,
} from '../../stage/DeployUnit/DeployUnitStatusConverter';
import { DeployUnitStatusView, MergeDeployUnitStatusLabel } from './model';
import { ReplicaSetStatusView } from '../replicaSet/model';

type GetDeployUnitStatusViewParams = {
   deployUnitStatus: DeployUnitStatus;
   deployUnit: DeployUnit;

   replicaSetData: ReplicaSetStatusView[];
};

type GetDeployUnitStatusViewResult = {
   deployUnitStatusView: DeployUnitStatusView;
};

export function getDeployUnitStatusView({
   deployUnitStatus,
   deployUnit,
   replicaSetData,
}: GetDeployUnitStatusViewParams): GetDeployUnitStatusViewResult {
   const deployUnitLabel = DeployUnitStatusConverter.getDuStatusLabel(deployUnit, deployUnitStatus);
   const deployUnitStatusView: DeployUnitStatusView = {
      ...deployUnitStatus,
      spec: deployUnit,
      deployUnitLabel,
      mergeDeployUnitLabel: {} as any, // заполняется ниже
      locationsWaitingForApprove: new Map(), // заполняется ниже
      waitingForApprove: false, // заполняется ниже
      lastUpdateTimestamp: 0, // заполняется ниже
   };

   deployUnitStatusView.locationsWaitingForApprove = getLocationsWaitingForApprove({
      perLocationSettings: deployUnit.perLocationSettings,
      locationApproves: deployUnitStatusView.locationApproves,
      locations: Object.keys(deployUnit.locations),
      overridePerLocationSettings: deployUnitStatusView.overrides?.deploySettings?.perLocationSettings ?? null,
   });

   deployUnitStatusView.waitingForApprove = Array.from(deployUnitStatusView.locationsWaitingForApprove.values()).some(
      Boolean,
   );

   deployUnitStatusView.mergeDeployUnitLabel = getMergeDeployUnitLabel({
      deployUnitLabel,
      replicaSetData,
      targetRevision: deployUnit.revision,
   });

   deployUnitStatusView.lastUpdateTimestamp = Date.now();

   return { deployUnitStatusView };
}

type GetMergeDeployUnitLabelParams = {
   deployUnitLabel: DuStatusLabel;
   replicaSetData: ReplicaSetStatusView[];
   targetRevision: number | null;
};

/**
 * подсчёт видимого статуса
 */
function getMergeDeployUnitLabel({
   deployUnitLabel,
   replicaSetData,
   targetRevision,
}: GetMergeDeployUnitLabelParams): MergeDeployUnitStatusLabel {
   const label: MergeDeployUnitStatusLabel = { ...deepClone(deployUnitLabel), calculate: false };

   // если ревизия совпадает с целевой и все rs этой же ревизии ready, то и сам du ready, но только если был inProgress
   if (label.state === DeployUnitState.InProgress && replicaSetData.length > 0) {
      const isAllRsReady = replicaSetData.every(replicaSetStatusView => {
         const { currentRevision, mergeStatusStateInfo } = replicaSetStatusView;
         if (currentRevision !== targetRevision) {
            return false;
         }
         const { ready, inProgress, failed } = mergeStatusStateInfo;
         return ready.active && !inProgress.active && !failed.active;
      });

      if (isAllRsReady) {
         label.state = DeployUnitState.Ready;
         label.text = 'pods are ready';
         label.message = null;
         label.calculate = true;

         return label;
      }
   }

   return label;
}

function getLocationsWaitingForApprove({
   perLocationSettings,
   locationApproves,
   locations,
   overridePerLocationSettings,
}: {
   perLocationSettings: PerLocationSettings;
   locationApproves: LocationApproves;
   locations: string[];
   overridePerLocationSettings: PerLocationSettings | null;
}): Map<string, boolean> {
   const settings = { ...perLocationSettings, ...(overridePerLocationSettings ?? {}) };
   const { isCustom } = settings;

   if (!isCustom) {
      return new Map();
   }

   const result: Map<string, boolean> = new Map();
   for (const location of locations) {
      result.set(location, getWaitingForApprove(settings, locationApproves, location));
   }

   return result;
}

function getWaitingForApprove(
   perLocationSettings: PerLocationSettings,
   locationApproves: LocationApproves,
   location: string,
): boolean {
   const { needApproval } = perLocationSettings;

   if (!needApproval.has(location)) {
      return false;
   }

   if (locationApproves.get(location)?.status === LocationApprovalStatus.Approved) {
      return false;
   }

   return true;
}
