import { deepClone, plural } from '@yandex-infracloud-ui/libs';

import {
   DeployUnitState,
   DuStatusLabel,
   Stage,
   duAllPrefix,
   pluralDeployUnits,
   StageStatus,
   StageStatusLabel,
   StageStatusState,
   getDisabledStageClusters,
} from '../../stage';
import { DeployUnitStatusView, MergeStageStatusLabel, StageStatusView } from './model';

type DeployUnitCountByState = Map<DeployUnitState, number>;
const getEmptyDeployUnitCountByState: () => DeployUnitCountByState = () =>
   new Map([
      [DeployUnitState.Ready, 0],
      [DeployUnitState.InProgress, 0],
      [DeployUnitState.Failed, 0],
      [DeployUnitState.Unknown, 0],
   ]);

export function getDeployUnitCountByState({
   deployUnitLabels,
}: {
   deployUnitLabels: Map<string, DuStatusLabel>;
}): Map<DeployUnitState, number> {
   const countMap: Map<DeployUnitState, number> = getEmptyDeployUnitCountByState();

   for (const duLabel of deployUnitLabels.values()) {
      countMap.set(duLabel.state, (countMap.get(duLabel.state) ?? 0) + 1);
   }

   return countMap;
}

type GetStageStatusViewParams = {
   stageStatus: StageStatus;
   stage: Stage;
   stageStatusLabel: StageStatusLabel;

   deployUnitData: DeployUnitStatusView[];
};

type GetStageStatusViewResult = {
   stageStatusView: StageStatusView;
};

export function getStageStatusView({
   stage,
   stageStatus,
   stageStatusLabel,
   deployUnitData,
}: GetStageStatusViewParams): GetStageStatusViewResult {
   const stageStatusView: StageStatusView = {
      ...stageStatus,
      spec: stage,
      stageStatusLabel,
      locations: new Set(stage.deployUnits.flatMap(du => Object.keys(du.locations))),
      disabledLocations: getDisabledStageClusters(stage),
      waitingForApproveDuCount: deployUnitData.reduce((sum, du) => sum + (du.waitingForApprove ? 1 : 0), 0),
      mergeStageStatusLabel: {} as any, // заполняется ниже
      lastUpdateTimestamp: 0, // заполняется ниже
   };

   stageStatusView.mergeStageStatusLabel = getMergeStageStatusLabel({ stageStatusLabel, deployUnitData });

   stageStatusView.lastUpdateTimestamp = Date.now();

   return { stageStatusView };
}

function getMergeStageStatusLabel({
   stageStatusLabel,
   deployUnitData,
}: {
   stageStatusLabel: StageStatusLabel;
   deployUnitData: DeployUnitStatusView[];
}): MergeStageStatusLabel {
   const label: MergeStageStatusLabel = { ...deepClone(stageStatusLabel), calculate: false };

   if (label.state === StageStatusState.InProgress && deployUnitData.length > 0) {
      const allDuAreReady = deployUnitData.every(view => view.mergeDeployUnitLabel.state === DeployUnitState.Ready);

      if (allDuAreReady) {
         const calculateCount = deployUnitData
            .map(view => view.mergeDeployUnitLabel.calculate)
            .reduce((sum, calculate) => sum + Number(calculate), 0);

         if (calculateCount > 0) {
            const isAllReady = calculateCount === deployUnitData.length;
            const total = deployUnitData.length;
            label.state = StageStatusState.Ready;
            label.text = `${duAllPrefix(calculateCount, total)}${pluralDeployUnits(calculateCount)} ${plural(
               calculateCount,
               'has',
               'have',
            )} ready pods`;

            label.additional = `${calculateCount}/${deployUnitData.length} DU have ready pods${
               !isAllReady ? `, ${deployUnitData.length - calculateCount}/${deployUnitData.length} DU are ready` : ''
            }`;
            label.calculate = true;
         }
      }
   }

   return label;
}
