import { deepClone, getSetDifference, isEmpty, isEqual } from '@yandex-infracloud-ui/libs';
import { GetFieldsByType } from '../typeHelpers';

// format-off
// prettier-ignore
type EmptyValue<T> =
   T extends string ? '' :
   T extends number ? 0 :
   T extends boolean ? false :
   T extends Array<any> ? [] :
   {};
// format-on

/**
 * @deprecated use specific patch function as patchString, patchNumber, patchObject
 */
export function patch<T extends object, K extends keyof T>(
   obj: T,
   field: K,
   emptyValue: EmptyValue<T[K]>,
   patcher: (v: NonNullable<T[K]>) => T[K] | undefined | null,
) {
   // if 'constructor' object keys only #DEPLOY-5610
   if (obj.hasOwnProperty(field)) {
      const oldValue = obj[field];
      const newValue = patcher(oldValue as any); // TODO typings
      if (newValue === undefined || newValue === null) {
         delete obj[field];
      } else {
         obj[field] = newValue;
      }
      return;
   }

   const newValue = patcher(deepClone(emptyValue) as any);
   if (!(newValue === undefined || newValue === null) && !isEqual(newValue, emptyValue as any)) {
      obj[field] = newValue;
   }
}

export function patchString<T extends object, K extends GetFieldsByType<T, string | undefined>>(
   obj: T,
   field: K,
   patcher: (v: NonNullable<T[K]>) => T[K] | undefined | null,
) {
   return patch(obj, field, '' as EmptyValue<T[K]>, patcher);
}

export function patchNumber<T extends object, K extends GetFieldsByType<T, number | undefined>>(
   obj: T,
   field: K,
   patcher: (v: NonNullable<T[K]>) => T[K] | undefined | null,
) {
   return patch(obj, field, 0 as EmptyValue<T[K]>, patcher);
}

export function patchBoolean<T extends object, K extends GetFieldsByType<T, boolean | undefined>>(
   obj: T,
   field: K,
   patcher: (v: NonNullable<T[K]>) => T[K] | undefined | null,
) {
   return patch(obj, field, false as EmptyValue<T[K]>, patcher);
}

type NonObjectTypes = boolean | number | string | null | Array<any> | Date | RegExp | Set<any> | Map<string, any>;

export function patchObject<T extends object, K extends GetFieldsByType<T, NonObjectTypes, false>>(
   obj: T,
   field: K,
   patcher: (v: NonNullable<T[K]>) => T[K] | undefined,
) {
   return patch(obj, field, {} as EmptyValue<T[K]>, patcher);
}

export function patchList<T extends object, K extends GetFieldsByType<T, Array<any> | undefined>>(
   obj: T,
   field: K,
   patcher: (v: NonNullable<T[K]>) => T[K] | undefined,
) {
   return patch(obj, field, [] as EmptyValue<T[K]>, patcher);
}

type ListItem<List extends Array<Record<string, any>>> = List extends Array<infer X> ? X : never;

// TODO: возможно стоит ввести необязательные порядки для минимального диффа
/**
 * Патчинг поля-массива(из объектов) по ключу с сохранением порядка
 * @param obj родительсий объект
 * @param field поле, которое патчим
 * @param getId получение ключа из элемента массива
 * @param newIds все новые ключи для массива
 * @param patcher функция преобразования
 */
export function patchObjectListById<
   T extends Record<string, any>,
   K extends GetFieldsByType<T, Record<string, any>[] | undefined>,
   List extends T[K] & Record<string, any>[],
   GetId extends (e: ListItem<T[K]>) => string
>(
   obj: T,
   field: K,
   getId: GetId,
   newIds: string[] | Iterable<string>,
   patcher: (v: Partial<ListItem<T[K]>>, id: string) => ListItem<List> | undefined | null,
) {
   const list: ListItem<List>[] = obj[field] ?? [];

   const oldRawMap = new Map(list.map(e => [getId(e), e]));
   const oldIds = [...oldRawMap.keys()];

   const { removed, added } = getSetDifference(new Set(oldIds), new Set(newIds));

   const isEmptyValue = (item: unknown) => item === null || item === undefined;

   // патчим старые значения
   const resultList = oldIds
      .map(id => {
         if (removed.has(id)) {
            return null;
         }
         if (oldRawMap.has(id)) {
            return patcher(oldRawMap.get(id)!, id);
         }
         return null;
      })
      .filter(item => !isEmptyValue(item));

   // добавляем новые в конец
   for (const id of added.values()) {
      const emptyValue: List = {} as List;
      const value = patcher(emptyValue, id) ?? null;
      if (!isEmptyValue(value)) {
         resultList.push(value);
      }
   }

   if (!isEqual(resultList, [])) {
      // непустой список записываем в поле
      obj[field] = resultList as T[K];
   } else if (field in obj) {
      // иначе если поле было и список пуст, удаляем его
      delete obj[field];
   }
   // иначе ничего не делаем - поля не было и записывать нечего
}

export function skipEmpty<T>(obj: T): T | undefined {
   return isEmpty(obj) ? undefined : obj;
}
