import { StatusState } from '../../../components/lib';
import {
   EEvictionState,
   EHfsmState,
   ENodeMaintenanceState,
   EPodCurrentState,
   EPodMaintenanceState,
   EResourceKind,
   ESchedulingState,
   TResource,
} from '../../../proto-typings';

export interface Resource {
   kind: EResourceKind;
   total: number;
   used: number;
   free: number;
   id: string;
}

export interface CPUResource extends Resource {
   cpuToVcpuFactor: number;
}

export interface DiskResource extends Resource {
   bandwidthTotal: number;
   bandwidthUsed: number;
   bandwidthFree: number;
   slotsTotal: number;
   slotsUsed: number;
   slotsFree: number;
   storageClass: string;
   storageProvisioner: string;
}

export interface GPUResource extends Resource {
   model: string;
   memory: number;
}

export function getResourcesByKind(resources: TResource[] | undefined, kind: EResourceKind): Resource[] | undefined {
   if (!resources) {
      return undefined;
   }

   const resourcesByKind = resources.filter(res => res.meta?.kind === kind);

   if (resources && resourcesByKind.length) {
      /* eslint-disable @typescript-eslint/no-non-null-asserted-optional-chain */
      switch (kind) {
         case EResourceKind.RK_CPU:
            return resourcesByKind.map(resourceData => {
               const resource: CPUResource = {
                  kind,
                  total: resourceData.spec?.cpu?.total_capacity!,
                  used: resourceData.status?.used?.cpu?.capacity!,
                  free: resourceData.status?.free?.cpu?.capacity!,
                  id: resourceData.meta?.id!,
                  cpuToVcpuFactor: resourceData.spec?.cpu?.cpu_to_vcpu_factor!,
               };

               return resource;
            });
         case EResourceKind.RK_DISK:
            return resourcesByKind.map(resourceData => {
               const resource: DiskResource = {
                  kind,
                  total: resourceData.spec?.disk?.total_capacity!,
                  used: resourceData.status?.used?.disk?.capacity!,
                  free: resourceData.status?.free?.disk?.capacity!,
                  bandwidthTotal: resourceData.spec?.disk?.total_bandwidth!,
                  bandwidthUsed: resourceData.status?.used?.disk?.bandwidth!,
                  bandwidthFree: resourceData.status?.free?.disk?.bandwidth!,
                  storageClass: resourceData.spec?.disk?.storage_class!,
                  slotsTotal: resourceData.spec?.disk?.total_volume_slots ?? 0,
                  slotsUsed: resourceData.status?.used?.disk?.volume_slots ?? 0,
                  slotsFree:
                     resourceData.status?.free?.disk?.volume_slots ?? resourceData.spec?.disk?.total_volume_slots ?? 0,
                  storageProvisioner: resourceData.spec?.disk?.storage_provisioner ?? 'unknown',
                  id: resourceData.meta?.id!,
               };

               return resource;
            });
         case EResourceKind.RK_GPU:
            return Object.values(
               resourcesByKind.reduce((res, resourceItem) => {
                  const groupKey = `${resourceItem.spec?.gpu?.model}:${resourceItem.spec?.gpu?.total_memory}`;

                  const existingItem = res[groupKey];

                  if (existingItem) {
                     existingItem.total += 1;
                     existingItem.used += resourceItem.status?.used?.gpu?.capacity!;
                  } else {
                     const newItem: GPUResource = {
                        id: `${groupKey}:${resourceItem.meta?.id}`,
                        kind,
                        memory: resourceItem.spec?.gpu?.total_memory!,
                        model: resourceItem.spec?.gpu?.model!,
                        total: 1,
                        free: 0,
                        used: resourceItem.status?.used?.gpu?.capacity!,
                     };

                     res[groupKey] = newItem;
                  }

                  return res;
               }, {} as Record<string, GPUResource>),
            );
         case EResourceKind.RK_MEMORY:
            return resourcesByKind.map(resourceData => ({
               kind,
               total: resourceData.spec?.memory?.total_capacity!,
               used: resourceData.status?.used?.memory?.capacity!,
               free: resourceData.status?.free?.memory?.capacity!,
               id: resourceData.meta?.id!,
            }));
         case EResourceKind.RK_NETWORK:
            return resourcesByKind.map(resourceData => ({
               kind,
               total: resourceData.spec?.network?.total_bandwidth!,
               used: resourceData.status?.used?.network?.bandwidth!,
               free: resourceData.status?.free?.network?.bandwidth!,
               id: resourceData.meta?.id!,
            }));
         case EResourceKind.RK_SLOT:
            return resourcesByKind.map(resourceData => ({
               kind,
               total: resourceData.spec?.slot?.total_capacity!,
               used: resourceData.status?.used?.slot?.capacity!,
               free: resourceData.status?.free?.slot?.capacity!,
               id: resourceData.meta?.id!,
            }));
         default:
            return undefined;
         /* eslint-enable @typescript-eslint/no-non-null-asserted-optional-chain */
      }
   } else {
      return undefined;
   }
}

export const NodeStateMap = new Map<EHfsmState, StatusState>([
   [EHfsmState.HS_UP, StatusState.Ok],
   [EHfsmState.HS_PROBATION, StatusState.Planned],
   [EHfsmState.HS_SUSPECTED, StatusState.Planned],
   [EHfsmState.HS_PREPARE_MAINTENANCE, StatusState.Planned],
   [EHfsmState.HS_DOWN, StatusState.Error],
   [EHfsmState.HS_INITIAL, StatusState.Inactive],
   [EHfsmState.HS_MAINTENANCE, StatusState.Planned],
   [EHfsmState.HS_UNKNOWN, StatusState.Inactive],
]);

export const NodeMaintenanceStateMap = new Map<ENodeMaintenanceState, StatusState>([
   [ENodeMaintenanceState.NMS_NONE, StatusState.Inactive],
   [ENodeMaintenanceState.NMS_REQUESTED, StatusState.Planned],
   [ENodeMaintenanceState.NMS_ACKNOWLEDGED, StatusState.Planned],
   [ENodeMaintenanceState.NMS_IN_PROGRESS, StatusState.Progress],
]);

export const EvictionStateMap = new Map<EEvictionState, StatusState>([
   [EEvictionState.ES_NONE, StatusState.Inactive],
   [EEvictionState.ES_REQUESTED, StatusState.Planned],
   [EEvictionState.ES_ACKNOWLEDGED, StatusState.Ok],
]);

export const MaintenanceStateMap = new Map<EPodMaintenanceState, StatusState>([
   [EPodMaintenanceState.PMS_NONE, StatusState.Inactive],
   [EPodMaintenanceState.PMS_REQUESTED, StatusState.Planned],
   [EPodMaintenanceState.PMS_IN_PROGRESS, StatusState.Progress],
   [EPodMaintenanceState.PMS_ACKNOWLEDGED, StatusState.Ok],
]);

export const SchedulingStateMap = new Map<ESchedulingState, StatusState>([
   [ESchedulingState.SS_NONE, StatusState.Inactive],
   [ESchedulingState.SS_DISABLED, StatusState.Inactive],
   [ESchedulingState.SS_PENDING, StatusState.Progress],
   [ESchedulingState.SS_ASSIGNED, StatusState.Ok],
]);

export const StatusMap = new Map<EPodCurrentState, StatusState>([
   [EPodCurrentState.PCS_UNKNOWN, StatusState.Unknown],
   [EPodCurrentState.PCS_STARTED, StatusState.Ok],
   [EPodCurrentState.PCS_START_PENDING, StatusState.Progress],
   [EPodCurrentState.PCS_STOP_PENDING, StatusState.Progress],
   [EPodCurrentState.PCS_START_FAILED, StatusState.Error],
   [EPodCurrentState.PCS_STOPPED, StatusState.Inactive],
]);

export enum ReplicaSetState {
   IN_PROGRESS = 'In progress',
   READY = 'Ready',
   FAILED = 'Failed',
}

export interface ReplicaSetFiltersParams {
   replicaSetId?: string;
   stageId?: string;
   state?: ReplicaSetState[];
   revision?: string;
   query?: string;
}

export const replicaSetStatusMap = new Map<ReplicaSetState, StatusState>([
   [ReplicaSetState.FAILED, StatusState.Error],
   [ReplicaSetState.IN_PROGRESS, StatusState.Progress],
   [ReplicaSetState.READY, StatusState.Ok],
]);
