import { toQuery } from '@yandex-infracloud-ui/libs';

import { SidecarType } from '../services/api/services/InfracomponentsApi';

import { StageLogsParams } from './ui';

// noinspection ES6PreferShortImport (circular)
import { DefaultReleaseRuleType, DeployTicket, ReleaseRuleType } from './ui/releases';

type QueryParams = Record<string, any>;

export type YPEntityTypes = 'nodes' | 'pods' | 'pod-sets' | 'replica-sets' | 'multicluster-replica-sets';

class UrlBuilder {
   private static baseStage(stageId: string) {
      return `/stages/${stageId}`;
   }

   private static buildQuery(params: QueryParams | undefined) {
      if (!params) {
         return '';
      }

      const queryString = toQuery(params);

      return queryString ? `?${queryString}` : '';
   }

   /**
    * хардкод во избежание циклической зависимости
    * TODO nikolaichev@ можно вынести в отдельный файл
    */
   private static stageFormInternal(deployUnitId?: string, boxId?: string, workloadId?: string) {
      let result = '';

      if (deployUnitId) {
         result += `/du-${deployUnitId}`;

         if (boxId) {
            result += `/box-${boxId}`;

            if (workloadId) {
               result += `/wl-${workloadId}`;
            }
         }
      }

      return result;
   }

   public build(path: string, params: { [k: string]: string }) {
      return path.replace(/:(\w+)\b/g, (match, param) => (param in params ? params[param] : match));
   }

   public admin() {
      return '/admin/summary';
   }

   public adminSummary() {
      return '/admin/summary';
   }

   public adminInfraComponentsSidecar(sidecar: SidecarType) {
      return `/admin/infracomponents/${sidecar}`;
   }

   public balancers() {
      return '/balancers';
   }

   public balancerCreate(stageId: string) {
      return `/balancers/create/${stageId}`;
   }

   public awacsNamespace(namespaceId: string) {
      return `/balancers/${namespaceId}`;
   }

   public awacsNamespaceMap(namespaceId: string) {
      return `${this.awacsNamespace(namespaceId)}/map`;
   }

   public changelog() {
      return '/changelog';
   }

   public deployTicket(stageId: string, id: string) {
      return `${this.stageDeployTickets(stageId)}/${id}`;
   }

   public home() {
      return this.projects();
   }

   public newStage(projectId?: string) {
      return `/new-stage${UrlBuilder.buildQuery({ projectId })}`;
   }

   public project(projectId: string) {
      return `/projects/${projectId}`;
   }

   public projects() {
      return `/${UrlBuilder.buildQuery({ my: 'yes' })}`;
   }

   public releasePage(releaseId: string) {
      return `/releases/${releaseId}`;
   }

   public releaseRuleCreate(stageId: string, type: ReleaseRuleType) {
      const params = {
         type: type !== DefaultReleaseRuleType ? type : undefined,
      };

      return `${UrlBuilder.baseStage(stageId)}/release_rules/new${UrlBuilder.buildQuery(params)}`;
   }

   public releaseRulePage(stageId: string, id: string) {
      return `${UrlBuilder.baseStage(stageId)}/release_rules/${id}`;
   }

   /**
    * Домашняя страница стейджа
    *
    * В текущей версии это страница статуса
    */
   public stage(stageId: string) {
      return this.stageStatus(stageId);
   }

   public stageBalancers(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/balancers`;
   }

   public stageConfig(stageId: string, deployUnitId?: string, boxId?: string, workloadId?: string) {
      return `${UrlBuilder.baseStage(stageId)}/config${UrlBuilder.stageFormInternal(deployUnitId, boxId, workloadId)}`;
   }

   public stageYaml(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/yaml`;
   }

   public stageDeployTickets(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/deploy-tickets`;
   }

   public stageEdit(stageId: string, options?: { duId?: string; boxId?: string; workloadId?: string }) {
      const { duId, boxId, workloadId } = options ?? {};
      return `${UrlBuilder.baseStage(stageId)}/edit${duId ? `/du-${duId}` : ''}${boxId ? `/box-${boxId}` : ''}${
         workloadId ? `/wl-${workloadId}` : ''
      }`;
   }

   public stageEditRaw(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/edit-raw`;
   }

   public stageEditDeployUnitDisks(stageId: string, duId: string) {
      return `${UrlBuilder.baseStage(stageId)}/edit/du-${duId}#disks`;
   }

   public stageHistory(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/history`;
   }

   public stageHistoryRevision(stageId: string, revisionId: number) {
      return `${this.stageHistory(stageId)}/${revisionId}`;
   }

   public stageHistoryApply(stageId: string, revisionId: number, asIs = false) {
      return `${this.stageHistoryRevision(stageId, revisionId)}/${asIs ? 'applyAsIs' : 'apply'}`;
   }

   public stageHistoryCompare(stageId: string, revisionId: number, revisionComparedId: number) {
      return `${this.stageHistoryRevision(stageId, revisionId)}/diff/${revisionComparedId}`;
   }

   public stageJson(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/json`;
   }

   public stageLogs(stageId: string, params?: StageLogsParams) {
      return `${UrlBuilder.baseStage(stageId)}/logs${UrlBuilder.buildQuery(params)}`;
   }

   public stageMonitoring(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/monitoring`;
   }

   public stageMonitoringLogs(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/monitoring/logs`;
   }

   public stageStatus(stageId: string, deployUnitId?: string, clusterId?: string, params?: QueryParams) {
      let url = UrlBuilder.baseStage(stageId);
      if (deployUnitId) {
         url += `/status/${deployUnitId}`;
         if (clusterId) {
            url += `/${clusterId}`;
         }
      }

      return `${url}${UrlBuilder.buildQuery(params)}`;
   }

   public stageStatusInfo(stageId: string, options?: { params: QueryParams }) {
      const url = UrlBuilder.baseStage(stageId);
      return `${url}/status/~stage-info${UrlBuilder.buildQuery(options?.params)}`;
   }

   public stageUpdate(stageId: string) {
      return `${UrlBuilder.baseStage(stageId)}/update`;
   }

   public ticketHighlightedPage(ticket: DeployTicket) {
      return `${this.stageDeployTickets(ticket.stageId)}/${ticket.id}`;
   }

   public yp() {
      return `/yp`;
   }

   public ypEntities(cluster: string, entityType: YPEntityTypes) {
      return `/yp/${cluster}/${entityType}`;
   }

   public ypNode(cluster: string, nodeId: string) {
      return `/yp/${cluster}/nodes/${nodeId}`;
   }

   public ypPod(cluster: string, podId: string) {
      return `/yp/${cluster}/pods/${podId}`;
   }

   public ypPodSet(cluster: string, podSetId: string) {
      return `/yp/${cluster}/pod-sets/${podSetId}`;
   }

   public ypPodSetPods(cluster: string, podSetId: string) {
      return `/yp/${cluster}/pod-sets/${podSetId}#pods`;
   }

   public ypReplicaSet(cluster: string, replicaSetId: string) {
      return `/yp/${cluster}/replica-sets/${replicaSetId}`;
   }

   public ypMulticlusterReplicaSet(replicaSetId: string) {
      return `/yp/xdc/multicluster-replica-sets/${replicaSetId}`;
   }

   public ypMulticlusterReplicaSets() {
      return '/yp/xdc/multicluster-replica-sets';
   }

   public ypQuotas() {
      return '/quotas';
   }
}

export const urlBuilder = new UrlBuilder();
