import React from 'react';

import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { ExtendedFieldConfig } from '@yandex-infracloud-ui/libs';
import { Formik, FormikConfig } from 'formik';
import { Button } from '@yandex-cloud/uikit';
import { array, object, string } from 'yup';

import { DevForm } from '../../../../components/lib';
import { apiServicesContextDecorator, modalsSupportDecorator } from '../../../../stories';
import { stageWithSecretsDecorator } from '../../../secrets/storybook/stageWithSecretsDecorator';
import { EnvironmentVariable, EnvironmentVariableType, LiteralVariable, SecretVariable } from '../../models';

import { EnvSubForm } from './EnvSubForm';

// noinspection SpellCheckingInspection
const initialValues = {
   default: [
      {
         type: EnvironmentVariableType.Literal,
         name: 'LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_NAME',
         value: 'LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_LONG_VALUE',
      } as LiteralVariable,
      {
         type: EnvironmentVariableType.Literal,
         name: 'HTTP_PORT',
         value: '8080',
      } as LiteralVariable,
      {
         type: EnvironmentVariableType.Literal,
         name: 'AUTH_KEY',
         value: 'ZGV2dG9vbHM6Zjg3ZGV2dG9vbHM6Zjg3ZGV2dG9vbHM6Zjg3',
      } as LiteralVariable,
      {
         type: EnvironmentVariableType.Secret,
         name: 'SECRET_VAR',
         value: {
            alias: 'INFRACLOUDUI',
            key: 'BITBUCKET_TOKEN',
         },
      } as SecretVariable,
      {
         type: EnvironmentVariableType.Secret,
         name: 'DENIED_VAR',
         value: {
            alias: 'DENIED_SECRET',
            key: 'reddi-test',
         },
      },
   ] as EnvironmentVariable[],
   withError: [
      {
         type: EnvironmentVariableType.Literal,
         name: ' INCORRECT_NAME',
         value: 'incorrect value ',
      } as LiteralVariable,
   ],
};

type FormValue = typeof initialValues;

function FieldStory(props: ExtendedFieldConfig<FormValue, any, any>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true as any,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         withError: array().of(string()).required(),
         xField: array().of(string()).min(10),
      }),
   };

   return (
      <Formik {...config}>
         {() => (
            <div style={{ ['--forms-field-layout' as any]: 'column' }}>
               <EnvSubForm readonly={boolean('readonly', false)} name={props.name} />
               <Button view={'action'} type={'submit'}>
                  Submit
               </Button>
               <DevForm />
            </div>
         )}
      </Formik>
   );
}

export const Default = () => <FieldStory name={'default'} label={'Default'} />;

export const WithError = () => <FieldStory name={'withError'} label={'With error'} />;

export const Playground = () => (
   <FieldStory
      name={'default'}
      label={text('label', 'label')}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
      controlProps={{}}
   />
);

export default {
   decorators: [modalsSupportDecorator, apiServicesContextDecorator, stageWithSecretsDecorator('du_manual_secrets')],
   title: 'modules/environment/EnvSubForm',
} as Meta;
