import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { classNames, EMPTY_VALUE, FormButton, modalService, sortHandler } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import tableClasses from '../../../../design/simpleTable.module.css';

import { noop } from '../../../../utils';
import { useSecretsContext } from '../../../secrets';

import { EnvironmentVariable } from '../../models';

import { EnvSubFormRow } from './__row/EnvSubFormRow';
import { EnvSubFormVariableModal } from './__variableModal/EnvSubFormVariableModal';
import classes from './EnvSubForm.module.css';

interface Props {
   name: string;
   readonly: boolean;
}

export const EnvSubForm: React.FC<Props> = React.memo(({ readonly, name }) => {
   const { duId, stageId, reloadUsages } = useSecretsContext();
   const form = useFormikContext();
   const { value } = form.getFieldMeta<EnvironmentVariable[]>(name);

   const openAddModal = useCallback(() => {
      modalService
         .open(
            EnvSubFormVariableModal,
            {
               duId,
               isNew: true,
               nameList: value.map(v => v.name),
               readonly: false,
               stageId,
            },
            { closeOnOutsideClick: false, dialogProps: { size: 'm' } },
         )
         .subscribe(values => {
            form.setFieldValue(name, [...value, ...values]);
            reloadUsages();
         }, noop);
   }, [duId, value, stageId, form, name, reloadUsages]);

   const openEditModal = useCallback(
      (variableName: string) => {
         const variable = value.find(v => v.name === variableName);

         modalService
            .open(
               EnvSubFormVariableModal,
               {
                  duId,
                  isNew: false,
                  nameList: value.filter(v => v.name !== variableName).map(v => v.name),
                  readonly,
                  stageId,
                  variable,
               },
               { closeOnOutsideClick: readonly, dialogProps: { size: 'm' } },
            )
            .subscribe(values => {
               form.setFieldValue(
                  name,
                  value.map(v => (v === variable ? values[0] : v)),
               );
               reloadUsages();
            }, noop);
      },
      [value, duId, readonly, stageId, form, name, reloadUsages],
   );

   const handleRemove = useCallback(
      (variableName: string) => {
         form.setFieldValue(
            name,
            value.filter(v => v.name !== variableName),
         );
         reloadUsages();
      },

      [form, name, value, reloadUsages],
   );

   return (
      <>
         {value.length === 0 ? (
            <>{readonly ? <div className={classes.empty}>{EMPTY_VALUE}</div> : null}</>
         ) : (
            <table className={classNames(tableClasses.simpleTable, classes.table, { [classes.readonly]: readonly })}>
               <thead>
                  <tr>
                     <th>Name</th>
                     <th>Value</th>
                     <th aria-label={'Copy'} />
                     <th aria-label={'Actions'} />
                  </tr>
               </thead>
               <tbody>
                  {Array.from(value)
                     .sort((a, b) => sortHandler(a.name.toLowerCase(), b.name.toLowerCase()))
                     .map(v => (
                        <EnvSubFormRow
                           key={v.name}
                           value={v}
                           onSelect={openEditModal}
                           onRemove={handleRemove}
                           readonly={readonly}
                        />
                     ))}
               </tbody>
            </table>
         )}

         {!readonly && (
            <FormButton onClick={openAddModal} icon={faPlus} skipLeftSpace={true}>
               Add variable
            </FormButton>
         )}
      </>
   );
});

EnvSubForm.displayName = 'EnvSubForm';
