// noinspection RedundantIfStatementJS

import {
   DISMISS_REASON,
   FormButton,
   IModalProps,
   isEmpty,
   ModalLayout,
   SmartTab,
   SmartTabs,
} from '@yandex-infracloud-ui/libs';
import React, { useCallback, useState } from 'react';

import { DevJson } from '../../../../../components/lib';
import { EnvironmentVariable, EnvironmentVariableType, LiteralVariable, SecretVariable } from '../../../models';
import { EnvSubFormIcon } from '../__icon/EnvSubFormIcon';
import {
   EnvSubFormLiteralForm,
   getLiteralFormParams,
   literalInitialFormParams,
} from '../__literalForm/EnvSubFormLiteralForm';
import {
   EnvSubFormSecretForm,
   getSecretFormParams,
   secretInitialFormParams,
} from '../__secretForm/EnvSubFormSecretForm';

import classes from './EnvSubFormVariableModal.module.css';

interface Props extends IModalProps<EnvironmentVariable[]> {
   duId: string;
   isNew: boolean;
   nameList: string[];
   readonly: boolean;
   stageId: string;
   variable?: EnvironmentVariable;
}

export const EnvSubFormVariableModal: React.FC<Props> = React.memo(
   ({ stageId, duId, cancel, isNew, nameList, ok, readonly, variable }) => {
      if (!isNew && !variable) {
         throw new Error('isNew=false, so you should define `variable` prop');
      }

      const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);
      const [tab, setTab] = useState(EnvironmentVariableType.Literal);
      const [isMultiple, setIsMultiple] = useState(false);

      const [literalFormParams, setLiteralFormParams] = useState(() =>
         isNew || variable!.type !== EnvironmentVariableType.Literal
            ? literalInitialFormParams
            : getLiteralFormParams(variable as LiteralVariable),
      );

      const [secretFormParams, setSecretFormParams] = useState(() =>
         isNew || variable!.type !== EnvironmentVariableType.Secret
            ? secretInitialFormParams
            : getSecretFormParams(variable as SecretVariable),
      );

      const handleOk = useCallback(() => {
         switch (isNew ? tab : variable!.type) {
            case EnvironmentVariableType.Literal: {
               const variables: LiteralVariable[] = isMultiple
                  ? parseVariablesFromStrings(literalFormParams.multipleList)
                  : [
                       {
                          type: EnvironmentVariableType.Literal,
                          name: literalFormParams.name ?? '',
                          value: literalFormParams.value ?? '',
                       } as LiteralVariable,
                    ];
               ok(variables.filter(v => v.type === EnvironmentVariableType.Literal && v.name));

               break;
            }

            case EnvironmentVariableType.Secret: {
               const secretVar: SecretVariable = {
                  type: EnvironmentVariableType.Secret,
                  name: secretFormParams.name,
                  value: secretFormParams.secret,
               };
               ok([secretVar]);

               break;
            }
         }
      }, [isMultiple, isNew, literalFormParams, ok, secretFormParams, tab, variable]);

      const toggleIsMultiple = useCallback(() => {
         setIsMultiple(v => !v);
      }, []);

      const isOkButtonDisabled = useCallback(
         type => {
            switch (type) {
               case EnvironmentVariableType.Literal: {
                  if (isMultiple) {
                     if (literalFormParams.multipleList.size === 0) {
                        return true;
                     }

                     const parsedNames = parseVariablesFromStrings(literalFormParams.multipleList).map(v => v.name);
                     const names: string[] = [];

                     for (const name of parsedNames) {
                        if (names.includes(name) || nameList.includes(name)) {
                           return true;
                        }

                        names.push(name);
                     }

                     return false;
                  }

                  const { name } = literalFormParams;

                  if (name && !isEmpty(name)) {
                     return nameList.includes(name);
                  }

                  break;
               }

               case EnvironmentVariableType.Secret: {
                  const { name, secret } = secretFormParams;

                  if (name && !isEmpty(name) && !isEmpty(secret)) {
                     if (!isEmpty(secret?.alias) && !isEmpty(secret?.key)) {
                        return nameList.includes(name);
                     }
                  }

                  break;
               }
            }

            return true;
         },
         [isMultiple, literalFormParams, nameList, secretFormParams],
      );

      // region render
      if (isNew) {
         const footerLeft =
            tab === EnvironmentVariableType.Literal ? (
               <FormButton onClick={toggleIsMultiple} className={classes.multipleSwitcher} skipLeftSpace={true}>
                  add by {isMultiple ? 'one' : 'list'}
               </FormButton>
            ) : null;

         return (
            <ModalLayout
               title={'Workload variables'}
               onDismiss={dismiss}
               okLabel={'Create'}
               onOk={handleOk}
               disabled={isOkButtonDisabled(tab)}
               footerLeft={footerLeft}
            >
               <SmartTabs activeTab={tab} onSelectTab={setTab as (t: string) => void}>
                  <SmartTab id={EnvironmentVariableType.Literal} title={'Literal variable'}>
                     <EnvSubFormLiteralForm
                        value={literalFormParams}
                        onChange={setLiteralFormParams}
                        isMultiple={isMultiple}
                     />
                     <DevJson>{literalFormParams}</DevJson>
                  </SmartTab>
                  <SmartTab id={EnvironmentVariableType.Secret} title={'Secret'}>
                     <EnvSubFormSecretForm
                        duId={duId}
                        onChange={setSecretFormParams}
                        stageId={stageId}
                        value={secretFormParams}
                     />
                  </SmartTab>
               </SmartTabs>
            </ModalLayout>
         );
      }

      return (
         <ModalLayout
            title={variable!.name}
            subtitle={
               <div className={classes.subtitle}>
                  <EnvSubFormIcon type={variable!.type} showText={true} />
               </div>
            }
            onDismiss={dismiss}
            okLabel={readonly ? 'Ok' : 'Apply'}
            disabled={isOkButtonDisabled(variable!.type)}
            onOk={handleOk}
         >
            {variable!.type === EnvironmentVariableType.Literal ? (
               <EnvSubFormLiteralForm
                  value={literalFormParams}
                  onChange={setLiteralFormParams}
                  isMultiple={isMultiple}
                  readonly={readonly}
               />
            ) : null}

            {variable!.type === EnvironmentVariableType.Secret ? (
               <EnvSubFormSecretForm
                  duId={duId}
                  value={secretFormParams}
                  onChange={setSecretFormParams}
                  stageId={stageId}
                  readonly={readonly}
               />
            ) : null}
         </ModalLayout>
      );
      // endregion
   },
);

EnvSubFormVariableModal.displayName = 'EnvSubFormVariableModal';

function parseVariablesFromStrings(v: Set<string>): LiteralVariable[] {
   return Array.from(v).map(row => {
      const [name, value] = row.split('=').map(s => s.trim());

      return {
         type: EnvironmentVariableType.Literal,
         name,
         value,
      } as LiteralVariable;
   });
}
