import { array, mixed, object, Schema, string, ValidationError } from 'yup';
import { hasOnlyUniqueValues } from '../../../models/ui/stage/validation';

// noinspection ES6PreferShortImport
import { getLinkToSecretValidationSchema } from '../../secrets/helpers';
import { EnvironmentVariable } from './EnvironmentVariable';
import { EnvironmentVariableType } from './EnvironmentVariableType';

/* eslint-disable no-template-curly-in-string */
const environmentVariableSchema = object<EnvironmentVariable>({
   name: string()
      .min(1)
      .required()
      .matches(/^((?!(=)).)*$/, "${path} must not contain '='")
      .test('noWhitespace', '${path} must not contain whitespace', (v: string) => (v ? !/\s/.test(v) : true))
      .label('Variable name'),
   type: (string()
      .label('Variable type')
      .oneOf([EnvironmentVariableType.Literal, EnvironmentVariableType.Secret])
      .required() as Schema<EnvironmentVariableType>) as any, // TODO typings
   value: mixed().when('type', {
      is: EnvironmentVariableType.Literal,
      then: string()
         .nullable()
         .matches(/[^\s]$/, '${path} cannot end with whitespace')
         .label('Literal variable value'),
      otherwise: getLinkToSecretValidationSchema('').required().label('Secret variable value'),
   }),
});

export const environmentValidationSchema = array()
   .of<EnvironmentVariable>(
      // Раньше был lazy, он он пересоздавал схему для каждой переменной, что жутко тормозило
      mixed().test(
         'envVarValid',
         () => '',
         function envVarValid(value: EnvironmentVariable): ValidationError | boolean {
            try {
               environmentVariableSchema.validateSync(value);
            } catch (e) {
               const ve: ValidationError = e as any;
               return new ValidationError(`${value.name}: ${ve.message}`, value, this.path);
            }

            return true;
         },
      ),
   )
   .test('uniqueName', 'Variables should have unique names', (v: EnvironmentVariable[]) =>
      hasOnlyUniqueValues(v.map((p: EnvironmentVariable) => p.name)),
   );
