import { Label } from '@yandex-cloud/uikit';
import { YCSelect } from '@yandex-data-ui/common';
import React, { ReactNode, useMemo } from 'react';
import { IssueClass, issueClasses, IssueLevel, issueLevels } from '../../model';
import { issueClassTitles, labelThemeByLevel, restoreOrder } from '../model';

export interface CommonIssueFilters {
   level: Set<IssueLevel>;
   class: Set<IssueClass>;
}

export const getInitialIssueFilters: () => CommonIssueFilters = () => ({
   level: new Set(),
   class: new Set(),
});

type FieldComponent<T> = (value: T, update: (v: T) => void) => ReactNode;
type FieldComponents<T> = { [K in keyof T]?: FieldComponent<T[K]> };

const defaultFieldComponents: FieldComponents<CommonIssueFilters> = {
   level: (value, update) => (
      <YCSelect
         type={'multiple'}
         value={Array.from(value)}
         showSearch={false}
         items={issueLevels.map(level => ({
            value: level,
            title: <Label theme={labelThemeByLevel[level]}>{level}</Label>,
         }))}
         onUpdate={levels => update(new Set(levels as IssueLevel[]))}
         placeholder={'Level'}
      />
   ),
   class: (value, update) => (
      <YCSelect
         type={'multiple'}
         showSearch={false}
         value={Array.from(value)}
         items={issueClasses.map(issueClass => ({
            value: issueClass,
            title: issueClassTitles[issueClass],
         }))}
         onUpdate={newValue => update(new Set(newValue as IssueClass[]))}
         placeholder={'Class'}
      />
   ),
};

interface Props<T> {
   filters: T;
   updateFilters(newFilters: T): void;
   components?: FieldComponents<T>;
   order?: (keyof T & string)[];
}

export function IssueFilters<T>(props: Props<T>) {
   const { filters, updateFilters, components, order } = props;
   const filterFields = Object.keys(filters) as (keyof T & string)[];
   const targetOrder: (keyof T & string)[] = restoreOrder(order ?? [], ['level', 'class'], filterFields);

   const filtersUpdater = useMemo(() => {
      const createFieldUpdater = <F extends keyof T>(field: F) => (value: T[F]) => {
         const newFilters = { ...filters, [field]: value };
         updateFilters(newFilters);
      };

      return filterFields.reduce((updater, name) => {
         updater[name] = createFieldUpdater(name);
         return updater;
      }, {} as { [K in keyof T & string]-?: (v: T[K]) => void });
   }, [filterFields, filters, updateFilters]);

   return (
      <div style={{ display: 'inline-flex', gap: '16px' }}>
         {targetOrder.map(name => {
            const component: FieldComponent<T[keyof T & string]> | undefined =
               components?.[name] ?? (defaultFieldComponents[name as keyof CommonIssueFilters] as any);
            if (!component) {
               return null;
            }

            return (
               <div key={String(name)} style={{ minWidth: '200px' }}>
                  {component(filters[name], filtersUpdater[name])}
               </div>
            );
         })}
      </div>
   );
}

IssueFilters.displayName = 'IssueFilters';
