import React, { ReactNode } from 'react';

import { Label, Table, withTableSorting } from '@yandex-cloud/uikit';

import { Issue } from '../../model';
import { issueClassTitles, labelThemeByLevel, restoreOrder } from '../model';

type Column = keyof Issue;
const defaultOrder: Column[] = ['description', 'level', 'class', 'locator'];
const defaultColumns: Set<Column> = new Set(defaultOrder);

const defaultColumnSettings: Partial<Record<Column, ColumnSettings<Issue>>> = {
   description: {
      title: 'Description',
      renderItem: issue => issue.description,
   },
   level: {
      title: 'Level',
      renderItem: issue => <Label theme={labelThemeByLevel[issue.level]}>{issue.level}</Label>,
      sort: true,
   },
   class: {
      title: 'Class',
      renderItem: issue => issueClassTitles[issue.class],
      sort: true,
   },
   locator: {
      title: 'Locator',
      renderItem: issue => issue.locator,
      sort: true,
   },
};

type ColumnSettings<I extends Issue> = {
   title: string;
   renderItem: (issue: I) => ReactNode;
   sort?: boolean;
};

type Props<T extends string, I extends Issue> = {
   columns?: Set<Column>;
   customColumns?: Set<T>;

   /**
    * порядок колонок, если колонки не указаны в порядке, то их место будет определено из порядка по умолчанию, либо из лексикографического порядка
    */
   order?: (T | Column)[];
   /**
    * необязательные настройки для колонок по умолчанию
    */
   columnSettings?: Partial<Record<Column | T, ColumnSettings<I>>>;

   data: I[];
};

const SortedTable = withTableSorting(Table);

export function IssueTable<T extends string, I extends Issue>(props: Props<T, I>) {
   const { columns = defaultColumns, order = defaultOrder, columnSettings = defaultColumnSettings, data } = props;
   const targetColumns: Set<Column | T> = new Set(columns);
   const targetSettings: Record<Exclude<T, Column>, ColumnSettings<I>> & Record<Column, ColumnSettings<I>> = {
      ...defaultColumnSettings,
      ...columnSettings,
   } as any;
   if ('customColumns' in props) {
      const { customColumns } = props;
      Array.from(customColumns ?? []).forEach(id => targetColumns.add(id));
   }

   const targetOrder = restoreOrder(order, defaultOrder, Array.from(targetColumns));

   return (
      <SortedTable
         columns={targetOrder.map(id => {
            const d: ColumnSettings<I> | undefined = (targetSettings as any)[id];

            return {
               id,
               name: d?.title ?? id,
               meta: {
                  sort: d?.sort,
               },
            };
         })}
         data={data.map(issue =>
            Object.fromEntries(
               targetOrder.map(id => {
                  const d: ColumnSettings<I> = (targetSettings as any)[id];
                  return [id, d ? d.renderItem(issue) : id];
               }),
            ),
         )}
      />
   );
}
