import { MeasureFormatOptions } from '../../utils';
import { formatBytesPerSecond, formatBytes, formatCores } from './helpers';

export enum ResourceGroupType {
   Single, // группа ресурса состоит из него одного
   Multi, // в группе несколько значений (разные диски, модели gpu)
}

export enum ResourceGroup {
   Cpu = 'cpu',
   Mem = 'mem',
   Disk = 'disk',
   DiskBandwidth = 'diskBandwidth',
   NetworkBandwidth = 'networkBandwidth',
   InternetAddress = 'internetAddress',
   Gpu = 'gpu',
}

// export const KnownResourceNames = {
//    [ResourceGroup.Disk]: new Set(['ssd', 'hdd']),
//    [ResourceGroup.DiskBandwidth]: new Set(['ssd', 'hdd']),
// };

export interface ResourceView<UnitName extends string> {
   getTitle: (resourceName: string) => string; // заголовок как функция от имени
   getValueView: (value: number, options?: MeasureFormatOptions<UnitName>) => string; // вывод форматированного значения
}

export interface ResourceGroupModel<UnitName extends string> {
   groupType: ResourceGroupType;
   groupTitle: string;
   resource: ResourceView<UnitName>; // отображение ресурса
}

function model<UnitName extends string>(s: ResourceGroupModel<UnitName>): ResourceGroupModel<UnitName> {
   return s;
}

export const resourcesModelRaw = {
   [ResourceGroup.Cpu]: model({
      groupType: ResourceGroupType.Single,
      groupTitle: 'CPU',
      resource: {
         getValueView: formatCores,
         getTitle: () => 'CPU',
      },
   }),
   [ResourceGroup.Mem]: model({
      groupType: ResourceGroupType.Single,
      groupTitle: 'RAM',
      resource: {
         getValueView: formatBytes,
         getTitle: () => 'RAM',
      },
   }),
   [ResourceGroup.Disk]: model({
      groupType: ResourceGroupType.Multi,
      groupTitle: 'Disk',
      resource: {
         getValueView: formatBytes,
         getTitle: name => name.toUpperCase(),
      },
   }),
   [ResourceGroup.DiskBandwidth]: model({
      groupType: ResourceGroupType.Multi,
      groupTitle: 'Disk Bandwidth',
      resource: {
         getValueView: formatBytesPerSecond,
         getTitle: name => name.toUpperCase(),
      },
   }),
   [ResourceGroup.NetworkBandwidth]: model({
      groupType: ResourceGroupType.Single,
      groupTitle: 'Network bandwidth',
      resource: {
         getValueView: formatBytesPerSecond,
         getTitle: () => 'Network bandwidth',
      },
   }),
};

type ResourcesModel = Record<ResourceGroup, ResourceGroupModel<string> | undefined>;

export const resourcesModel = resourcesModelRaw as ResourcesModel;

export const volumeStorageClasses = ['ssd', 'hdd'];

export const mainSegmentsConfig = {
   dev: {
      title: 'Development',
   },
   default: {
      title: 'Default',
   },
};
