import { Button } from '@yandex-cloud/uikit';
import {
   CheckboxField2,
   DISMISS_REASON,
   FixedFormik,
   FormHooks,
   FormHooksContext,
   IModalProps,
   InputField2,
   ModalLayout,
} from '@yandex-infracloud-ui/libs';
import { Form, FormikProps } from 'formik';
import React, { useCallback, useMemo, useRef } from 'react';
import { useStore } from 'react-redux';
import { boolean, object, string } from 'yup';

// noinspection ES6PreferShortImport
import { DevForm } from '../../../../components/lib';
import { NEW_SECRET_STORAGE_BY_DEFAULT } from '../../../../models';
import { getAvailableAliases, secretAliasSchema, SecretValidationContext } from '../../helpers';
import { useUsedAliases } from '../../hooks';
import { useSecretsContext } from '../../hooks/useSecretsContext';
import { RootStateWithSecrets, selectSecret } from '../../slice';
import { AliasHint, NewStorageHint } from '../hints';

import { NewSecretSelectField } from '../NewSecretSelectField/NewSecretSelectField';
import { SecretVersionField } from '../SecretVersionField/SecretVersionField';

import classes from './AddSecretModal.module.css';

const initialValues = {
   alias: '',
   newStorage: NEW_SECRET_STORAGE_BY_DEFAULT,
   secretUuid: '',
   versionUuid: '',
};

type FormValue = typeof initialValues;

const validationSchema = object<FormValue>({
   alias: secretAliasSchema,
   newStorage: boolean().label('New storage'),
   secretUuid: string().label('Secret').required(),
   versionUuid: string().label('Version').required(),
});

export const AddSecretModal: React.FC<IModalProps<FormValue>> = ({ ok, cancel }) => {
   const { duId } = useSecretsContext();

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const store = useStore<RootStateWithSecrets>();
   const formRef = useRef<FormikProps<FormValue>>();

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange(field: keyof FormValue) {
               if (!formRef.current) {
                  return;
               }

               const state = store.getState();

               // Обновляет алиас, если его еще не трогали
               if (field === 'versionUuid') {
                  if (!formRef.current.touched.alias) {
                     const { secretUuid } = formRef.current.values;

                     const secret = selectSecret(state, secretUuid);
                     const [alias] = getAvailableAliases(secret ?? { uuid: secretUuid }, [], [] /* TODO */);

                     formRef.current.setFieldValue('alias', alias);
                     formRef.current.setFieldTouched('alias', true, true);
                  }
               }
            },
         } as FormHooks<FormValue>),
      [store],
   );

   const usedAliases = useUsedAliases();
   const validationContext: SecretValidationContext = useMemo(() => ({ usedAliases }), [usedAliases]);

   return (
      <ModalLayout
         title={'Add secret'}
         onDismiss={dismiss}
         showFooter={false}
         subtitle={`to deploy unit "${duId}"`}
         data-e2e={'AddSecretModal'}
      >
         <FixedFormik
            initialValues={initialValues}
            onSubmit={ok}
            validationSchema={validationSchema}
            validationContext={validationContext}
         >
            {form => {
               formRef.current = form;
               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <Form className={classes.form}>
                        <NewSecretSelectField
                           label={'Secret'}
                           name={'secretUuid'}
                           required={true}
                           data-test={'AddSecretModal-Secret'}
                        />

                        <SecretVersionField
                           controlProps={{ secretUuid: form.values.secretUuid, selectLatest: true }}
                           disabled={!form.values.secretUuid}
                           label={'Version'}
                           name={'versionUuid'}
                           required={true}
                           data-test={'AddSecretModal-Version'}
                        />

                        <InputField2
                           disabled={!form.values.versionUuid}
                           label={'Alias'}
                           name={'alias'}
                           placeholder={'Edit alias'}
                           required={true}
                           data-test={'AddSecretModal:Alias'}
                           hint={<AliasHint />}
                        />

                        <CheckboxField2 name={'newStorage'} label={'New storage'} hint={<NewStorageHint />} />

                        <footer className={classes.footer}>
                           <div className={classes.spacer} />

                           <div className={classes.actions}>
                              <Button
                                 view={'flat'}
                                 className={classes.button}
                                 onClick={dismiss}
                                 disabled={form.isSubmitting}
                                 qa={'AddSecretModal:CancelButton'}
                              >
                                 Cancel
                              </Button>

                              <Button
                                 view={'action'}
                                 type={'submit'}
                                 className={classes.button}
                                 disabled={!form.isValid || form.isSubmitting}
                                 qa={'AddSecretModal:AddButton'}
                              >
                                 Add
                              </Button>
                           </div>
                        </footer>

                        <DevForm />
                     </Form>
                  </FormHooksContext.Provider>
               );
            }}
         </FixedFormik>
      </ModalLayout>
   );
};

AddSecretModal.displayName = 'AddSecretModal';
