import { Button } from '@yandex-cloud/uikit';
import {
   CheckboxField2,
   DISMISS_REASON,
   FixedFormik,
   FormHooks,
   FormHooksContext,
   IModalProps,
   InputField2,
   ModalLayout,
} from '@yandex-infracloud-ui/libs';
import { Form, FormikProps } from 'formik';
import React, { useCallback, useMemo, useRef } from 'react';
import { useSelector } from 'react-redux';
import { boolean, object, string } from 'yup';

// noinspection ES6PreferShortImport
import { DevForm } from '../../../../components/lib';
import { NEW_SECRET_STORAGE_BY_DEFAULT } from '../../../../models';

import { getAvailableAliases, secretAliasSchema, SecretValidationContext } from '../../helpers';
import { useUsedAliases } from '../../hooks';
import { RootStateWithSecrets, selectSecret, selectSecretVersions } from '../../slice';
import { AliasHint, NewStorageHint } from '../hints';
import { SecretName } from '../SecretName/SecretName';
import { SecretVersionField } from '../SecretVersionField/SecretVersionField';

import classes from './AddVersionModal.module.css';

const initialValues = {
   alias: '',
   newStorage: NEW_SECRET_STORAGE_BY_DEFAULT,
   versionUuid: '',
};

type FormValue = typeof initialValues;

const validationSchema = object<FormValue>({
   alias: secretAliasSchema,
   newStorage: boolean().label('New storage'),
   versionUuid: string().label('Version').required(),
});

interface Props extends IModalProps<FormValue> {
   secretName?: string;
   secretUuid: string;
}

export const AddVersionModal: React.FC<Props> = ({ secretName, secretUuid, ok, cancel }) => {
   const versionsSelector = useCallback((s: RootStateWithSecrets) => selectSecretVersions(s, secretUuid), [secretUuid]);
   const allSecretVersions = useSelector(versionsSelector);

   const secretSelector = useCallback((s: RootStateWithSecrets) => selectSecret(s, secretUuid), [secretUuid]);
   const secret = useSelector(secretSelector);

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const formRef = useRef<FormikProps<FormValue>>();

   const setAlias = useCallback(
      (versionUuid: string) => {
         const [alias] = getAvailableAliases(secret!, allSecretVersions, [] /* TODO */, versionUuid);

         formRef.current?.setFieldValue('alias', alias);
         formRef.current?.setFieldTouched('alias', true, true);
      },
      [allSecretVersions, secret],
   );

   const formHooks = useMemo(
      () =>
         ({
            // Обновляет алиас, если его еще не трогали, дописывая в конец ID версии
            onFieldChange(field: keyof FormValue, value: string) {
               if (field === 'versionUuid' && formRef.current) {
                  if (!formRef.current?.touched.alias) {
                     setAlias(value);
                  }
               }
            },
         } as FormHooks<FormValue>),
      [setAlias],
   );

   const usedAliases = useUsedAliases();
   const validationContext: SecretValidationContext = useMemo(() => ({ usedAliases }), [usedAliases]);

   return (
      <ModalLayout
         title={'Add version'}
         onDismiss={dismiss}
         showFooter={false}
         subtitle={
            <>
               to secret: <SecretName secretUuid={secretUuid} className={classes.secretName} />
            </>
         }
         data-e2e={'AddSecretVersionModal'}
      >
         <FixedFormik
            initialValues={{ ...initialValues, alias: secretName ?? secretUuid }}
            onSubmit={ok}
            validationSchema={validationSchema}
            validationContext={validationContext}
         >
            {form => {
               formRef.current = form;

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <Form className={classes.form}>
                        <SecretVersionField
                           name={'versionUuid'}
                           label={'Version'}
                           required={true}
                           controlProps={{ secretUuid, selectLatest: false }}
                        />

                        <InputField2 name={'alias'} label={'Alias'} required={true} hint={<AliasHint />} />

                        <CheckboxField2 name={'newStorage'} label={'New storage'} hint={<NewStorageHint />} />

                        <footer className={classes.footer}>
                           <div className={classes.spacer} />

                           <div className={classes.actions}>
                              <Button
                                 view={'flat'}
                                 className={classes.button}
                                 onClick={dismiss}
                                 disabled={form.isSubmitting}
                                 qa={'AddSecretVersionModal:CancelButton'}
                              >
                                 Cancel
                              </Button>

                              <Button
                                 view={'action'}
                                 type={'submit'}
                                 className={classes.button}
                                 disabled={!form.isValid || form.isSubmitting}
                                 qa={'AddSecretVersionModal:OkButton'}
                              >
                                 Add
                              </Button>
                           </div>
                        </footer>

                        <DevForm />
                     </Form>
                  </FormHooksContext.Provider>
               );
            }}
         </FixedFormik>
      </ModalLayout>
   );
};

AddVersionModal.displayName = 'AddVersionModal';
