import { Button } from '@yandex-cloud/uikit';

import { DISMISS_REASON, FixedFormik, IModalProps, InputField2, ModalLayout } from '@yandex-infracloud-ui/libs';
import { Form, FormikProps } from 'formik';
import React, { useCallback, useMemo, useRef } from 'react';
import { object } from 'yup';

import { DevForm } from '../../../../components/lib';
import { secretAliasSchema, SecretValidationContext } from '../../helpers';
import { useUsedAliases } from '../../hooks';
import { DeployUnitSecretVersion } from '../../models';

import classes from './EditAliasModal.module.css';

const initialValues = {
   alias: '',
};

type FormValue = typeof initialValues;

const validationSchema = object<FormValue>({
   alias: secretAliasSchema,
});

interface Props extends IModalProps<FormValue> {
   duVersion: DeployUnitSecretVersion;
   suggestedAliases: string[];
}

export const EditAliasModal: React.FC<Props> = ({ ok, cancel, duVersion, suggestedAliases }) => {
   const usedAliases = useUsedAliases();
   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const validationContext: SecretValidationContext = useMemo(() => ({ usedAliases }), [usedAliases]);

   const formRef = useRef<FormikProps<FormValue>>();

   const handleResetOldAlias = useCallback(() => {
      if (formRef.current) {
         formRef.current?.setFieldValue('alias', duVersion.alias);
      }
   }, [duVersion.alias]);

   return (
      <ModalLayout
         title={'Edit secret alias'}
         subtitle={'Alias is used for linking secret usage entry to secret version in secret store'}
         onDismiss={dismiss}
         showFooter={false}
         data-e2e={'SecretList:EditAliasModal'}
      >
         <FixedFormik
            initialValues={{ ...initialValues, alias: suggestedAliases[0] }}
            onSubmit={ok}
            validationSchema={validationSchema}
            validationContext={validationContext}
         >
            {form => {
               formRef.current = form;

               return (
                  <Form className={classes.form}>
                     <InputField2 name={'alias'} label={'New alias'} controlProps={{ autoFocus: true }} />

                     <footer className={classes.footer}>
                        {duVersion.newAlias || duVersion.alias !== form.values.alias ? (
                           <Button
                              className={classes.button}
                              onClick={handleResetOldAlias}
                              disabled={form.isSubmitting}
                              extraProps={{ 'data-e2e': 'EditAliasModal:ResetButton' }}
                           >
                              Reset
                           </Button>
                        ) : null}

                        <div className={classes.spacer} />

                        <div className={classes.actions}>
                           <Button
                              view={'flat'}
                              className={classes.button}
                              onClick={dismiss}
                              disabled={form.isSubmitting}
                              extraProps={{ 'data-e2e': 'EditAliasModal:CancelButton' }}
                           >
                              Cancel
                           </Button>

                           <Button
                              view={'action'}
                              type={'submit'}
                              className={classes.button}
                              disabled={!form.isValid || form.isSubmitting}
                              extraProps={{ 'data-e2e': 'EditAliasModal:OkButton' }}
                           >
                              Submit
                           </Button>
                        </div>
                     </footer>

                     <DevForm />
                  </Form>
               );
            }}
         </FixedFormik>
      </ModalLayout>
   );
};

EditAliasModal.displayName = 'EditAliasModal';
