import {
   YCSelect,
   YCSelectGetItemsArgs,
   YCSelectGetItemsArgsFetchItems,
   YCSelectGetItemsArgsInitItems,
   YCSelectItem,
} from '@yandex-data-ui/common';
import { BaseInputProps, classNames, EMPTY_VALUE } from '@yandex-infracloud-ui/libs';
import React, { useCallback } from 'react';
import { useDispatch } from 'react-redux';
import { firstValueFrom } from 'rxjs';
import { tap } from 'rxjs/operators';

import { EXTERNAL_LINKS } from '../../../../../models';
import { logPromiseError, useRafDelay } from '../../../../../utils';
import { useAddedSecrets } from '../../../hooks';

import { Secret } from '../../../models';
import { getSecretsApi } from '../../../services';
import { secretsSlice } from '../../../slice';

interface Props extends BaseInputProps<string> {
   dataTest?: string;
}

export const NewSecretSelectFieldInput: React.FC<Props> = React.memo(
   ({
      className,
      dataTest,
      disabled,
      // hasError,  TODO
      // id,  TODO
      // name,  TODO
      onBlur,
      onChange,
      placeholder = 'Select new secret',
      readonly,
      readonlyClassName,
      value,
   }) => {
      const addedSecrets = useAddedSecrets();
      const dispatch = useDispatch();

      const getItems = useCallback(
         (args: YCSelectGetItemsArgs): Promise<{ items: YCSelectItem[] }> => {
            const initArgs = args as YCSelectGetItemsArgsInitItems;
            if (initArgs.exactKeys) {
               console.log('TODO exactKeys', initArgs);
               return Promise.resolve({ items: [] });
            }

            const fetchArgs = args as YCSelectGetItemsArgsFetchItems;

            return firstValueFrom(
               getSecretsApi()
                  .getList(fetchArgs.searchPattern ?? '')
                  .pipe(tap(items => dispatch(secretsSlice.actions.addSecrets(items)))),
            )
               .then(x => x ?? [])
               .then(items => items.map(item => secretToSelectItem(item, addedSecrets.includes(item.uuid))))
               .then(items => ({ items }), logPromiseError);
         },
         [addedSecrets, dispatch],
      );

      const delayedBlur = useRafDelay(onBlur);

      const handleChange = useCallback(
         v => {
            onChange(v);
            delayedBlur();
         },
         [delayedBlur, onChange],
      );

      // region render
      if (readonly) {
         return <div className={readonlyClassName}>{value || EMPTY_VALUE}</div>;
      }

      return (
         <YCSelect
            className={classNames(className, `test-${dataTest}`)}
            disabled={disabled}
            getItems={getItems}
            onUpdate={handleChange}
            placeholder={placeholder}
            type={'single'}
            value={value}
         />
      );
      // endregion
   },
);

NewSecretSelectFieldInput.displayName = 'NewSecretSelectFieldInput';

function secretToSelectItem(secret: Secret, disabled: boolean): YCSelectItem {
   return {
      disabled,
      key: secret.uuid,
      title: secret.name ?? secret.uuid,
      url: EXTERNAL_LINKS.secretLink(secret.uuid),
      value: secret.uuid,
   };
}
