import { toaster } from '@yandex-cloud/uikit';
import { DISMISS_REASON, modalService, toasts, useDismounted } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect } from 'react';
import { useDispatch, useStore } from 'react-redux';
import { takeUntil } from 'rxjs/operators';
import { getAvailableAliases } from '../../helpers';
import { addSecretContext } from '../../hocs';

import { useDuSecretsWithVersions, useUsedAliases } from '../../hooks';
import { useSecretsContext } from '../../hooks/useSecretsContext';
import { DeployUnitSecretVersion } from '../../models';
import {
   loadSecretVersions,
   RootStateWithSecrets,
   secretsSlice,
   selectSecretVersions,
   upsertDeployUnitSecretVersion,
} from '../../slice';
import { AddSecretModal } from '../AddSecretModal/AddSecretModal';

import { AddVersionModal } from '../AddVersionModal/AddVersionModal';
import { EditAliasModal } from '../EditAliasModal/EditAliasModal';
import { SecretListLayout } from './__layout/SecretListLayout';

interface Props {
   disabled?: boolean;
   readonly?: boolean;
}

export const SecretList: React.FC<Props> = React.memo(({ disabled = false, readonly = false }) => {
   const { stageId, duId } = useSecretsContext();
   const store = useStore<RootStateWithSecrets>();
   const dismounted = useDismounted();

   const dispatch = useDispatch();
   const { duSecrets, versions, secrets } = useDuSecretsWithVersions(stageId, duId);
   const usedAliases = useUsedAliases();

   const handleAddSecret = useCallback(() => {
      modalService
         .open(
            addSecretContext(AddSecretModal, stageId, duId),
            {},
            { closeOnOutsideClick: false, dialogProps: { size: 'm' } },
         )
         .pipe(takeUntil(dismounted))
         .subscribe(
            version => dispatch(upsertDeployUnitSecretVersion({ ...version, stageId, duId })),
            reason => {
               if (reason !== DISMISS_REASON) {
                  toasts.error(reason.toString(), 'Unknown error on adding secret');
               }
            },
         );
   }, [dismounted, dispatch, duId, stageId]);

   const handleAddVersion = useCallback(
      (secretUuid: string) => {
         const secret = secrets[secretUuid];
         const secretName = secret?.name;

         modalService
            .open(
               addSecretContext(AddVersionModal, stageId, duId),
               { secretName, secretUuid },
               { closeOnOutsideClick: false, dialogProps: { size: 'm' } },
            )
            .pipe(takeUntil(dismounted))
            .subscribe(
               version => dispatch(upsertDeployUnitSecretVersion({ ...version, stageId, duId, secretUuid })),
               reason => {
                  if (reason !== DISMISS_REASON) {
                     toasts.error(reason.toString(), 'Unknown error on adding version');
                  }
               },
            );
      },
      [dismounted, dispatch, duId, secrets, stageId],
   );

   const handleAliasEdit = useCallback(
      (secretUuid: string, duVersion: DeployUnitSecretVersion) => {
         const secret = secrets[secretUuid]!;
         const allSecretVersions = selectSecretVersions(store.getState(), secretUuid);

         const suggestedAliases = [
            duVersion.newAlias,
            ...getAvailableAliases(secret, allSecretVersions, usedAliases, duVersion.versionUuid),
            duVersion.alias,
         ]
            .filter(Boolean)
            .map(x => x!);

         modalService
            .open(
               addSecretContext(EditAliasModal, stageId, duId),
               { duVersion, suggestedAliases },
               { closeOnOutsideClick: false, dialogProps: { size: 'm' } },
            )
            .pipe(takeUntil(dismounted))
            .subscribe(
               ({ alias: newAlias }) => {
                  dispatch(secretsSlice.actions.editAlias({ stageId, duId, duVersion, newAlias }));
               },
               reason => {
                  if (reason !== DISMISS_REASON) {
                     toasts.error(reason.toString(), 'Unknown error on editing alias');
                  }
               },
            );
      },
      [dismounted, dispatch, duId, secrets, stageId, store, usedAliases],
   );

   const handleRemoveUnused = useCallback(
      (secretUuid: string) => {
         dispatch(secretsSlice.actions.removeUnusedVersionsFromSecret({ duId, secretUuid, stageId }));
         showSuccessToast('Secret will be removed');
      },
      [dispatch, duId, stageId],
   );

   const handleMigrate = useCallback(
      (alias: string) => {
         dispatch(secretsSlice.actions.migrateFromLegacy({ stageId, duId, alias }));
         showSuccessToast('Secret will be migrated to new storage');
      },
      [dispatch, duId, stageId],
   );

   const handleUndoMigration = useCallback(
      (alias: string) => {
         dispatch(secretsSlice.actions.undoMigrationFromLegacy({ stageId, duId, alias }));
         showSuccessToast('Secret migration was canceled');
      },
      [dispatch, duId, stageId],
   );

   const handleMigrateAll = useCallback(() => {
      dispatch(secretsSlice.actions.migrateAllFromLegacy({ stageId, duId }));
      showSuccessToast('All legacy secrets will be migrated to new storage');
   }, [dispatch, duId, stageId]);

   useEffect(() => {
      dispatch(loadSecretVersions(duSecrets?.map(s => s.secretUuid) ?? []));
   }, [dismounted, dispatch, duId, duSecrets, stageId]);

   return (
      <SecretListLayout
         disabled={disabled}
         duSecrets={duSecrets}
         onAddSecret={handleAddSecret}
         onAddSecretVersion={handleAddVersion}
         onAliasEdit={handleAliasEdit}
         onMigrate={handleMigrate}
         onMigrateAll={handleMigrateAll}
         onRemoveUnused={handleRemoveUnused}
         onUndoMigration={handleUndoMigration}
         readonly={readonly}
         secrets={secrets}
         versions={versions}
      />
   );
});

SecretList.displayName = 'SecretList';

function showSuccessToast(title: string) {
   // noinspection JSIgnoredPromiseFromCall
   toaster!.createToast({ name: title, title, type: 'success' });
}
